/*
 * *##%
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.decorator.Decorator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.*;

/**
 * Interface to create a builder, this object is design to build a {@link
 * NavigationModel}.
 *
 * @author sletellier
 * @since 2.0.0
 */
public abstract class AbstractNavigationModelBuilder<E extends NavigationNode<E>> implements NavigationModelBuilder<E> {

    /** Logger */
    static private final Log log =
            LogFactory.getLog(AbstractNavigationModelBuilder.class);

    /**
     * The model dealed by the builder.
     * <p/>
     * <b>Note:</b> It is a good idea to keep only one instance of the model. If
     * reset is required, should empty the model but not reinstanciate it.
     */
    protected NavigationModel<E> model;

    /** default ui class to use if node does not define an ui class */
    protected Class<? extends JAXXObject> defaultUIClass;

    /** [optional] default action class */
    protected Class<? extends JAXXAction> defaultUIHandlerClass;

    /**
     * if sets to {@code true} will not fires any event (this is a convinient
     * state when we are building the tree to avoid reload of nodes)
     */
    protected boolean adjustingValue;

    public AbstractNavigationModelBuilder(
            Class<? extends JAXXObject> defaultUIClass,
            Class<? extends JAXXAction> defaultUIHandlerClass,
            NavigationModel<E> model) {
        this.defaultUIClass = defaultUIClass;
        this.defaultUIHandlerClass = defaultUIHandlerClass;
        this.model = model;
    }

    @Override
    public NavigationModel<E> getModel() {
        return model;
    }

    @Override
    public E buildEmptyRoot(JAXXContextEntryDef<?> entryDef,
                            String contextName) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                entryDef,
                null
        );
        addI18nNodeRenderer(node, "");
        return addChildNode(null, node);
    }

    @Override
    public E build(E parentNode,
                   String libelle,
                   JAXXContextEntryDef<?> entryDef,
                   String entryPath,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                entryDef,
                entryPath
        );
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E build(E parentNode,
                   String libelle,
                   JAXXContextEntryDef<?> entryDef,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                entryDef,
                null
        );
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E build(E parentNode,
                   String libelle,
                   String entryPath,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                null,
                entryPath
        );
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E build(E parentNode,
                   Decorator<?> decorator,
                   JAXXContextEntryDef<?> entryDef,
                   String entryPath,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                entryDef,
                entryPath
        );
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E build(E parentNode,
                   Decorator<?> decorator,
                   JAXXContextEntryDef<?> entryDef,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                entryDef,
                null
        );
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E build(E parentNode,
                   Decorator<?> decorator,
                   String entryPath,
                   String contextName,
                   Class<? extends JAXXObject> uiClass,
                   Class<? extends JAXXAction> actionClass) {
        E node = createNavigationNode(
                model.getPathSeparator(),
                contextName,
                null,
                entryPath
        );
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    @Override
    public E removeChildNode(E node) {
        E parentNode = node.getParent();
        model.removeNodeFromParent(node);
        return parentNode;
    }

    @Override
    public void moveNode(E parentNode,
                         E node,
                         int position) {
        parentNode.remove(node);
        parentNode.insert(node, position);
        NavigationModel<E> treeModel = getModel();
        treeModel.nodeStructureChanged(parentNode);
    }

    @Override
    public void addI18nNodeRenderer(E node, String libelle) {
        node.setRenderer(new NavigationNodeRendererI18nImpl(libelle));
    }

    @Override
    public void addDecoratorNodeRenderer(E node, Decorator<?> decorator) {
        node.setRenderer(new NavigationNodeRendererDecoratorImpl(decorator));
    }

    @Override
    public void addNodeJaxxClasses(
            E node,
            Class<? extends JAXXObject> uIClass,
            Class<? extends JAXXAction> uIHandlerClass) {
        node.setUIClass(uIClass);
        node.setUIHandlerClass(uIHandlerClass);
    }

    @Override
    public void printModel(E node) {
        if (node == null) {
            return;
        }
        log.info("node " + node.getFullPath() + ", jxpath: " +
                 node.getJaxxContextEntryPath() + ", entryContextDef: " +
                 node.getJaxxContextEntryDef());
        if (log.isDebugEnabled()) {
            log.debug("node userObject" + node.getUserObject());
            log.debug("value from node " + node.getBean(getModel().getContext()));
            log.debug("value from model " + getModel().getBean(node));
        }
        Enumeration<? extends E> children = node.children();
        while (children.hasMoreElements()) {
            printModel(children.nextElement());
        }
    }

    protected E addChildNode(E parentNode, E node) {

        if (node.getUIClass() == null) {
            // no ui is associated with this node, use the default one
            node.setUIClass(defaultUIClass);
        }

        if (node.getUIHandlerClass() == null) {
            // no ui handler associated with this node, use the default one
            node.setUIHandlerClass(defaultUIHandlerClass);
        }
        if (parentNode == null) {
            model.setRoot(node);
        } else {
            parentNode.add(node);
        }
        model.nodeStructureChanged(parentNode == null ? node : parentNode);
        return node;
    }

}
