/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;

import java.util.ArrayList;
import java.util.List;

import javax.swing.JTree;

/**
 * To help getting and setting navigation tree objects from a {@link JAXXContext}.
 * <p/>
 * There is six types of data which can be hold in a context :
 * <ul>
 * <li>tree : the tree </li>
 * <li>tree model : the navigation tree model</li>
 * <li>tree handler : the navigation tree handler</li>
 * <li>selected path : the navigation path of the selected node</li>
 * <li>selected node : the selected node</li>
 * <li>selected bean : the selected bean</li>
 * <li>selected ui : the selected content ui</li>
 * </ul>
 * <p/>
 * To make possible the use of more than one navigation tree system in a same
 * context, we <b>MUST</b> distinguish the context entries definition. For this
 * purpose, entries definition are normalized and prefixed by a unique {@link #prefix}.
 * <p/>
 * Here is the keys mapping :
 * <ul>
 * <li>tree : {@code prefix + "-tree"}</li>
 * <li>tree model : {@code prefix + "-tree-model"}</li>
 * <li>tree handler : {@code prefix + "-tree-handler"}</li>
 * <li>selected path : {@code prefix + "-selected-path"}</li>
 * <li>selected node : {@code prefix + "-selected-node"}</li>
 * <li>selected bean : {@code prefix + "-selected-bean"}</li>
 * <li>selected ui: {@code prefix + "-selected-ui"}</li>
 * </ul>
 *
 * @author chemit
 * @since 1.7.2
 */
public class NavigationTreeContextHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(NavigationTreeContextHelper.class);

    protected final String prefix;
    protected JAXXContextEntryDef<List<String>> selectedPathsContextEntry;
    protected JAXXContextEntryDef<List<Object>> selectedBeansContextEntry;
    protected JAXXContextEntryDef<List<NavigationTreeNode>> selectedNodesContextEntry;
    protected JAXXContextEntryDef<NavigationModel> modelContextEntry;
    protected JAXXContextEntryDef<NavigationTreeHandler> handlerContextEntry;
    protected JAXXContextEntryDef<JTree> treeContextEntry;
    protected JAXXContextEntryDef<JXTreeTable> treeTableContextEntry;
//    protected JAXXContextEntryDef<Component> selectedUIContextEntry;

    public NavigationTreeContextHelper(String prefix) {
        this.prefix = prefix;
        treeContextEntry = JAXXUtil.newContextEntryDef(prefix + "-tree", JTree.class);
        treeTableContextEntry = JAXXUtil.newContextEntryDef(prefix + "-tree-table", JXTreeTable.class);
        modelContextEntry = JAXXUtil.newContextEntryDef(prefix + "-model", NavigationModel.class);
        handlerContextEntry = JAXXUtil.newContextEntryDef(prefix + "-handler", NavigationTreeHandler.class);
        selectedBeansContextEntry = JAXXUtil.newListContextEntryDef(prefix + "-selected-beans");
        selectedNodesContextEntry = JAXXUtil.newListContextEntryDef(prefix + "-selected-nodes");
        selectedPathsContextEntry = JAXXUtil.newListContextEntryDef(prefix + "-selected-paths");
//        selectedUIContextEntry = JAXXUtil.newContextEntryDef(prefix + "-selected-ui", Component.class);
    }

    public String getPrefix() {
        return prefix;
    }

    public JTree getTree(JAXXContext context) {
        JTree r = getTreeContextEntry().getContextValue(context);
        return r;
    }

    public JXTreeTable getTreeTable(JAXXContext context) {
        JXTreeTable r = getTreeTableContextEntry().getContextValue(context);
        return r;
    }

    public NavigationModel getModel(JAXXContext context) {
        NavigationModel r = getModelContextEntry().getContextValue(context);
        return r;
    }

    /**
     * @param context where to find model
     * @deprecated please use {@link #getModel(JAXXContext)}, will be remove
     *             soon
     */
    @Deprecated
    public NavigationTreeModel getTreeModel(JAXXContext context) {
        NavigationModel r = getModel(context);
        if (r instanceof NavigationTreeModel) {
            return (NavigationTreeModel) r;
        }
        return null;
    }

    public NavigationTreeHandler getTreeHandler(JAXXContext context) {
        NavigationTreeHandler r =
                getTreeHandlerContextEntry().getContextValue(context);
        return r;
    }

    public String getSelectedPath(JAXXContext context) {
        List<String> values = getSelectedPathContextEntry().getContextValue(context);
        if (log.isWarnEnabled()){
            if (values.size() > 1){
                log.warn("More than one values are selected, return first one");
            }
        }
        return values.get(0);
    }

    public List<String> getSelectedPaths(JAXXContext context) {
        return getSelectedPathContextEntry().getContextValue(context);
    }

    public NavigationTreeNode getSelectedNode(JAXXContext context) {
        List<NavigationTreeNode> values = getSelectedNodeContextEntry().getContextValue(context);
        if (log.isWarnEnabled()){
            if (values.size() > 1){
                log.warn("More than one values are selected, return first one");
            }
        }
        return values.get(0);
    }

    public List<NavigationTreeNode> getSelectedNodes(JAXXContext context) {
        return getSelectedNodeContextEntry().getContextValue(context);
    }

    public Object getSelectedBean(JAXXContext context) {
        List<Object> values = getSelectedBeanContextEntry().getContextValue(context);
        if (log.isWarnEnabled()){
            if (values.size() > 1){
                log.warn("More than one values are selected, return first one");
            }
        }
        return values.get(0);
    }

    public List<Object> getSelectedBeans(JAXXContext context) {
        return getSelectedBeanContextEntry().getContextValue(context);
    }

//    public Component getSelectedUI(JAXXContext context) {
//        Component r = getSelectedUIContextEntry().getContextValue(context);
//        return r;
//    }

    /**
     * @param context where to store model
     * @param model   model to store
     * @deprecated please use {@link #setModel(JAXXContext, NavigationModel)},
     *             will be remove soon
     */
    @Deprecated
    public void setTreeModel(JAXXContext context, NavigationModel model) {
        setModel(context, model);
    }

    public void setModel(JAXXContext context, NavigationModel model) {
        getModelContextEntry().setContextValue(context, model);
    }

    public void setTree(JAXXContext context, JTree tree) {
        getTreeContextEntry().setContextValue(context, tree);
    }

    public void setTreeTable(JAXXContext context, JXTreeTable treeTable) {
        getTreeTableContextEntry().setContextValue(context, treeTable);
    }

    public void setTreeHandler(JAXXContext context,
                               NavigationTreeHandler handler) {
        getTreeHandlerContextEntry().setContextValue(context, handler);
    }

    public void setSelectedPath(JAXXContext context, String path) {
        if (path == null) {
            getSelectedPathContextEntry().removeContextValue(context);
        } else {
            List<String> selecteds = new ArrayList<String>();
            selecteds.add(path);
            getSelectedPathContextEntry().setContextValue(context, selecteds);
        }
    }

    public void setSelectedPaths(JAXXContext context, List<String> paths) {
        if (paths == null || paths.isEmpty()) {
            getSelectedPathContextEntry().removeContextValue(context);
        } else {
            getSelectedPathContextEntry().setContextValue(context, paths);
        }
    }

    public void setSelectedNode(JAXXContext context, NavigationTreeNode node) {
        if (node == null) {
            getSelectedNodeContextEntry().removeContextValue(context);
        } else {
            List<NavigationTreeNode> selecteds = new ArrayList<NavigationTreeNode>();
            selecteds.add(node);
            getSelectedNodeContextEntry().setContextValue(context, selecteds);
        }
    }

    public void setSelectedNodes(JAXXContext context, List<NavigationTreeNode> nodes) {
        if (nodes == null || nodes.isEmpty()) {
            getSelectedNodeContextEntry().removeContextValue(context);
        } else {
            getSelectedNodeContextEntry().setContextValue(context, nodes);
        }
    }

    public void setSelectedBean(JAXXContext context, Object bean) {
        if (bean == null) {
            getSelectedBeanContextEntry().removeContextValue(context);
        } else {
            List<Object> selecteds = new ArrayList<Object>();
            selecteds.add(bean);
            getSelectedBeanContextEntry().setContextValue(context, selecteds);
        }
    }

    public void setSelectedBeans(JAXXContext context, List<Object> beans) {
        if (beans == null || beans.isEmpty()) {
            getSelectedBeanContextEntry().removeContextValue(context);
        } else {
            getSelectedBeanContextEntry().setContextValue(context, beans);
        }
    }

//    public void setSelectedUI(JAXXContext context, Component ui) {
//        if (ui == null) {
//            getSelectedUIContextEntry().removeContextValue(context);
//        } else {
//            getSelectedUIContextEntry().setContextValue(context, ui);
//        }
//    }

    protected JAXXContextEntryDef<NavigationModel> getModelContextEntry() {
        return modelContextEntry;
    }

    protected JAXXContextEntryDef<NavigationTreeHandler>
    getTreeHandlerContextEntry() {
        return handlerContextEntry;
    }

    protected JAXXContextEntryDef<List<Object>> getSelectedBeanContextEntry() {
        return selectedBeansContextEntry;
    }

    protected JAXXContextEntryDef<List<NavigationTreeNode>> getSelectedNodeContextEntry() {
        return selectedNodesContextEntry;
    }

    protected JAXXContextEntryDef<List<String>> getSelectedPathContextEntry() {
        return selectedPathsContextEntry;
    }

    protected JAXXContextEntryDef<JTree> getTreeContextEntry() {
        return treeContextEntry;
    }

    protected JAXXContextEntryDef<JXTreeTable> getTreeTableContextEntry() {
        return treeTableContextEntry;
    }

//    public JAXXContextEntryDef<Component> getSelectedUIContextEntry() {

//        return selectedUIContextEntry;

//    }
}
