/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.finalizers;

import jaxx.compiler.*;
import jaxx.compiler.binding.DataBinding;
import jaxx.compiler.java.JavaArgument;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import static jaxx.compiler.java.JavaFileGenerator.newField;
import static jaxx.compiler.java.JavaFileGenerator.newMethod;
import jaxx.compiler.java.JavaMethod;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorLoader;
import jaxx.compiler.reflect.FieldDescriptor;
import jaxx.compiler.reflect.MethodDescriptor;
import jaxx.runtime.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import static java.lang.reflect.Modifier.*;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This class is a refactoring of the {@link jaxx.compiler.JAXXCompiler}.
 * <p/>
 * We delegate now the generation of a {@link jaxx.runtime.JAXXObject} to this class, the
 * {@link jaxx.compiler.JAXXCompiler} now only deals with the compilation of files.
 *
 * @author chemit
 */
public class DefaultFinalizer implements JAXXCompilerFinalizer {

    /**
     * Logger
     */
    protected static final Log log = LogFactory.getLog(DefaultFinalizer.class);
    /**
     * serialVersionUID field
     */
    protected static final JavaField SERIAL_VERSION_UID_FIELD = newField(PRIVATE | STATIC | FINAL,
            "long", "serialVersionUID", false, "1L");
    /**
     * $activeBindings field
     */
    protected static final JavaField ACTIVE_BINDINGS_FIELD = newField(PROTECTED,
            "java.util.List<Object>", "$activeBindings", false, "new ArrayList<Object>()");
    /**
     *
     */
    protected static final JavaField BINDING_SOURCES_FIELD = newField(PROTECTED,
            "java.util.Map<String,Object>", "$bindingSources", false, "new HashMap<String,Object>()");
    /**
     *
     */
    protected static final JavaField OBJECT_MAP_FIELD = newField(PROTECTED,
            "Map<String,Object>", "$objectMap", true, "new HashMap<String,Object>()");
    /**
     *
     */
    protected static final JavaField ALL_COMPONENTS_CREATED_FIELD = newField(PRIVATE,
            "boolean", "allComponentsCreated", false);
    /**
     *
     */
    protected static final JavaField CONTEXT_INITIALIZED = newField(PRIVATE,
            "boolean", "contextInitialized", false, "true");
    /**
     *
     */
    protected static final JavaField PREVIOUS_VALUES_FIELD = newField(PROTECTED,
            "java.util.Map<?,?>", "$previousValues", false, "new java.util.HashMap<Object,Object>()");
    /**
     *
     */
    protected static final JavaField PROPERTY_CHANGE_SUPPORT_FIELD = newField(0,
            "java.beans.PropertyChangeSupport", "$propertyChangeSupport", false);
    /**
     *
     */
    protected static final JavaMethod GET_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> T", "getContextValue",
            "return delegateContext.getContextValue(clazz, null);", true,
            new JavaArgument("Class<T>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> T", "getContextValue",
            "return delegateContext.getContextValue(clazz, name);", true,
            new JavaArgument("Class<T>", "clazz"), new JavaArgument("String", "name"));
    /**
     *
     */
    protected static final JavaMethod SET_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> void", "setContextValue",
            "delegateContext.setContextValue(o, name);", true,
            new JavaArgument("T", "o"), new JavaArgument("String", "name"));
    /**
     *
     */
    protected static final JavaMethod SET_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> void", "setContextValue",
            "delegateContext.setContextValue(o, null);", true,
            new JavaArgument("T", "o"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> void", "removeContextValue",
            "delegateContext.removeContextValue(clazz, name);", true,
            new JavaArgument("Class<T>", "clazz"), new JavaArgument("String", "name"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> void", "removeContextValue",
            "delegateContext.removeContextValue(clazz, null);", true,
            new JavaArgument("Class<T>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_PARENT_CONTAINER_MORE_METHOD = newMethod(PUBLIC, "<O extends Container> O", "getParentContainer",
            "return Util.getParentContainer(source, clazz);", true,
            new JavaArgument("Object", "source"), new JavaArgument("Class<O>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_PARENT_CONTAINER_METHOD = newMethod(PUBLIC, "<O extends Container> O", "getParentContainer",
            "return Util.getParentContainer(this, clazz);", true,
            new JavaArgument("Class<O>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_OBJECT_BY_ID_METHOD = newMethod(PUBLIC, "java.lang.Object", "getObjectById",
            "return $objectMap.get(id);", true,
            new JavaArgument("String", "id"));
    /**
     *
     */
    protected static final JavaMethod GET_JAXX_OBJECT_DESCRIPTOR_METHOD = newMethod(PUBLIC | STATIC, "jaxx.runtime.JAXXObjectDescriptor", "$getJAXXObjectDescriptor",
            "return jaxx.runtime.Util.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);", false);
    /**
     *
     */
    protected static final JavaMethod PROCESS_DATA_BINDING_METHOD = newMethod(PUBLIC, "void", "processDataBinding",
            "processDataBinding(dest, false);", true,
            new JavaArgument("String", "dest"));
    /**
     *
     */
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_METHOD = newMethod(PUBLIC, "void", "firePropertyChange",
            "super.firePropertyChange(propertyName, oldValue, newValue);", true,
            new JavaArgument("String", "propertyName"), new JavaArgument("Object", "oldValue"), new JavaArgument("Object", "newValue"));
    /**
     *
     */
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_NAMED_METHOD = newMethod(PUBLIC, "void", "firePropertyChange",
            "$getPropertyChangeSupport().firePropertyChange(propertyName, oldValue, newValue);", true,
            new JavaArgument("String", "propertyName"), new JavaArgument("Object", "oldValue"), new JavaArgument("Object", "newValue"));
    /**
     *
     */
    protected static final JavaMethod GET_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(0, "java.beans.PropertyChangeSupport", "$getPropertyChangeSupport",
            "if ($propertyChangeSupport == null)\n" +
            "    $propertyChangeSupport = new PropertyChangeSupport(this);\n" +
            "return $propertyChangeSupport;", false);
    /**
     *
     */
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(PUBLIC, "void", "addPropertyChangeListener",
            "$getPropertyChangeSupport().addPropertyChangeListener(listener);", true,
            new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    /**
     *
     */
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = newMethod(PUBLIC, "void", "addPropertyChangeListener",
            "$getPropertyChangeSupport().addPropertyChangeListener(property, listener);", true,
            new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(PUBLIC, "void", "removePropertyChangeListener",
            "$getPropertyChangeSupport().removePropertyChangeListener(listener);", true,
            new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = newMethod(PUBLIC, "void", "removePropertyChangeListener",
            "$getPropertyChangeSupport().removePropertyChangeListener(property, listener);", true,
            new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener"));

    @Override
    public void finalizeCompiler(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) {

        String fullClassName = packageName != null ? packageName + "." + className : className;
        if (root == null) {
            throw new CompilerException("root tag must be a class tag");
        }
        ClassDescriptor superclass = root.getObjectClass();
        boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(superclass);
        javaFile.setModifiers(PUBLIC);
        javaFile.setName(fullClassName);
        javaFile.setSuperClass(JAXXCompiler.getCanonicalName(superclass));
        javaFile.setSuperclassIsJAXXObject(superclassIsJAXXObject);

        javaFile.addInterfaces(compiler.getExtraInterfaces());
        javaFile.setAbstractClass(compiler.isAbstractClass());
        javaFile.setGenericType(compiler.getGenericType());
        javaFile.setSuperGenericType(compiler.getSuperGenericType());

        // finalize all objects via their decorator
        for (CompiledObject object : compiler.getObjects().values()) {
            CompiledObjectDecorator decorator = object.getDecorator();
            decorator.finalizeCompiler(compiler, root, object, javaFile, packageName, className, fullClassName);
        }

        finalizeBindings(compiler, superclassIsJAXXObject);

        if (!superclassIsJAXXObject) {
            javaFile.addInterface(JAXXCompiler.getCanonicalName(JAXXObject.class));
        }
    }

    @Override
    public void prepareJavaFile(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) throws ClassNotFoundException {

        String fullClassName = javaFile.getName();

        String jaxxContextImplementorClass = compiler.getConfiguration().getJaxxContextClass().getName();
        boolean superclassIsJAXXObject = javaFile.isSuperclassIsJAXXObject();
        if (!superclassIsJAXXObject) {
            // add logger
            if (compiler.getConfiguration().isAddLogger()) {
                javaFile.addImport(Log.class);
                javaFile.addImport(LogFactory.class);
                javaFile.addSimpleField(newField(PUBLIC | STATIC | FINAL, "Log", "log", false, "LogFactory.getLog(" + fullClassName + ".class)"));
            }

            // JAXXObject
            javaFile.addField(OBJECT_MAP_FIELD);
            javaFile.addMethod(GET_OBJECT_BY_ID_METHOD);
            javaFile.addField(BINDING_SOURCES_FIELD);
            javaFile.addField(ACTIVE_BINDINGS_FIELD);

            // JAXXContext
            javaFile.addField(newField(PROTECTED | FINAL, JAXXContext.class.getName(), "delegateContext", true, "new " + jaxxContextImplementorClass + "()"));
//            javaFile.addField(newField(PROTECTED | FINAL, JAXXContext.class.getName(), "delegateContext", true, "new " + jaxxContextImplementorClass + "(this)"));
            javaFile.addMethod(SET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(SET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_MORE_METHOD);

            // PropertyChangeSupport
            addPropertyChangeSupport(root, javaFile);

            // DataBinding
            javaFile.addMethod(PROCESS_DATA_BINDING_METHOD);
        }
        javaFile.addSimpleField(SERIAL_VERSION_UID_FIELD);
        javaFile.addSimpleField(ALL_COMPONENTS_CREATED_FIELD);
        boolean overrideContextInitialized = false;
        FieldDescriptor[] scriptFields = compiler.getScriptFields();
        for (FieldDescriptor f : scriptFields) {
            if ("contextInitialized".equals(f.getName())) {
                overrideContextInitialized = true;
                break;
            }
        }
        if (!overrideContextInitialized) {
            javaFile.addSimpleField(CONTEXT_INITIALIZED);
        }
        javaFile.addSimpleField(createJAXXObjectDescriptorField(compiler, javaFile));

        if (compiler.getStylesheet() != null) {
            boolean needField = true;
            if (superclassIsJAXXObject) {
                // check alreay exists on parent
                ClassDescriptor superclass = root.getObjectClass();
                if (log.isDebugEnabled()) {
                    log.debug("superclass : " + superclass);
                }
                JAXXCompiler parentCompiler = compiler.getEngine().getJAXXCompiler(superclass.getName());
                if (parentCompiler != null) {
                    needField = parentCompiler.getStylesheet() == null;
                } else {
                    try {
                        superclass.getDeclaredFieldDescriptor(PREVIOUS_VALUES_FIELD.getName());
                        needField = false;
                    } catch (NoSuchFieldException ex) {
                        // field not found                        
                    }
                }
                if (needField && log.isDebugEnabled()) {
                    log.debug("no " + PREVIOUS_VALUES_FIELD.getName() + " field in super class");
                }
            }
            if (needField) {
                javaFile.addSimpleField(PREVIOUS_VALUES_FIELD);
            }
        }
        //TC 20090228 - only generate constructors if not done in scripts
        boolean constructorDetected = false;
        MethodDescriptor[] methods = compiler.getScriptMethods();
        for (MethodDescriptor m : methods) {
            try {
                m.getReturnType();
                if (className.equals(m.getName())) {
                    constructorDetected = true;
                    break;
                }
            } catch (Exception e) {
                log.warn("could not find return type " + m);
            }
        }
        if (!constructorDetected) {
            javaFile.addMethod(createConstructor(compiler, className, superclassIsJAXXObject));
            javaFile.addMethod(createConstructorWithInitialContext(compiler, className, superclassIsJAXXObject));
        }

        javaFile.addMethod(createInitializer(compiler));
        javaFile.addMethod(GET_JAXX_OBJECT_DESCRIPTOR_METHOD);

        javaFile.addBodyCode(compiler.getBodyCode().toString());

        javaFile.addMethod(createCompleteSetupMethod(compiler, javaFile, compiler.getInitDataBindings()));

        javaFile.addMethod(newMethod(PUBLIC, "void", "applyDataBinding",
                compiler.getApplyDataBinding().toString() + JAXXCompiler.getLineSeparator() + "processDataBinding($binding);",
                true,
                new JavaArgument("String", "$binding")));

        javaFile.addMethod(newMethod(PUBLIC, "void", "removeDataBinding",
                compiler.getRemoveDataBinding().toString(), true, new JavaArgument("String", "$binding")));

        javaFile.addMethod(createProcessDataBindingMethod(compiler, superclassIsJAXXObject));

        addEventHandlers(compiler, javaFile);

    }

    protected void finalizeBindings(JAXXCompiler compiler, boolean superclassIsJAXXObject) {
        String eol = JAXXCompiler.getLineSeparator();

        if (compiler.getDataBindings() != null && !compiler.getDataBindings().isEmpty()) {
            compiler.addImport(DataBindingListener.class.getName());
        }

        for (DataBinding dataBinding : compiler.getDataBindings()) {
            if (dataBinding.compile(compiler, true)) {
                String expression = "applyDataBinding(" + compiler.getJavaCode(dataBinding.getId()) + ");" + eol;
                compiler.appendInitDataBindings(expression);
            }
        }

        if (superclassIsJAXXObject) {
            boolean hasBind = compiler.hasApplyDataBinding();
            if (hasBind) {
                compiler.appendApplyDataBinding(" else {");
                compiler.appendApplyDataBinding(eol);
                compiler.appendApplyDataBinding("    ");
            }
            compiler.appendApplyDataBinding("super.applyDataBinding($binding);");
            compiler.appendApplyDataBinding(eol);

            if (hasBind) {
                compiler.appendApplyDataBinding("    return;");
                compiler.appendApplyDataBinding(eol);
                compiler.appendApplyDataBinding("}");
            }

            hasBind = compiler.hasRemoveDataBinding();
            if (hasBind) {
                compiler.appendRemoveDataBinding(" else {");
                compiler.appendRemoveDataBinding(eol);
                compiler.appendRemoveDataBinding("    ");
            }
            compiler.appendRemoveDataBinding("super.removeDataBinding($binding);");
            compiler.appendRemoveDataBinding(eol);

            if (hasBind) {
                compiler.appendRemoveDataBinding("}");
            }
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create fields ----------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected JavaField createJAXXObjectDescriptorField(JAXXCompiler compiler, JavaFile javaFile) {
        try {
            JAXXObjectDescriptor descriptor = compiler.getJAXXObjectDescriptor();
            String data = Base64Coder.serialize(descriptor, true);
            /*ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(new GZIPOutputStream(buffer));
            out.writeObject(descriptor);
            out.close();
            // the use of the weird deprecated constructor is deliberate -- we need to store the data as a String
            // in the compiled class file, since byte array initialization is horribly inefficient compared to
            // String initialization.  So we store the bytes in the String, and we quite explicitly want a 1:1
            // mapping between bytes and chars, with the high byte of the char set to zero.  We can then safely
            // reconstitute the original byte[] at a later date.  This is unquestionably an abuse of the String
            // type, but if we could efficiently store a byte[] we wouldn't have to do this.
            String data = new String(buffer.toByteArray(), 0);*/

            int sizeLimit = 65000; // constant strings are limited to 64K, and I'm not brave enough to push right up to the limit
            if (data.length() < sizeLimit) {
                return newField(PRIVATE | STATIC, "String", "$jaxxObjectDescriptor", false, compiler.getJavaCode(data));
            } else {
                StringBuffer initializer = new StringBuffer();
                for (int i = 0; i < data.length(); i += sizeLimit) {
                    String name = "$jaxxObjectDescriptor" + i;
                    javaFile.addField(new JavaField(PRIVATE | STATIC, "String", name, false,
                            compiler.getJavaCode(data.substring(i, Math.min(i + sizeLimit, data.length())))));
                    if (initializer.length() > 0) {
                        initializer.append(" + ");
                    }
                    initializer.append("String.valueOf(").append(name).append(")");
                }
                return newField(PRIVATE | STATIC | FINAL, "String", "$jaxxObjectDescriptor", false, initializer.toString());
            }
        } catch (IOException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected void addPropertyChangeSupport(CompiledObject root, JavaFile javaFile) {
        ClassDescriptor currentClass = root.getObjectClass();
        MethodDescriptor firePropertyChange = null;
        while (firePropertyChange == null && currentClass != null) {
            try {
                firePropertyChange = currentClass.getDeclaredMethodDescriptor("firePropertyChange", ClassDescriptorLoader.getClassDescriptor(String.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class));

            } catch (NoSuchMethodException e) {
                currentClass = currentClass.getSuperclass();
            }
        }

        int modifiers = firePropertyChange != null ? firePropertyChange.getModifiers() : 0;
        if (isPublic(modifiers)) {
            // we have all the support we need
            return;
        }
        if (isProtected(modifiers)) {
            // there is property change support but the firePropertyChange method is protected
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_METHOD);
        } else {
            // either no support at all or firePropertyChange isn't accessible
            javaFile.addField(PROPERTY_CHANGE_SUPPORT_FIELD);
            javaFile.addMethod(GET_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_NAMED_METHOD);
        }
    }

    protected void addEventHandlers(JAXXCompiler compiler, JavaFile javaFile) {
        for (Map.Entry<String, Map<ClassDescriptor, List<EventHandler>>> e1 : compiler.getEventHandlers().entrySet()) {
            // outer loop is iterating over different objects (well, technically, different Java expressions)
            for (Map.Entry<ClassDescriptor, List<EventHandler>> e2 : e1.getValue().entrySet()) {
                // iterate over different types of listeners for this particular object (MouseListener, ComponentListener, etc.)
                for (EventHandler handler : e2.getValue()) {
                    // iterate over individual event handlers of a single type
                    String methodName = compiler.getEventHandlerMethodName(handler);
                    MethodDescriptor listenerMethod = handler.getListenerMethod();
                    if (listenerMethod.getParameterTypes().length != 1) {
                        throw new CompilerException("Expected event handler " + listenerMethod.getName() + " of class " + handler.getListenerClass() + " to have exactly one argument");
                    }
                    javaFile.addMethod(newMethod(PUBLIC, "void", methodName, handler.getJavaCode(), false,
                            new JavaArgument(JAXXCompiler.getCanonicalName(listenerMethod.getParameterTypes()[0]), "event")));
                }
            }
        }
    }

    protected JavaMethod createConstructor(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        String eol = JAXXCompiler.getLineSeparator();
        if (constructorParams != null) {
            code.append("        super(").append(constructorParams).append(");").append(eol);
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super();").append(eol);
            }
        }
        code.append("$initialize();");
        code.append(eol);
        return newMethod(PUBLIC, null, className, code.toString(), false);
    }

    protected JavaMethod createConstructorWithInitialContext(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        String eol = JAXXCompiler.getLineSeparator();
        if (constructorParams != null) {
            if (superclassIsJAXXObject) {
                //TODO-TC20091127 This MUST be a convention in JAXX : all specialized constructor must a second
                //TODO constructor with extra first parameter as parentContext
                constructorParams = "parentContext, " + constructorParams;
            }
            code.append("        super(").append(constructorParams).append(");").append(eol);
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super(parentContext);").append(eol);
            }
        }
        if (!superclassIsJAXXObject) {
            code.append(Util.class.getName()).append(".initContext(this, parentContext);");
            code.append(eol);
        }
        code.append("$initialize();");
        code.append(eol);
        return newMethod(PUBLIC, null, className, code.toString(), false, new JavaArgument(JAXXContext.class.getName(), "parentContext"));
    }

    public JavaMethod createInitializer(JAXXCompiler compiler) throws CompilerException {
        String eol = JAXXCompiler.getLineSeparator();
        StringBuffer code = new StringBuffer();
        CompiledObject root = compiler.getRootObject();
        code.append("if (allComponentsCreated || !contextInitialized) {");
        code.append(eol);
        code.append("    return;");
        code.append(eol);
        code.append("}");
        code.append(eol);
        //TODO-TC20091025 we should remove this if no used anywhere
        code.append("$objectMap.put(").append(compiler.getJavaCode(root.getId())).append(", this);");
        code.append(eol);

        Iterator<CompiledObject> i = compiler.getObjectCreationOrder();
        boolean lastWasMethodCall = false;
        //TODO-TC20091025 should do init of root first ?
//        root.getDecorator().createInitializer(compiler, root, root, code, lastWasMethodCall);
        while (i.hasNext()) {
            CompiledObject object = i.next();
            if (object == root) {
                continue;
            }
            CompiledObjectDecorator decorator = object.getDecorator();
            lastWasMethodCall = decorator.createInitializer(compiler, root, object, code, lastWasMethodCall);
        }
        root.getDecorator().createInitializer(compiler, root, root, code, lastWasMethodCall);
        if (compiler.getInitializer().length() > 0) {
            code.append(compiler.getInitializer());
        }
        code.append("$completeSetup();");
        code.append(eol);
        return newMethod(PRIVATE, "void", "$initialize", code.toString(), false);
    }

    protected JavaMethod createCompleteSetupMethod(JAXXCompiler compiler, JavaFile javaFile, StringBuffer initDataBindings) {
        StringBuffer code = new StringBuffer();
        code.append("allComponentsCreated = true;");
        String eol = JAXXCompiler.getLineSeparator();
        code.append(eol);
        for (CompiledObject object : compiler.getObjects().values()) {
            CompiledObjectDecorator decorator = object.getDecorator();
            code.append(decorator.createCompleteSetupMethod(compiler, object, javaFile, initDataBindings));
        }
        code.append("// init data bindings").append(eol);
        code.append(initDataBindings);

        if (compiler.getLateInitializer().length() > 0) {
            code.append("// late initializer").append(eol);
            code.append(compiler.getLateInitializer()).append(eol);
        }
        //TC-20090313 add an extra method after complete setup
        MethodDescriptor method = compiler.getScriptMethod("$afterCompleteSetup");
        if (method != null) {
            code.append("$afterCompleteSetup();").append(eol);
        }
        return newMethod(PRIVATE, "void", "$completeSetup", code.toString(), false);
    }

    protected JavaMethod createProcessDataBindingMethod(JAXXCompiler compiler, boolean superclassIsJAXXObject) {
        StringBuffer code = new StringBuffer();
        String eol = JAXXCompiler.getLineSeparator();
        //boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(compiler.getRootObject().getObjectClass());
        // the force parameter forces the update to happen even if it is already in activeBindings.  This
        // is used on superclass invocations b/c by the time the call gets to the superclass, it is already
        // marked active and would otherwise be skipped
        if (compiler.getProcessDataBinding().length() > 0) {
            code.append("    if (!$force && $activeBindings.contains($dest)) { ");
            code.append(eol);
            code.append("    return;").append(eol);
            code.append("}").append(eol);
            code.append("$activeBindings.add($dest);").append(eol);
            code.append("try {").append(eol);
            code.append(compiler.getProcessDataBinding().toString());
            if (superclassIsJAXXObject) {
                code.append(" else {").append(eol);
                code.append("        super.processDataBinding($dest, true);").append(eol);
                code.append("    }");
            }
            code.append(eol);
            code.append("} finally {").append(eol);
            code.append("    $activeBindings.remove($dest);").append(eol);
            code.append("}").append(eol);
        } else if (superclassIsJAXXObject) {
            code.append("super.processDataBinding($dest, true);").append(eol);
        }
        return newMethod(PUBLIC, "void", "processDataBinding", code.toString(), superclassIsJAXXObject,
                new JavaArgument("String", "$dest"), new JavaArgument("boolean", "$force"));
    }
}
