package org.nuiton.eugene.config.templates;

/*-
 * #%L
 * EUGene :: Config templates
 * %%
 * Copyright (C) 2016 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.JavaGeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;

import java.util.Set;




/**
 * To generate configuration java files from the options enum file.
 * Created on 15/09/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.config.templates.ApplicationConfigTransformer"
 * @since 3.0
 */
public class ApplicationConfigTransformer extends ObjectModelTransformerToJava {

    private static final Log log = LogFactory.getLog(ApplicationConfigTransformer.class);
    public static final String PROP_OPTION_CLASS_NAME = "optionClassName";
    public static final String PROP_ACTION_CLASS_NAME = "actionClassName";

    @Override
    public void transformFromClass(ObjectModelClass clazz) {

        String packageName = clazz.getPackageName();
        String abstractClassName = "Generated" + clazz.getName();

        if (canGenerate(packageName + "." + abstractClassName)) {

            generateAbstractClass(packageName, abstractClassName, clazz);

        } else {

            if (log.isDebugEnabled()) {
                log.debug("Skip generation for " + abstractClassName);
            }

        }

        String className = clazz.getName();
        if (canGenerate(packageName + "." + className)) {

            generateClass(packageName, className, abstractClassName);

        } else {

            if (log.isDebugEnabled()) {
                log.debug("Skip generation for " + className);
            }

        }
    }

    protected void generateAbstractClass(String packageName, String abstractClassName, ObjectModelClass input) {

        String optionClassName = getProperty(PROP_OPTION_CLASS_NAME);
        String optionClassSimpleName = GeneratorUtil.getSimpleName(optionClassName);

        String actionClassName = getProperty(PROP_ACTION_CLASS_NAME);

        ObjectModelClass output = createAbstractClass(abstractClassName, packageName);

        addInterface(output, "java.util.function.Supplier<ApplicationConfig>");
        addImport(output, "java.util.function.Supplier");
        addImport(output, optionClassName);

        if (log.isInfoEnabled()) {
            log.info("Generate " + output.getQualifiedName());
        }
        addAttribute(output, "applicationConfig", ApplicationConfig.class, "", ObjectModelJavaModifier.PRIVATE);

        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PROTECTED);
        StringBuilder builder = new StringBuilder();
        builder.append(""
                       +"\n"
+"        this.applicationConfig = new ApplicationConfig();\n"
+"        this.applicationConfig.loadDefaultOptions("+optionClassSimpleName+".values());\n"
+"    "
        );
        if (actionClassName != null) {
            addImport(output, actionClassName);
            String actionClassSimpleName = GeneratorUtil.getSimpleName(actionClassName);
            builder.append(""
                       +"\n"
+"        for ("+actionClassSimpleName+" a : "+actionClassSimpleName+".values()) {\n"
+"\n"
+"            for (String alias : a.getAliases()) {\n"
+"                applicationConfig.addActionAlias(alias, a.getAction());\n"
+"            }\n"
+"        }\n"
+"    "
            );
        }
        setOperationBody(constructor, builder.toString());

        ObjectModelOperation getApplicationConfig = addOperation(
                output, "get", ApplicationConfig.class, ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, getApplicationConfig, Override.class);
        setOperationBody(getApplicationConfig, ""
    +"\n"
+"        return applicationConfig;\n"
+"    "
        );

        ObjectModelOperation setOption = addOperation(
                output, "setOption", "void", ObjectModelJavaModifier.PROTECTED);
        addParameter(setOption, String.class, "key");
        addParameter(setOption, Object.class, "attrName");

        setOperationBody(setOption, ""
    +"\n"
+"        applicationConfig.setOption(key, String.valueOf(attrName));\n"
+"    "
        );

        for (ObjectModelAttribute attribute : input.getAttributes()) {

            String attrName = attribute.getName();
            String attrType = attribute.getType();
            String simpleType = JavaGeneratorUtil.getSimpleName(attrType);
            String constantName = JavaGeneratorUtil.getSimpleName(optionClassName) + "." + JavaGeneratorUtil.convertVariableNameToConstantName(attrName);

            addImport(output, attrType);

            // generate getter
            createGetMethod(output,
                            attrName,
                            simpleType,
                            constantName);


            // generate setter
            createSetMethod(output,
                            attrName,
                            simpleType,
                            constantName
            );

        }

    }

    protected void generateClass(String packageName, String className, String abstractClassName) {

        ObjectModelClass output = createClass(className, packageName);
        if (log.isInfoEnabled()) {
            log.info("Generate " + output.getQualifiedName());
        }
        setSuperClass(output, abstractClassName);

    }

    protected boolean canGenerate(String input) {
        return !getResourcesHelper().isJavaFileInClassPath(input);
    }

    private static final Set<String> KNOWN_TYPES = ImmutableSet.of(
            "File",
            "Color",
            "KeyStroke",
            "URL",
            "Class",
            "Date",
            "Time",
            "Timestamp",
            "Locale",
            "Version",
            "String",
            "int",
            "Integer",
            "long",
            "Long",
            "float",
            "Float",
            "boolean",
            "Boolean",
            "byte",
            "Byte",
            "char",
            "Character",
            "double",
            "Double");

    protected void createGetMethod(ObjectModelClass output,
                                   String attrName,
                                   String simpleType,
                                   String constantName) {

        boolean booleanProperty = GeneratorUtil.isBooleanPrimitive(simpleType);
        String methodPrefix = booleanProperty ? JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX : JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;

        String methodName = "getOptionAs" + StringUtils.capitalise(simpleType);
        if (simpleType.equals("String")) {
            methodName = "getOption";
        } else if (simpleType.equals("Integer")) {
            methodName = "getOptionAsInt";
        }

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName(methodPrefix, attrName),
                simpleType,
                ObjectModelJavaModifier.PUBLIC
        );

        if (!KNOWN_TYPES.contains(simpleType)) {
            methodName = "getOptionAsObject";
            setOperationBody(operation, ""
    +"\n"
+"        return ("+simpleType+") applicationConfig."+methodName+"("+simpleType+".class, "+constantName+".getKey());\n"
+"    "
            );
        } else

        {
            setOperationBody(operation, ""
    +"\n"
+"        return applicationConfig."+methodName+"("+constantName+".getKey());\n"
+"    "
            );

            if ("Boolean".equals(simpleType)) {
                operation = addOperation(
                        output,
                        getJavaBeanMethodName(JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX, attrName),
                        "boolean",
                        ObjectModelJavaModifier.PUBLIC
                );

                setOperationBody(operation, ""
    +"\n"
+"        return applicationConfig."+methodName+"("+constantName+".getKey());\n"
+"    "
                );
            }
        }
    }

    protected void createSetMethod(ObjectModelClass output,
                                   String attrName,
                                   String simpleType,
                                   String constantName) {
        boolean booleanProperty = GeneratorUtil.isBooleanPrimitive(simpleType);
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("set", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, simpleType, attrName);

        String methodPrefix = booleanProperty ? JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX : JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;
        String methodName = getJavaBeanMethodName(methodPrefix, attrName);
        setOperationBody(operation, ""
    +"\n"
+"        setOption("+constantName+".getKey(), "+attrName+");\n"
+"    "
        );

        if ("Boolean".equals(simpleType)) {
            operation = addOperation(
                    output,
                    getJavaBeanMethodName("set", attrName),
                    "void",
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "boolean", attrName);

            setOperationBody(operation, ""
    +"\n"
+"        setOption("+constantName+".getKey(), "+attrName+");\n"
+"    "
            );
        }


    }

}
