/*     */ package org.quartz;
/*     */ 
/*     */ import java.io.PrintStream;
/*     */ import java.util.Date;
/*     */ import java.util.List;
/*     */ 
/*     */ public class SimpleTrigger extends Trigger
/*     */ {
/*     */   public static final int MISFIRE_INSTRUCTION_FIRE_NOW = 1;
/*     */   public static final int MISFIRE_INSTRUCTION_RESCHEDULE_NOW_WITH_EXISTING_REPEAT_COUNT = 2;
/*     */   public static final int MISFIRE_INSTRUCTION_RESCHEDULE_NOW_WITH_REMAINING_REPEAT_COUNT = 3;
/*     */   public static final int MISFIRE_INSTRUCTION_RESCHEDULE_NEXT_WITH_REMAINING_COUNT = 4;
/*     */   public static final int MISFIRE_INSTRUCTION_RESCHEDULE_NEXT_WITH_EXISTING_COUNT = 5;
/* 159 */   public static int REPEAT_INDEFINITELY = -1;
/*     */ 
/* 169 */   private Date startTime = null;
/*     */ 
/* 171 */   private Date endTime = null;
/*     */ 
/* 173 */   private Date nextFireTime = null;
/*     */ 
/* 175 */   private Date previousFireTime = null;
/*     */ 
/* 177 */   private int repeatCount = 0;
/*     */ 
/* 179 */   private long repeatInterval = 0L;
/*     */ 
/* 181 */   private int timesTriggered = 0;
/*     */ 
/* 183 */   private boolean complete = false;
/*     */ 
/*     */   public SimpleTrigger()
/*     */   {
/*     */   }
/*     */ 
/*     */   public SimpleTrigger(String name, String group)
/*     */   {
/* 209 */     this(name, group, new Date(), null, 0, 0L);
/*     */   }
/*     */ 
/*     */   public SimpleTrigger(String name, String group, int repeatCount, long repeatInterval)
/*     */   {
/* 220 */     this(name, group, new Date(), null, repeatCount, repeatInterval);
/*     */   }
/*     */ 
/*     */   public SimpleTrigger(String name, String group, Date startTime)
/*     */   {
/* 230 */     this(name, group, startTime, null, 0, 0L);
/*     */   }
/*     */ 
/*     */   public SimpleTrigger(String name, String group, Date startTime, Date endTime, int repeatCount, long repeatInterval)
/*     */   {
/* 254 */     super(name, group);
/*     */ 
/* 256 */     setStartTime(startTime);
/* 257 */     setEndTime(endTime);
/* 258 */     setRepeatCount(repeatCount);
/* 259 */     setRepeatInterval(repeatInterval);
/*     */   }
/*     */ 
/*     */   public SimpleTrigger(String name, String group, String jobName, String jobGroup, Date startTime, Date endTime, int repeatCount, long repeatInterval)
/*     */   {
/* 284 */     super(name, group, jobName, jobGroup);
/*     */ 
/* 286 */     setStartTime(startTime);
/* 287 */     setEndTime(endTime);
/* 288 */     setRepeatCount(repeatCount);
/* 289 */     setRepeatInterval(repeatInterval);
/*     */   }
/*     */ 
/*     */   public Date getStartTime()
/*     */   {
/* 306 */     return this.startTime;
/*     */   }
/*     */ 
/*     */   public void setStartTime(Date startTime)
/*     */   {
/* 318 */     if (startTime == null) {
/* 319 */       throw new IllegalArgumentException("Start time cannot be null");
/*     */     }
/* 321 */     Date eTime = getEndTime();
/* 322 */     if ((eTime != null) && (startTime != null) && (eTime.before(startTime))) {
/* 323 */       throw new IllegalArgumentException("End time cannot be before start time");
/*     */     }
/*     */ 
/* 326 */     this.startTime = startTime;
/*     */   }
/*     */ 
/*     */   public Date getEndTime()
/*     */   {
/* 338 */     return this.endTime;
/*     */   }
/*     */ 
/*     */   public void setEndTime(Date endTime)
/*     */   {
/* 351 */     Date sTime = getStartTime();
/* 352 */     if ((sTime != null) && (endTime != null) && (sTime.after(endTime))) {
/* 353 */       throw new IllegalArgumentException("End time cannot be before start time");
/*     */     }
/*     */ 
/* 356 */     this.endTime = endTime;
/*     */   }
/*     */ 
/*     */   public int getRepeatCount()
/*     */   {
/* 368 */     return this.repeatCount;
/*     */   }
/*     */ 
/*     */   public void setRepeatCount(int repeatCount)
/*     */   {
/* 382 */     if ((repeatCount < 0) && (repeatCount != REPEAT_INDEFINITELY)) {
/* 383 */       throw new IllegalArgumentException("Repeat count must be >= 0, use the constant REPEAT_INDEFINITELY for infinite.");
/*     */     }
/*     */ 
/* 387 */     this.repeatCount = repeatCount;
/*     */   }
/*     */ 
/*     */   public long getRepeatInterval()
/*     */   {
/* 397 */     return this.repeatInterval;
/*     */   }
/*     */ 
/*     */   public void setRepeatInterval(long repeatInterval)
/*     */   {
/* 410 */     if (repeatInterval < 0L) {
/* 411 */       throw new IllegalArgumentException("Repeat interval must be >= 0");
/*     */     }
/*     */ 
/* 414 */     this.repeatInterval = repeatInterval;
/*     */   }
/*     */ 
/*     */   public int getTimesTriggered()
/*     */   {
/* 424 */     return this.timesTriggered;
/*     */   }
/*     */ 
/*     */   public void setTimesTriggered(int timesTriggered)
/*     */   {
/* 434 */     this.timesTriggered = timesTriggered;
/*     */   }
/*     */ 
/*     */   protected boolean validateMisfireInstruction(int misfireInstruction) {
/* 438 */     if (misfireInstruction < 0) {
/* 439 */       return false;
/*     */     }
/*     */ 
/* 442 */     return misfireInstruction <= 5;
/*     */   }
/*     */ 
/*     */   public void updateAfterMisfire(Calendar cal)
/*     */   {
/* 473 */     int instr = getMisfireInstruction();
/* 474 */     if (instr == 0) {
/* 475 */       if (getRepeatCount() == 0) instr = 1;
/* 476 */       else if (getRepeatCount() == REPEAT_INDEFINITELY) instr = 4;
/*     */       else
/*     */       {
/* 479 */         instr = 2;
/*     */       }
/* 480 */     } else if ((instr == 1) && (getRepeatCount() != 0))
/*     */     {
/* 482 */       instr = 3;
/*     */     }
/* 484 */     if (instr == 1) {
/* 485 */       setNextFireTime(new Date());
/* 486 */     } else if (instr == 5) {
/* 487 */       Date newFireTime = getFireTimeAfter(new Date());
/*     */ 
/* 489 */       while ((newFireTime != null) && (cal != null) && (!cal.isTimeIncluded(newFireTime.getTime()))) {
/* 490 */         newFireTime = getFireTimeAfter(newFireTime);
/*     */       }
/* 492 */       setNextFireTime(newFireTime);
/* 493 */     } else if (instr == 4) {
/* 494 */       Date newFireTime = getFireTimeAfter(new Date());
/*     */ 
/* 496 */       while ((newFireTime != null) && (cal != null) && (!cal.isTimeIncluded(newFireTime.getTime()))) {
/* 497 */         newFireTime = getFireTimeAfter(newFireTime);
/*     */       }
/* 499 */       if (newFireTime != null) {
/* 500 */         int timesMissed = computeNumTimesFiredBetween(this.nextFireTime, newFireTime);
/*     */ 
/* 502 */         setTimesTriggered(getTimesTriggered() + timesMissed);
/*     */       }
/*     */ 
/* 505 */       setNextFireTime(newFireTime);
/* 506 */     } else if (instr == 2) {
/* 507 */       Date newFireTime = new Date();
/* 508 */       if ((this.repeatCount != 0) && (this.repeatCount != REPEAT_INDEFINITELY)) {
/* 509 */         setRepeatCount(getRepeatCount() - getTimesTriggered());
/* 510 */         setTimesTriggered(0);
/*     */       }
/* 512 */       setStartTime(newFireTime);
/* 513 */       setNextFireTime(newFireTime);
/* 514 */     } else if (instr == 3) {
/* 515 */       Date newFireTime = new Date();
/*     */ 
/* 517 */       int timesMissed = computeNumTimesFiredBetween(this.nextFireTime, newFireTime);
/*     */ 
/* 520 */       if ((this.repeatCount != 0) && (this.repeatCount != REPEAT_INDEFINITELY)) {
/* 521 */         int remainingCount = getRepeatCount() - (getTimesTriggered() + timesMissed);
/*     */ 
/* 523 */         if (remainingCount <= 0) {
/* 524 */           remainingCount = 0;
/*     */         }
/* 526 */         setRepeatCount(remainingCount);
/* 527 */         setTimesTriggered(0);
/*     */       }
/*     */ 
/* 530 */       if ((getEndTime() != null) && (getEndTime().before(newFireTime))) {
/* 531 */         setEndTime(new Date(newFireTime.getTime() + 50L));
/*     */       }
/* 533 */       setStartTime(newFireTime);
/* 534 */       setNextFireTime(newFireTime);
/*     */     }
/*     */   }
/*     */ 
/*     */   public void triggered(Calendar calendar)
/*     */   {
/* 550 */     this.timesTriggered += 1;
/* 551 */     this.previousFireTime = this.nextFireTime;
/* 552 */     this.nextFireTime = getFireTimeAfter(this.nextFireTime);
/*     */ 
/* 555 */     while ((this.nextFireTime != null) && (calendar != null) && (!calendar.isTimeIncluded(this.nextFireTime.getTime())))
/* 556 */       this.nextFireTime = getFireTimeAfter(this.nextFireTime);
/*     */   }
/*     */ 
/*     */   public void updateWithNewCalendar(Calendar calendar, long misfireThreshold)
/*     */   {
/* 566 */     this.nextFireTime = getFireTimeAfter(this.previousFireTime);
/*     */ 
/* 568 */     Date now = new Date();
/*     */ 
/* 571 */     while ((this.nextFireTime != null) && (calendar != null) && (!calendar.isTimeIncluded(this.nextFireTime.getTime()))) {
/* 572 */       this.nextFireTime = getFireTimeAfter(this.nextFireTime);
/*     */     }
/*     */ 
/* 575 */     if ((this.nextFireTime != null) && (this.nextFireTime.before(now))) {
/* 576 */       long diff = now.getTime() - this.nextFireTime.getTime();
/* 577 */       if (diff >= misfireThreshold)
/* 578 */         this.nextFireTime = getFireTimeAfter(this.nextFireTime);
/*     */     }
/*     */   }
/*     */ 
/*     */   public Date computeFirstFireTime(Calendar calendar)
/*     */   {
/* 603 */     this.nextFireTime = getStartTime();
/*     */ 
/* 606 */     while ((this.nextFireTime != null) && (calendar != null) && (!calendar.isTimeIncluded(this.nextFireTime.getTime()))) {
/* 607 */       this.nextFireTime = getFireTimeAfter(this.nextFireTime);
/*     */     }
/*     */ 
/* 610 */     return this.nextFireTime;
/*     */   }
/*     */ 
/*     */   public int executionComplete(JobExecutionContext context, JobExecutionException result)
/*     */   {
/* 636 */     if ((result != null) && (result.refireImmediately())) {
/* 637 */       return 1;
/*     */     }
/* 639 */     if ((result != null) && (result.unscheduleFiringTrigger())) {
/* 640 */       return 2;
/*     */     }
/* 642 */     if ((result != null) && (result.unscheduleAllTriggers())) {
/* 643 */       return 4;
/*     */     }
/* 645 */     if (!mayFireAgain()) return 3;
/*     */ 
/* 647 */     return 0;
/*     */   }
/*     */ 
/*     */   public Date getNextFireTime()
/*     */   {
/* 659 */     return this.nextFireTime;
/*     */   }
/*     */ 
/*     */   public Date getPreviousFireTime()
/*     */   {
/* 669 */     return this.previousFireTime;
/*     */   }
/*     */ 
/*     */   public void setNextFireTime(Date nextFireTime)
/*     */   {
/* 682 */     this.nextFireTime = nextFireTime;
/*     */   }
/*     */ 
/*     */   public void setPreviousFireTime(Date previousFireTime)
/*     */   {
/* 695 */     this.previousFireTime = previousFireTime;
/*     */   }
/*     */ 
/*     */   public Date getFireTimeAfter(Date afterTime)
/*     */   {
/* 706 */     if (this.complete) return null;
/*     */ 
/* 708 */     if ((this.timesTriggered > this.repeatCount) && (this.repeatCount != REPEAT_INDEFINITELY)) {
/* 709 */       return null;
/*     */     }
/* 711 */     if (afterTime == null) afterTime = new Date();
/*     */ 
/* 713 */     if ((this.repeatCount == 0) && (afterTime.compareTo(getStartTime()) >= 0)) {
/* 714 */       return null;
/*     */     }
/* 716 */     long startMillis = getStartTime().getTime();
/* 717 */     long afterMillis = afterTime.getTime();
/* 718 */     long endMillis = getEndTime() == null ? 9223372036854775807L : getEndTime().getTime();
/*     */ 
/* 721 */     if (endMillis <= afterMillis) return null;
/*     */ 
/* 723 */     if ((startMillis < afterMillis) && (this.repeatCount == 0)) return null;
/*     */ 
/* 725 */     if (afterMillis < startMillis) return new Date(startMillis);
/*     */ 
/* 727 */     long numberoftimesexecutedplusone = (afterMillis - startMillis) / this.repeatInterval + 1L;
/*     */ 
/* 729 */     if ((numberoftimesexecutedplusone > this.repeatCount) && (this.repeatCount != REPEAT_INDEFINITELY)) {
/* 730 */       return null;
/*     */     }
/* 732 */     Date time = new Date(numberoftimesexecutedplusone * this.repeatInterval + startMillis);
/*     */ 
/* 735 */     if (endMillis <= time.getTime()) return null;
/*     */ 
/* 737 */     return time;
/*     */   }
/*     */ 
/*     */   public Date getFireTimeBefore(Date end)
/*     */   {
/* 748 */     if (end.getTime() < getStartTime().getTime()) return null;
/*     */ 
/* 750 */     int numFires = computeNumTimesFiredBetween(getStartTime(), end);
/*     */ 
/* 752 */     return new Date(getStartTime().getTime() + numFires * this.repeatInterval);
/*     */   }
/*     */ 
/*     */   public int computeNumTimesFiredBetween(Date start, Date end) {
/* 756 */     long time = end.getTime() - start.getTime();
/*     */ 
/* 758 */     return (int)(time / this.repeatInterval);
/*     */   }
/*     */ 
/*     */   public Date getFinalFireTime()
/*     */   {
/* 772 */     if (this.repeatCount == 0) return this.startTime;
/*     */ 
/* 774 */     if ((this.repeatCount == REPEAT_INDEFINITELY) && (getEndTime() == null)) {
/* 775 */       return null;
/*     */     }
/* 777 */     if ((this.repeatCount == REPEAT_INDEFINITELY) && (getEndTime() == null)) return null;
/* 778 */     if (this.repeatCount == REPEAT_INDEFINITELY) {
/* 779 */       return getFireTimeBefore(getEndTime());
/*     */     }
/* 781 */     long lastTrigger = this.startTime.getTime() + this.repeatCount * this.repeatInterval;
/*     */ 
/* 783 */     if ((getEndTime() == null) || (lastTrigger < getEndTime().getTime())) return new Date(lastTrigger);
/*     */ 
/* 786 */     return getFireTimeBefore(getEndTime());
/*     */   }
/*     */ 
/*     */   public boolean mayFireAgain()
/*     */   {
/* 796 */     return getNextFireTime() != null;
/*     */   }
/*     */ 
/*     */   public void validate()
/*     */     throws SchedulerException
/*     */   {
/* 809 */     super.validate();
/*     */ 
/* 811 */     if ((this.repeatCount != 0) && (this.repeatInterval < 1L))
/* 812 */       throw new SchedulerException("Repeat Interval cannot be zero.", 100);
/*     */   }
/*     */ 
/*     */   public static void main(String[] args)
/*     */     throws Exception
/*     */   {
/* 820 */     Date sdt = new Date();
/*     */ 
/* 822 */     Date edt = new Date(sdt.getTime() + 55000L);
/*     */ 
/* 824 */     SimpleTrigger st = new SimpleTrigger("t", "g", "j", "g", sdt, edt, 10, 10000L);
/*     */ 
/* 827 */     System.err.println();
/*     */ 
/* 829 */     st.computeFirstFireTime(null);
/*     */ 
/* 831 */     System.err.println("lastTime=" + st.getFinalFireTime());
/*     */ 
/* 833 */     List times = TriggerUtils.computeFireTimes(st, null, 50);
/*     */ 
/* 835 */     for (int i = 0; i < times.size(); i++)
/* 836 */       System.err.println("firetime = " + times.get(i));
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.quartz.SimpleTrigger
 * JD-Core Version:    0.6.0
 */