package org.chorem.android.saymytexts;

/*
 * #%L
 * Say My Texts
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import android.app.Service;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothHeadset;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.media.AudioManager;
import android.os.IBinder;
import android.preference.PreferenceManager;
import android.speech.tts.TextToSpeech;
import android.speech.tts.UtteranceProgressListener;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;
import android.util.Log;
import com.google.common.base.Function;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Service which say out loud the text passed in the intent
 *
 * @author Kevin Morin (Code Lutin)
 * @since 1.0
 */
public class SayMyTextService extends Service implements TextToSpeech.OnInitListener, SharedPreferences.OnSharedPreferenceChangeListener {

    private static final String TAG = "SayMyTextService";

    public static final String ACTION_READ_SMS = "org.chorem.android.saymytexts.READ_SMS";
    public static final String ACTION_READ_NEXT_SMS = "org.chorem.android.saymytexts.READ_NEXT_SMS";
    public static final String ACTION_MANAGE_BT_DEVICE = "org.chorem.android.saymytexts.ADD_BT_DEVICE";
    public static final String ACTION_REASK_ACTION = "org.chorem.android.saymytexts.REASK_ACTION";
    public static final String ACTION_DICTATE_SMS = "org.chorem.android.saymytexts.DICTATE_SMS";
    public static final String ACTION_CONFIRM_SMS_SENDING = "org.chorem.android.saymytexts.CONFIRM_SMS_SENDING";

    /** SMS to read */
    public static final String INTENT_EXTRA_SMS = "sms";
    /** Bluetooth device which has just connected or disconnected */
    public static final String INTENT_EXTRA_BT_DEVICE = "btDevice";
    /** If true, the device has just connected, else disconnected */
    public static final String INTENT_EXTRA_ADD_BT_DEVICE = "addBtDevice";
    /** Bluetooth device which has just connected or disconnected */
    public static final String INTENT_EXTRA_DICTATED_MESSAGE = "dictatedMessage";
    /** Attempt number: if set, it means that the user said something not understandable, so ask again */
    public static final String INTENT_EXTRA_ATTEMPT_NUMBER = "attemptNumber";

    /** utterance id when the bluetooth device is connected */
//    protected static final String BT_UTTERANCE_ID = "btUtteranceId";
    protected static final String BT_ASK_NEXT_ACTION_UTTERANCE_ID = "btAskNextActionUtteranceId";
    protected static final String ASK_NEXT_ACTION_UTTERANCE_ID = "askNextActionUtteranceId";
    protected static final String OTHER_UTTERANCE_ID = "oherUtteranceId";

    protected int maxAttemptNumber;

    protected String readingProfile;

    protected boolean heisendroidModeEnabled;

    protected boolean interactionEnabled;

    protected AudioManager audioManager;

    /** null if the texttospeech is not initialized */
    protected Boolean canSpeak = null;

    // true if music was playing when the first message in queue arrived
    protected boolean musicWasActive;

    // true if the speaker was on when the first message in queue arrived
    protected boolean speakerWasOn;

    protected TextToSpeech textToSpeech;

    /** texts to read, received before the textospeech is ready or while a call is in progress */
    protected List<SMS> awaitingTexts = new ArrayList<>();

    /** bluetooth devices which are currently connected */
    protected Map<BluetoothDevice, Integer> bluetoothDevices = new HashMap<>();

    /**
     * Listener to call state change
     */
    protected final PhoneStateListener callStateListener = new PhoneStateListener() {
        @Override
        public void onCallStateChanged(int state, String incomingNumber) {
            super.onCallStateChanged(state, incomingNumber);
            if (canSpeak != null) {
                setCanSpeak(state == TelephonyManager.CALL_STATE_IDLE);
            }
        }
    };

//    BluetoothHeadset mBluetoothHeadset;

    @Override
    public void onCreate() {
        super.onCreate();

        textToSpeech = new TextToSpeech(this, this);

        audioManager = (AudioManager) getSystemService(AUDIO_SERVICE);

        TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
        tm.listen(callStateListener, PhoneStateListener.LISTEN_CALL_STATE);

        IntentFilter intentFilter = new IntentFilter(SayNextActionBroadcastReceiver.ACTION_SAY_NEXT_ACTION);
        registerReceiver(new SayNextActionBroadcastReceiver(), intentFilter);

        intentFilter = new IntentFilter(DictateSmsBroadcastReceiver.ACTION_DICTATE_SMS);
        registerReceiver(new DictateSmsBroadcastReceiver(), intentFilter);

        SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(this);
        String key = getString(R.string.preference_voice_recognizer_max_attempt_number_key);
        String maxAttemptValue = sharedPref.getString(key, null);
        try {
            maxAttemptNumber = Integer.parseInt(maxAttemptValue);
        } catch (NumberFormatException e) {
            maxAttemptNumber = 3;
        }

        String[] readingProfileValues = getResources().getStringArray(R.array.preferences_reading_profile_values);
        String readingProfileKey = getString(R.string.preference_reading_profile_key);
        readingProfile = sharedPref.getString(readingProfileKey, readingProfileValues[0]);

        String heisendroidModeEnabledKey = getString(R.string.preference_enable_heisendroid_mode_key);
        heisendroidModeEnabled = sharedPref.getBoolean(heisendroidModeEnabledKey, true);

        String interactionPrefKey = getString(R.string.preference_enable_interaction_key);
        interactionEnabled = sharedPref.getBoolean(interactionPrefKey, true);

        sharedPref.registerOnSharedPreferenceChangeListener(this);

//// Get the default adapter
//        final BluetoothAdapter mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
//
//// Define Service Listener of BluetoothProfile
//        BluetoothProfile.ServiceListener mProfileListener = new BluetoothProfile.ServiceListener() {
//            public void onServiceConnected(int profile, BluetoothProfile proxy) {
//                if (profile == BluetoothProfile.HEADSET) {
//                    mBluetoothHeadset = (BluetoothHeadset) proxy;
//                    List<BluetoothDevice> devices = mBluetoothHeadset.getConnectedDevices();
//                    for ( final BluetoothDevice dev : devices ) {
//                        if (mBluetoothHeadset.isAudioConnected(dev)) {
//                            bluetoothDevices.put(dev, dev.getBluetoothClass().getDeviceClass());
//                        }
//                    }
//                    mBluetoothAdapter.closeProfileProxy(BluetoothProfile.HEADSET, mBluetoothHeadset);
//                }
//            }
//            public void onServiceDisconnected(int profile) {
//                if (profile == BluetoothProfile.HEADSET) {
//                    mBluetoothHeadset = null;
//                }
//            }
//        };
//
//// Establish connection to the proxy.
//        mBluetoothAdapter.getProfileProxy(this, mProfileListener, BluetoothProfile.HEADSET);

    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
        tm.listen(callStateListener, PhoneStateListener.LISTEN_NONE);
        textToSpeech.shutdown();

        SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(this);
        sharedPref.unregisterOnSharedPreferenceChangeListener(this);
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        int result = super.onStartCommand(intent, flags, startId);

        if (!SayMyTextsUtils.checkVoiceRecognition(this)) {
            SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(this);
            sharedPref.edit().putBoolean(getString(R.string.preference_enable_interaction_key), false).commit();
        }

        if (intent != null) {
            String action = intent.getAction();
            Log.d(TAG, "action " + action);

            final SMS sms = (SMS) intent.getSerializableExtra(INTENT_EXTRA_SMS);
            int attemptNumber = intent.getIntExtra(INTENT_EXTRA_ATTEMPT_NUMBER, 0);

            switch (action) {
                case ACTION_MANAGE_BT_DEVICE:
                    BluetoothDevice device = intent.getParcelableExtra(INTENT_EXTRA_BT_DEVICE);
                    // if a device is passed to the service
                    // add or remove the device from the connected devices
                    if (device != null) {
                        boolean addBtDevice = intent.getBooleanExtra(INTENT_EXTRA_ADD_BT_DEVICE, false);
                        if (addBtDevice) {
                            bluetoothDevices.put(device, device.getBluetoothClass().getDeviceClass());
                        } else {
                            bluetoothDevices.remove(device);
                        }
                    }
                    break;

                case ACTION_REASK_ACTION:
                    askForActionAfterReading(sms, !bluetoothDevices.isEmpty(), ++attemptNumber);
                    break;

                case ACTION_DICTATE_SMS:
                    dictateSMS(sms, attemptNumber + 1);
                    break;

                case ACTION_CONFIRM_SMS_SENDING:
                    // if a message has just been dictated
                    final String dictatedMessage = intent.getStringExtra(INTENT_EXTRA_DICTATED_MESSAGE);
                    askSendingConfirmation(dictatedMessage, sms, ++attemptNumber);
                    break;

                case ACTION_READ_NEXT_SMS:
                    setCanSpeak(true);
                    break;

                default:
                    boolean readingEnabled;
                    String[] readingProfileValues = getResources().getStringArray(R.array.preferences_reading_profile_values);
                    if (readingProfileValues[0].equals(readingProfile)) {
                        readingEnabled = true;

                    } else if (readingProfileValues[1].equals(readingProfile)) {
                        readingEnabled = audioManager.isWiredHeadsetOn() || !bluetoothDevices.isEmpty();

                    } else if (readingProfileValues[2].equals(readingProfile)) {
                        readingEnabled = audioManager.isWiredHeadsetOn();

                    } else {
                        readingEnabled = false;
                    }

                    if (readingEnabled) {
                        if (!Boolean.FALSE.equals(canSpeak)) {
                            musicWasActive = audioManager.isMusicActive();
                            speakerWasOn = audioManager.isSpeakerphoneOn();

                            if (musicWasActive) {
                                audioManager.setStreamMute(AudioManager.STREAM_MUSIC, musicWasActive);
                            }

                            if (readingProfileValues[0].equals(readingProfile)
                                    && !audioManager.isWiredHeadsetOn() && bluetoothDevices.isEmpty()) {
                                audioManager.setMode(AudioManager.MODE_IN_CALL);
                                audioManager.setSpeakerphoneOn(true);
                            }
                        }

                        if (Boolean.TRUE.equals(canSpeak)) {
                            requestReading(sms);

                        } else {
                            awaitingTexts.add(sms);
                        }
                    }
            }

            result = START_STICKY;
        }

        return result;
    }

    @Override
    public void onInit(int status) {
        if (status == TextToSpeech.SUCCESS) {
            TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
            setCanSpeak(tm.getCallState() == TelephonyManager.CALL_STATE_IDLE);

        } else {
            setCanSpeak(null);
            textToSpeech = new TextToSpeech(this, this);
        }
    }

    /**
     * Sets if the texts can be read
     *
     * @param canSpeak null if the texttospeech is not ready,
     *                 false if a call is in progress
     *                 true otherwise
     */
    protected void setCanSpeak(Boolean canSpeak) {
        this.canSpeak = canSpeak;
        if (Boolean.TRUE.equals(canSpeak)) {
            if (!awaitingTexts.isEmpty()) {
                SMS sms = awaitingTexts.remove(0);
                requestReading(sms);

            } else {
                if (!bluetoothDevices.isEmpty()) {
                    audioManager.stopBluetoothSco();
                }
                if (musicWasActive) {
                    audioManager.setStreamMute(AudioManager.STREAM_MUSIC, false);
                }
                audioManager.setSpeakerphoneOn(speakerWasOn);
                audioManager.setMode(AudioManager.MODE_NORMAL);
            }
        }
    }

    /**
     * Requests the reading of one text through the wired headset or bluetooth device
     * @param sms the text to read
     */
    protected void requestReading(final SMS sms) {
        if (bluetoothDevices.isEmpty()) {
            readText(sms, false);
        } else {
            requestBluetoothSpeakingActivation(new Function<Void, Void>() {
                @Override
                public Void apply(Void input) {
                    readText(sms, true);
                    return null;
                }
            });
        }
    }

    /**
     * Starts the connection with the bluetooth device and requests the reading
     * @param callback the function called when the bluetooth is ready
     */
    protected void requestBluetoothSpeakingActivation(final Function<Void, Void> callback) {
        registerReceiver(new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                int state = intent.getExtras().getInt(AudioManager.EXTRA_SCO_AUDIO_STATE);
                if (state == AudioManager.SCO_AUDIO_STATE_CONNECTED) {
                    context.unregisterReceiver(this);
                    callback.apply(null);
                }
            }
        }, new IntentFilter(AudioManager.ACTION_SCO_AUDIO_STATE_UPDATED));
        audioManager.setMode(AudioManager.MODE_IN_CALL);
        audioManager.startBluetoothSco();
    }

    /**
     * Reads the texts out loud
     * @param sms the text to read
     * @param btConnected if true, adds the utterance id for the bluetooth device
     */
    protected void readText(final SMS sms, boolean btConnected) {
        // disable the reading of the nexts sms while reading the current one
        setCanSpeak(false);

        textToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
            @Override
            public void onStart(String utteranceId) {
            }

            @Override
            public void onError(String utteranceId) {
                Log.e(TAG, "Error speaking: " + utteranceId);
            }

            @Override
            public void onDone(String utteranceId) {
                Log.d(TAG, "done");
                setCanSpeak(true);
            }
        });

        HashMap<String, String> params = new HashMap<>();
        params.put(TextToSpeech.Engine.KEY_PARAM_STREAM, String.valueOf(AudioManager.STREAM_VOICE_CALL));

        if (heisendroidModeEnabled) {
            textToSpeech.setLanguage(Locale.US);
            textToSpeech.setSpeechRate(0.3f);
            textToSpeech.setPitch(0.1f);
            textToSpeech.speak("Say my text.", TextToSpeech.QUEUE_ADD, params);
        }

        if (!interactionEnabled && !heisendroidModeEnabled) {
            params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, OTHER_UTTERANCE_ID);
        }
        textToSpeech.setLanguage(Locale.getDefault());
        textToSpeech.setSpeechRate(1f);
        textToSpeech.setPitch(1f);
        String text = getString(R.string.sms_received, sms.getSenderName(), sms.getMessage());
        textToSpeech.speak(text, TextToSpeech.QUEUE_ADD, params);

        if (heisendroidModeEnabled) {
            if (interactionEnabled) {
                params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, OTHER_UTTERANCE_ID);
            }
            textToSpeech.setLanguage(Locale.US);
            textToSpeech.setSpeechRate(0.3f);
            textToSpeech.setPitch(0.1f);
            textToSpeech.speak("You're goddamn right.", TextToSpeech.QUEUE_ADD, params);
        }

        if (interactionEnabled) {
            askForActionAfterReading(sms, btConnected, 1);
        }
    }

    protected void askForActionAfterReading(final SMS sms, boolean btConnected, final int attemptNumber) {
        textToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
            @Override
            public void onStart(String utteranceId) {
            }

            @Override
            public void onError(String utteranceId) {
                Log.e(TAG, "Error speaking: " + utteranceId);
            }

            @Override
            public void onDone(String utteranceId) {
                if (ASK_NEXT_ACTION_UTTERANCE_ID.equals(utteranceId) ||
                        BT_ASK_NEXT_ACTION_UTTERANCE_ID.equals(utteranceId)) {
                    Intent sayaction = new Intent(SayNextActionBroadcastReceiver.ACTION_SAY_NEXT_ACTION);
                    sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_SMS, sms);
                    sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_ATTEMPT_NUMBER, attemptNumber);
                    sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_FALLBACK_ACTION, ACTION_REASK_ACTION);

                    sendBroadcast(sayaction);
                }
            }
        });

        HashMap<String, String> params = new HashMap<>();
        params.put(TextToSpeech.Engine.KEY_PARAM_STREAM, String.valueOf(AudioManager.STREAM_VOICE_CALL));

        textToSpeech.setLanguage(Locale.getDefault());
        textToSpeech.setSpeechRate(1f);
        textToSpeech.setPitch(1f);

        if (attemptNumber > 1) {
            textToSpeech.speak(getString(R.string.voice_not_recognized), TextToSpeech.QUEUE_ADD, params);
        }

        if (attemptNumber <= maxAttemptNumber) {
            params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID,
                       btConnected ? BT_ASK_NEXT_ACTION_UTTERANCE_ID : ASK_NEXT_ACTION_UTTERANCE_ID);
            textToSpeech.speak(getString(R.string.ask_next_action), TextToSpeech.QUEUE_ADD, params);

        } else {
            setCanSpeak(true);
        }
    }

    protected void dictateSMS(final SMS sms, final int attemptNumber) {
        Log.d(TAG, "dictateSMS " );
        textToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
            @Override
            public void onStart(String utteranceId) {
            }

            @Override
            public void onError(String utteranceId) {
                Log.e(TAG, "Error speaking: " + utteranceId);
            }

            @Override
            public void onDone(String utteranceId) {
                Intent dictateAction = new Intent(DictateSmsBroadcastReceiver.ACTION_DICTATE_SMS);
                dictateAction.putExtra(DictateSmsBroadcastReceiver.INTENT_EXTRA_SMS, sms);
                dictateAction.putExtra(DictateSmsBroadcastReceiver.INTENT_EXTRA_ATTEMPT_NUMBER, attemptNumber);
                sendBroadcast(dictateAction);

            }
        });

        HashMap<String, String> params = new HashMap<>();
        params.put(TextToSpeech.Engine.KEY_PARAM_STREAM, String.valueOf(AudioManager.STREAM_VOICE_CALL));

        if (attemptNumber > 1) {
            textToSpeech.speak(getString(R.string.voice_not_recognized), TextToSpeech.QUEUE_ADD, params);
        }

        if (attemptNumber <= maxAttemptNumber) {
            params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, OTHER_UTTERANCE_ID);
            textToSpeech.setLanguage(Locale.getDefault());
            textToSpeech.setSpeechRate(1f);
            textToSpeech.setPitch(1f);
            textToSpeech.speak(getString(R.string.dictate_sms), TextToSpeech.QUEUE_ADD, params);

        } else {
            setCanSpeak(true);
        }

    }

    protected void askSendingConfirmation(final String message, final SMS originSms, final int attemptNumber) {
        Log.d(TAG, "askSendingConfirmation " + message);
        textToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
            @Override
            public void onStart(String utteranceId) {
            }

            @Override
            public void onError(String utteranceId) {
                Log.e(TAG, "Error speaking: " + utteranceId);
            }

            @Override
            public void onDone(String utteranceId) {
                Intent sayaction = new Intent(SayNextActionBroadcastReceiver.ACTION_SAY_NEXT_ACTION);
                sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_SMS, originSms);
                sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_MESSAGE, message);
                sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_ATTEMPT_NUMBER, attemptNumber);
                sayaction.putExtra(SayNextActionBroadcastReceiver.INTENT_EXTRA_FALLBACK_ACTION, ACTION_CONFIRM_SMS_SENDING);
                sendBroadcast(sayaction);

            }
        });

        HashMap<String, String> params = new HashMap<>();
        params.put(TextToSpeech.Engine.KEY_PARAM_STREAM, String.valueOf(AudioManager.STREAM_VOICE_CALL));

        textToSpeech.setLanguage(Locale.getDefault());
        textToSpeech.setSpeechRate(1f);
        textToSpeech.setPitch(1f);

        if (attemptNumber > 1) {
            textToSpeech.speak(getString(R.string.voice_not_recognized), TextToSpeech.QUEUE_ADD, params);
        }

        if (attemptNumber <= maxAttemptNumber) {
            params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, OTHER_UTTERANCE_ID);
            String text = getString(R.string.send_sms_confirmation, message);
            textToSpeech.speak(text, TextToSpeech.QUEUE_ADD, params);

        } else {
            setCanSpeak(true);
        }

    }

    @Override
    public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String key) {
        if (key.equals(getString(R.string.preference_voice_recognizer_max_attempt_number_key))) {
            String maxAttemptValue = sharedPreferences.getString(key, null);
            try {
                maxAttemptNumber = Integer.parseInt(maxAttemptValue);
            } catch (NumberFormatException e) {
                maxAttemptNumber = 3;
            }

        } else if (key.equals(getString(R.string.preference_reading_profile_key))) {
            String[] readingProfileValues =
                    getResources().getStringArray(R.array.preferences_reading_profile_values);
            readingProfile = sharedPreferences.getString(key, readingProfileValues[0]);

        } else if (key.equals(getString(R.string.preference_enable_heisendroid_mode_key))) {
            heisendroidModeEnabled = sharedPreferences.getBoolean(key, true);

        } else if (key.equals(getString(R.string.preference_enable_interaction_key))) {
            interactionEnabled = sharedPreferences.getBoolean(key, true);
        }
    }
}
