package org.chorem.android.saymytexts;

/*
 * #%L
 * Say My Texts
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import android.app.Service;
import android.bluetooth.BluetoothDevice;
import android.content.BroadcastReceiver;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.media.AudioManager;
import android.net.Uri;
import android.os.IBinder;
import android.preference.PreferenceManager;
import android.provider.BaseColumns;
import android.provider.ContactsContract;
import android.speech.tts.TextToSpeech;
import android.speech.tts.UtteranceProgressListener;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;
import android.util.Log;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Service which say out loud the text passed in the intent
 *
 * @author Kevin Morin (Code Lutin)
 * @since 1.0
 */
public class SayMyTextService extends Service implements TextToSpeech.OnInitListener {

    private static final String TAG = "SayMyTextService";

    /** Body of the SMS */
    public static final String INTENT_EXTRA_SMS_BODY = "smsBody";
    /** Number of the sender of the SMS */
    public static final String INTENT_EXTRA_SMS_SENDER = "smsSender";
    /** Bluetooth device which has just connected or disconnected */
    public static final String INTENT_EXTRA_BT_DEVICE = "btDevice";
    /** If true, the device has just connected, else disconnected */
    public static final String INTENT_EXTRA_ADD_BT_DEVICE = "addBtDevice";

    /** utterance id when the bluetooth device is connected */
    protected static final String BT_UTTERANCE_ID = "btUtteranceId";

    protected AudioManager audioManager;

    /** null if the texttospeech is not initialized */
    protected Boolean canSpeak = null;

    protected TextToSpeech textToSpeech;

    /** texts to read, received before the textospeech is ready or while a call is in progress */
    protected List<String> awaitingTexts = new ArrayList<>();

    /** bluetooth devices which are currently connected */
    protected Map<BluetoothDevice, Integer> bluetoothDevices = new HashMap<>();

    /**
     * Listener to clal state change
     */
    protected final PhoneStateListener callStateListener = new PhoneStateListener() {
        @Override
        public void onCallStateChanged(int state, String incomingNumber) {
            super.onCallStateChanged(state, incomingNumber);
            if (canSpeak != null) {
                setCanSpeak(state == TelephonyManager.CALL_STATE_IDLE);
            }
        }
    };

    @Override
    public void onCreate() {
        super.onCreate();

        textToSpeech = new TextToSpeech(this, this);

        audioManager = (AudioManager) getSystemService(AUDIO_SERVICE);

        TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
        tm.listen(callStateListener, PhoneStateListener.LISTEN_CALL_STATE);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
        tm.listen(callStateListener, PhoneStateListener.LISTEN_NONE);
        textToSpeech.shutdown();
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        int result = super.onStartCommand(intent, flags, startId);

        SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(this);
        String readingEnabledKey = getString(R.string.preference_enable_reading_key);
        boolean readingEnabled = sharedPref.getBoolean(readingEnabledKey, true);

        if (intent != null) {
            BluetoothDevice device = intent.getParcelableExtra(INTENT_EXTRA_BT_DEVICE);
            // if a device is passed to the service
            // add or remove the device from the connected devices
            if (device != null) {
                boolean addBtDevice = intent.getBooleanExtra(INTENT_EXTRA_ADD_BT_DEVICE, false);
                if (addBtDevice) {
                    bluetoothDevices.put(device, device.getBluetoothClass().getDeviceClass());
                } else {
                    bluetoothDevices.remove(device);
                }


            // else if the user enabled the reading and
            // if the headset is plugged, or if there is a bluetooth device connected
            } else if (readingEnabled && (audioManager.isWiredHeadsetOn() || !bluetoothDevices.isEmpty())) {

                String sms = intent.getStringExtra(INTENT_EXTRA_SMS_BODY);
                String sender = intent.getStringExtra(INTENT_EXTRA_SMS_SENDER);
                sender = getContactDisplayNameByNumber(sender);
                String text = getString(R.string.sms_received, sender, sms);

                if (canSpeak != null && canSpeak) {
                    requestReading(text);
                } else {
                    awaitingTexts.add(text);
                }
            }

            result = START_STICKY;
        }

        return result;
    }

    @Override
    public void onInit(int status) {
        if (status == TextToSpeech.SUCCESS) {
            // init texttospeech
            textToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
                @Override
                public void onStart(String utteranceId) {
                }

                @Override
                public void onError(String utteranceId) {
                    Log.e(TAG, "Error speaking: " + utteranceId);
                }

                @Override
                public void onDone(String utteranceId) {
                    // when the text has been read by the bluetooth device, stop the connection
                    audioManager.stopBluetoothSco();
                    audioManager.setMode(AudioManager.MODE_NORMAL);
                }
            });

            TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
            setCanSpeak(tm.getCallState() == TelephonyManager.CALL_STATE_IDLE);

        } else {
            setCanSpeak(null);
            textToSpeech = new TextToSpeech(this, this);
        }
    }

    /**
     * Sets if the texts can be read
     *
     * @param canSpeak null if the texttospeech is not ready,
     *                 false if a call is in progress
     *                 true otherwise
     */
    protected void setCanSpeak(Boolean canSpeak) {
        this.canSpeak = canSpeak;
        if (canSpeak != null && canSpeak) {
            requestReading(awaitingTexts);
            awaitingTexts.clear();
        }
    }

    /**
     * Requests the reading of one text through the wired headset or bluetooth device
     * @param text the text to read
     */
    protected void requestReading(String text) {
        requestReading(Collections.singletonList(text));
    }

    /**
     * Requests the reading of a list of texts through the wired headset or bluetooth device
     * @param texts the texts to read
     */
    protected void requestReading(List<String> texts) {
        if (bluetoothDevices.isEmpty()) {
            readText(texts, false);
        } else {
            requestReadingOverBt(texts);
        }
    }

    /**
     * Starts the connection with the bluetooth device and requests the reading
     * @param texts the texts to read
     */
    protected void requestReadingOverBt(final List<String> texts) {
        registerReceiver(new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                int state = intent.getExtras().getInt(AudioManager.EXTRA_SCO_AUDIO_STATE);
                if (state == AudioManager.SCO_AUDIO_STATE_CONNECTED) {
                    context.unregisterReceiver(this);
                    readText(texts, true);
                }
            }
        }, new IntentFilter(AudioManager.ACTION_SCO_AUDIO_STATE_UPDATED));
        audioManager.setMode(AudioManager.MODE_IN_CALL);
        audioManager.startBluetoothSco();
    }

    /**
     * Reads the texts out loud
     * @param texts the texts to read
     * @param btConnected if true, adds the utterance id for the bluetooth device
     */
    protected void readText(List<String> texts, boolean btConnected) {
        HashMap<String, String> params = new HashMap<>();
        params.put(TextToSpeech.Engine.KEY_PARAM_STREAM, String.valueOf(AudioManager.STREAM_VOICE_CALL));
        if (btConnected) {
            params.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, BT_UTTERANCE_ID);
        }

        SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(this);
        String heisendroidModeEnabledKey = getString(R.string.preference_enable_heisendroid_mode_key);
        boolean heisendroidModeEnabled = sharedPref.getBoolean(heisendroidModeEnabledKey, true);

        for (String text : texts) {
            if (heisendroidModeEnabled) {
                textToSpeech.setLanguage(Locale.US);
                textToSpeech.setSpeechRate(0.3f);
                textToSpeech.setPitch(0.1f);
                textToSpeech.speak("Say my text.", TextToSpeech.QUEUE_ADD, params);
            }

            textToSpeech.setLanguage(Locale.getDefault());
            textToSpeech.setSpeechRate(1f);
            textToSpeech.setPitch(1f);
            textToSpeech.speak(text, TextToSpeech.QUEUE_ADD, params);

            if (heisendroidModeEnabled) {
                textToSpeech.setLanguage(Locale.US);
                textToSpeech.setSpeechRate(0.3f);
                textToSpeech.setPitch(0.1f);
                textToSpeech.speak("You're goddamn right.", TextToSpeech.QUEUE_ADD, params);
            }
        }
    }

    /**
     * Finds the contact name in the contact book
     * @param number the number of the contact
     * @return the name if the contact is known, the number otherwise
     */
    protected String getContactDisplayNameByNumber(String number) {
        Uri uri = Uri.withAppendedPath(ContactsContract.PhoneLookup.CONTENT_FILTER_URI, Uri.encode(number));
        String name = number;

        ContentResolver contentResolver = getContentResolver();
        Cursor contactLookup = contentResolver.query(uri,
                                                     new String[] { BaseColumns._ID, ContactsContract.PhoneLookup.DISPLAY_NAME },
                                                     null, null, null);

        try {
            if (contactLookup != null && contactLookup.getCount() > 0) {
                contactLookup.moveToNext();
                name = contactLookup.getString(contactLookup.getColumnIndex(ContactsContract.Data.DISPLAY_NAME));
            }
        } finally {
            if (contactLookup != null) {
                contactLookup.close();
            }
        }

        return name;
    }

}
