package fr.ifremer.tutti.ui.swing.content.config;

/*
 * #%L
 * Tutti :: UI
 * $Id: SampleOrderIdsTableCell.java 627 2013-03-15 00:32:22Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.4.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/config/SampleOrderIdsTableCell.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import org.nuiton.util.decorator.Decorator;

import javax.swing.AbstractCellEditor;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.util.List;
import java.util.Map;

/**
 * To edit sampling oder ids in a table cell.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.1
 */
public class SampleOrderIdsTableCell {

    protected final Decorator<Caracteristic> decorator;

    protected final Map<Integer, SampleCategoryEnum> mapping;

    protected final Map<Integer, Caracteristic> caracteristicMap;

    private final TuttiUIContext context;

    public SampleOrderIdsTableCell(TuttiUIContext context) {

        this.context = context;
        this.mapping = SampleCategoryEnum.toIdMapping();
        this.caracteristicMap = Maps.newTreeMap();

        PersistenceService persistenceService = context.getPersistenceService();
        this.caracteristicMap.put(SampleCategoryEnum.size.getFieldValue(), persistenceService.getSizeCategoryCaracteristic());
        this.caracteristicMap.put(SampleCategoryEnum.sex.getFieldValue(), persistenceService.getSexCaracteristic());
        this.caracteristicMap.put(SampleCategoryEnum.maturity.getFieldValue(), persistenceService.getMaturityCaracteristic());
        this.caracteristicMap.put(SampleCategoryEnum.age.getFieldValue(), persistenceService.getAgeCaracteristic());

        this.decorator = context.getDecoratorService().getDecoratorByType(
                Caracteristic.class,
                DecoratorService.CARACTERISTIC_PARAMETER_ONLY);
    }

    public TableCellEditor getNewTableCellEditor() {
        return new SampleOrderCellEditor(context);
    }

    public TableCellRenderer getNewTableCellRenderer() {
        return new SamplingOrderIdsTableCellRenderer();
    }

    protected class SamplingOrderIdsTableCellRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {

            Integer[] ids = (Integer[]) value;
            List<String> texts = Lists.newArrayListWithCapacity(ids.length);
            for (Integer id : ids) {
                Caracteristic caracteristic = caracteristicMap.get(id);
                texts.add(decorator.toString(caracteristic));
            }
            String text = Joiner.on(",").join(texts);
            return super.getTableCellRendererComponent(table, text, isSelected, hasFocus, row, column);
        }
    }

    /**
     * @author tchemit <chemit@codelutin.com>
     * @since 1.1
     */
    public class SampleOrderCellEditor extends AbstractCellEditor implements TableCellEditor {

        private static final long serialVersionUID = 1L;

        protected Integer rowIndex;

        protected Integer columnIndex;

        protected final SampleOrderButton editorButton;

        protected SampleOrderCellEditor(TuttiUIContext context) {

            this.editorButton = new SampleOrderButton(context, null);
            this.editorButton.setBorder(new LineBorder(Color.BLACK));
            addCellEditorListener(new CellEditorListener() {
                @Override
                public void editingStopped(ChangeEvent e) {
                    editorButton.setSelected(false);
                }

                @Override
                public void editingCanceled(ChangeEvent e) {
                    editorButton.setSelected(false);
                }
            });
        }

        @Override
        public Component getTableCellEditorComponent(JTable table,
                                                     Object value,
                                                     boolean isSelected,
                                                     int row,
                                                     int column) {
            Integer[] ids = (Integer[]) value;

            rowIndex = row;
            columnIndex = column;

            List<Caracteristic> result = Lists.newArrayList();
            for (Integer id : ids) {
                SampleCategoryEnum sampleCategoryEnum = mapping.get(id);
                if (sampleCategoryEnum == null) {
                    throw new TuttiTechnicalException("caracteristic with id '" + id + "' is not designed form sampling.");
                }
                Caracteristic caracteristic = caracteristicMap.get(id);
                result.add(caracteristic);
            }

            editorButton.init(result);

            return editorButton;
        }

        @Override
        public Object getCellEditorValue() {

            List<Integer> l = Lists.newArrayList();
            List<Caracteristic> model = editorButton.getBean();
            for (Caracteristic caracteristic : model) {
                String id = caracteristic.getId();
                l.add(Integer.valueOf(id));
            }
            return l.toArray(new Integer[l.size()]);
        }

        @Override
        public boolean stopCellEditing() {
            boolean b = super.stopCellEditing();
            if (b) {
                editorButton.setSelected(false);
            }
            return b;
        }

        @Override
        public void cancelCellEditing() {
            editorButton.setSelected(false);
            super.cancelCellEditing();
        }
    }
}
