package fr.ifremer.tutti.ichtyometer.feed;

/*
 * #%L
 * Tutti :: Ichtyometer API
 * $Id: FeedReaderRecord.java 1574 2014-02-06 17:34:03Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4.2/tutti-ichtyometer/src/main/java/fr/ifremer/tutti/ichtyometer/feed/FeedReaderRecord.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Record_id,length(mm),weight(g),species,cruise,station,pan,Dt,sz class,gender,text,
 * <p/>
 * Created on 1/24/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.1
 */
public class FeedReaderRecord {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FeedReaderRecord.class);

    /**
     * Incoming raw record.
     */
    protected final String record;

    /**
     * Incoming crc.
     */
    protected final String crc;

    /**
     * Our computed crc.
     */
    protected final String computedCrc;

    /**
     * Extracted length (the first data of the record).
     */
    protected final Float length;

    /**
     * Computed valid flag (if crc == computedCrc and length != null).
     */
    protected final boolean valid;

    public FeedReaderRecord(String record, String crc) {
        this.record = record;
        this.crc = crc;
        this.computedCrc = computeCRC(record);
        this.length = computeLength(record);

        //FIXME Uncomment when computedCrc will be ok
        this.valid = length != null; /*&& crc.equals(computedCrc);*/
    }

    public boolean isValid() {
        return valid;
    }

    public String getRecord() {
        return record;
    }

    public String getCrc() {
        return crc;
    }

    public String getComputedCrc() {
        return computedCrc;
    }

    public Float getLength() {
        return length;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("record", record)
                .append("crc", crc)
                .append("computedCrc", computedCrc)
                .append("valid", valid)
                .append("length", length)
                .toString();
    }

    public static String computeCRC(String record) {

        // use long because int are signed and add doesn't work correctly
        long checksum = 0;

        // to keep using 4 octets (32bites)
        long mask = 0b0000000000000000000000000000000011111111111111111111111111111111L;

        for (int i = 0, recordLength = record.length(); i < recordLength; i++) {

            // keep only the first 32 bites
            checksum = checksum & mask;

            // get msb bite value
            long msb = checksum >> 31;

            log.debug(String.format("i=%4d char=%s(int:%3d) incoming checksum=%10d msb=%d bits=%33s", i, record.charAt(i), (int) record.charAt(i), checksum, msb, Long.toBinaryString(checksum)));

            // logical shift left
            checksum <<= 1;
            log.debug(String.format("After lef shift   : %33s", Long.toBinaryString(checksum)));

            // rotates msb to lsb
            checksum += msb;
            log.debug(String.format("After add smb     : %33s", Long.toBinaryString(checksum)));

            // add new caracter
            checksum += record.charAt(i);
            log.debug(String.format("After add car     : %33s", Long.toBinaryString(checksum)));
        }

        // get a hexadecimal on 32bites -> 8 hexadecimal caracters
        String hex = Long.toHexString(checksum).toUpperCase();
        log.debug("checksum= " + checksum);
        return hex;
    }

    protected Float computeLength(String record) {
        Float length;
        try {
            String[] cells = record.split("\\s*,\\s*");
            // first cell is the record number, second one is the length
            String lengthCell = cells[1];
            // remove any spaces
            lengthCell = lengthCell.replaceAll("\\s*", "");

            // remove any starting 0
            lengthCell = lengthCell.replaceAll("^0*", "");

            length = Float.valueOf(lengthCell);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Could not get length from " + record, e);
            }
            length = null;
        }
        return length;
    }
}
