/*
 * #%L
 * SGQ :: Web
 * $Id: LabelImportAction.java 333 2013-04-22 15:04:32Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.4/sgq-web/src/main/java/com/herbocailleau/sgq/web/actions/user/LabelImportAction.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.web.actions.user;

import java.io.File;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.business.model.ImportLog;
import com.herbocailleau.sgq.business.services.BatchService;
import com.herbocailleau.sgq.business.services.ProductionService;
import com.herbocailleau.sgq.entities.Zone;
import com.herbocailleau.sgq.web.SgqActionSupport;
import com.opensymphony.xwork2.Preparable;

/**
 * Action d'import de reception des lots.
 * 
 * @author echatellier
 */
public class LabelImportAction extends SgqActionSupport implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = 3074255662512738956L;

    private static final Log log = LogFactory.getLog(LabelImportAction.class);

    protected Date productionLastImportDate;

    protected Date expeditionLastImportDate;

    protected Date batchMaxEntryDate;

    protected long labelErrorCount;

    protected boolean forceImport;

    protected File file;

    protected String fileFileName;

    protected List<ImportLog> importLogs;

    protected long importedLines;

    @Override
    public void prepare() throws Exception {
        BatchService batchService = newService(BatchService.class);
        batchMaxEntryDate = batchService.getBatchMaxEntryDate();
        if (SgqUtils.daysBetween(batchMaxEntryDate, new Date()) < 5) {
            batchMaxEntryDate = null;
        }
    }

    @Override
    public String input() throws Exception {
        ProductionService productionService = newService(ProductionService.class);
        productionLastImportDate = productionService.getLastImportDate(Zone.ZE);
        expeditionLastImportDate = productionService.getLastImportDate(Zone.ZP);
        labelErrorCount = productionService.getLabelErrorCount();
        return super.input();
    }

    public Date getProductionStartDate() {
        return productionLastImportDate;
    }

    public Date getExpeditionStartDate() {
        return expeditionLastImportDate;
    }

    public Date getBatchMaxEntryDate() {
        return batchMaxEntryDate;
    }

    public long getLabelErrorCount() {
        return labelErrorCount;
    }

    public void setForceImport(boolean forceImport) {
        this.forceImport = forceImport;
    }

    public void setFile(File file) {
        this.file = file;
    }

    public void setFileFileName(String fileFileName) {
        this.fileFileName = fileFileName;
    }

    @Override
    public String execute() throws Exception {
        String result = SUCCESS;
        if (file == null) {
            result = input();
        } else {

            // validation
            if (batchMaxEntryDate != null && !forceImport) {
                addActionError("L'import ne peut être réalisé s'il n'est pas forcé");
                result = input();
            } else {
                ProductionService productionService = newService(ProductionService.class);
                try {
                    importLogs = productionService.importLabelsFile(fileFileName, file);
                    
                    // count how many lines were imported without errors
                    importedLines = 0;
                    for (ImportLog importLog : importLogs) {
                        importedLines++;
                    }
                } catch (Exception ex) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't import labels file", ex);
                    }
                    addActionError(ex.getMessage());
                    result = input();
                }
            }
        }

        return result;
    }

    public List<ImportLog> getImportLogs() {
        return importLogs;
    }
    
    public long getImportedLines() {
        return importedLines;
    }
}
