/*
 * #%L
 * SGQ :: Business
 * $Id: BatchDAOImpl.java 400 2013-06-23 12:38:25Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.3/sgq-business/src/main/java/com/herbocailleau/sgq/entities/BatchDAOImpl.java $
 * %%
 * Copyright (C) 2012, 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.model.BatchModel;
import com.herbocailleau.sgq.business.model.SearchModel;

public class BatchDAOImpl<E extends Batch> extends BatchDAOAbstract<E> {

    /**
     * Count non expired and valid batch.
     * 
     * @return non expired and valid batch
     * @throws TopiaException 
     */
    public long countOperating() throws TopiaException {
        String query = "select count(*)" +
                " from " + Batch.class.getName() +
                " where " + Batch.PROPERTY_INVALID + " = false" +
                " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";

        Number result = (Number)context.findUnique(query);
        return result.longValue();
    }

    /**
     * Get max batch number < max batch range.
     * 
     * @param maxBatchRange max batch range
     * @return max batch number
     * @throws TopiaException 
     */
    public int getLastBatchNumber(int maxBatchRange) throws TopiaException {
        String query = "select max(" + Batch.PROPERTY_NUMBER + ")" +
                " from " + Batch.class.getName() +
                " where " + Batch.PROPERTY_INVALID + " = false" +
                " and " + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and " + Batch.PROPERTY_NUMBER + " <= :max";

        Number result = (Number)context.findUnique(query, "max", maxBatchRange);
        return result.intValue();
    }

    /**
     * Retourne le prochain id non utilisé parmis la liste des numeros
     * de lots entre start et end.
     * 
     * @param start range start
     * @param end range end
     * @return next non used batch number
     */
    public int getNextRangeIdBetween(int start, int end) throws TopiaException {
        String query = "select min(main." + Batch.PROPERTY_NUMBER + ") + 1" +
            " from " + Batch.class.getName() + " as main" +
            " where :start < main." + Batch.PROPERTY_NUMBER +
            " and main." + Batch.PROPERTY_NUMBER + " < :end" +
            " and main." + Batch.PROPERTY_NUMBER + " + 1 not in " +
            " (select sub.number from " + Batch.class.getName() + " as sub)";

        Number id = (Number)context.findUnique(query, "start", start, "end", end);

        int result = 0;
        if (id == null) {
            // nothing found in range start..end, just return start
            result = start;
        } else {
            result = id.intValue();
        }

        return result;
    }

    /**
     * Return all batches between range of numbers (non elapsed or in error).
     * 
     * @param start start range
     * @param end end range
     * @return all batches between range of numbers
     * @throws TopiaException
     */
    public List<Batch> findAllBetweenNumbers(int start, int end) throws TopiaException {
        String query = " from " + Batch.class.getName() +
            " where :start <= " + Batch.PROPERTY_NUMBER +
            " and " + Batch.PROPERTY_NUMBER + " <= :end" +
            " and " + Batch.PROPERTY_INVALID + " = false" +
            " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";

        List<Batch> result = context.findAll(query, "start", start, "end", end);

        return result;
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot...
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public List<BatchModel> findAllModel(SearchModel search) throws TopiaException {
        return findAllModel(search, 0, -1);
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot...
     * 
     * @param search search model
     * @param offset offset
     * @param limit limit (use -1 for no limit)
     * @return
     * @throws TopiaException
     */
    public List<BatchModel> findAllModel(SearchModel search, int offset, int limit) throws TopiaException {
        String queryPrefix = getFindBatchModelQueryPrefix();

        String querySuffix = " order by B." + Batch.PROPERTY_NUMBER + " ASC";

        List<Object[]> beanAndQts = (List<Object[]>)performQueryWithFilter(search, queryPrefix, querySuffix, offset, limit);
        List<BatchModel> results = new ArrayList<BatchModel>(beanAndQts.size());
        for (Object[] beanAndQt : beanAndQts) {
            Batch batch = (Batch)beanAndQt[0];
            Double pres = (Double)beanAndQt[1];
            Double sale = (Double)beanAndQt[2];
            Date dmes = (Date)beanAndQt[3];
            BatchModel model = new BatchModel(batch, pres == null ? 0 : pres, sale == null ? 0 : sale, dmes);
            results.add(model);
        }
        return results;
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot.
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public long findAllCount(SearchModel search) throws TopiaException {
        String queryPrefix = "select count(B)";

        Number result = (Number)performQueryWithFilter(search, queryPrefix, "", -1, -1);
        return result.longValue();
    }

    /**
     * Retourne le prefix de requete sql qui permet d'obtenir les meta
     * infos en plus du batch pour remplir le model
     * 
     * @return query prefix
     */
    protected String getFindBatchModelQueryPrefix() {
        String query = "select B," +
                " (select sum(P.quantity) from " + Presentation.class.getName() + " P where P.batch = B)," +
                " (select COALESCE(sum(E.quantity),0) from " + Expedition.class.getName() + " E where E.presentation.batch = B)" +
                " - (select COALESCE(sum(E.quantity),0) from " + Expedition.class.getName() + " E where E.correction.batch = B)," +
                " (select min(E.date) from " + Expedition.class.getName() + " E where E.presentation.batch = B)";
        return query;
    }

    /**
     * Construit une requete ou les champs sont comparés sans tenir
     * compte de la case et de la présence d'accents.
     * 
     * @return query
     */
    protected static String getFieldLikeInsensitive(String field1, String field2) {
        String query = "REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(lower(" +
            field1 + "),'é|è|ê','e'),'à|â','a'),'î|ï','i'),'ô','o'),'ù','u') like lower(" + field2 + ")";
        return query;
    }

    /**
     * Construit la requete réutilisable avec des bout de requete (avant/après)
     * suivant ce que doit effectivement retourner la requete.
     * 
     * @param search le filtre de recherche
     * @param queryPrefix prefix
     * @param querySuffix suffix
     * @param offset offset (use -1 for no limit)
     * @param limit offset (use -1 for no limit)
     * @return le resultat (le type dépend du prefix)
     * @throws TopiaException 
     */
    protected Object performQueryWithFilter(SearchModel search, String queryPrefix,
            String querySuffix, int offset, int limit) throws TopiaException {

        String query = queryPrefix;
        query += " from " + Batch.class.getName() + " B";
        query += " where " + Batch.PROPERTY_INVALID + " != true";

        List<Object> params = new ArrayList<Object>();

        // query field
        if (StringUtils.isNotBlank(search.getQuery())) {
            query += " and (B.number = :batch";
            int batch = -1;
            try {
                batch = Integer.parseInt(search.getQuery());
            } catch (NumberFormatException ex) {
                
            }
            params.add("batch");
            params.add(batch);
            // product name
            query += " or " + getFieldLikeInsensitive("B.product.name", ":query");
            // product code
            query += " or " + getFieldLikeInsensitive("B.product.code", ":query") + ")";
            params.add("query");
            params.add("%" + StringUtils.stripAccents(search.getQuery()) + "%");
        }

        // client field
        if (StringUtils.isNotBlank(search.getClient())) {
            query += " and (" + getFieldLikeInsensitive("B.dedicatedClient.code", ":clientcode");
            params.add("clientcode");
            params.add("%" + StringUtils.stripAccents(search.getClient()) + "%");
            query += " or " + getFieldLikeInsensitive("B.dedicatedClient.name", ":client") + ")";
            params.add("client");
            params.add("%" + StringUtils.stripAccents(search.getClient()) + "%");
        }

        // supplier
        if (StringUtils.isNotBlank(search.getSupplier())) {
            query += " and (" + getFieldLikeInsensitive("B.supplier.code", ":suppliercode");
            params.add("suppliercode");
            params.add("%" + StringUtils.stripAccents(search.getSupplier()) + "%");
            query += " or " + getFieldLikeInsensitive("B.supplier.name", ":supplier") + ")";
            params.add("supplier");
            params.add("%" + StringUtils.stripAccents(search.getSupplier()) + "%");
        }

        // origin
        if (search.getOrigin() != null) {
            query += " and B.origin = :origin";
            params.add("origin");
            params.add(search.getOrigin());
        }

        // product status
        if (CollectionUtils.isNotEmpty(search.getProductStatus())) {
            int index = 0;
            query += " and (1=1";
            for (ProductStatus status : search.getProductStatus()) {
                if (status == null) { // cas de l'option "vide"
                    query += " AND B.product.productStatus IS EMPTY";
                } else {
                    query += " AND :status" + index + " member of B.product.productStatus";
                    params.add("status" + index);
                    params.add(status);
                    index++;
                }
            }

            query += ")";
        }

        // product categories
        if (CollectionUtils.isNotEmpty(search.getProductCategories())) {
            int index = 0;
            query += " and (0=1";
            for (String cat : search.getProductCategories()) {
                query += " OR :category" + index + " = B.product.category";
                params.add("category" + index);
                params.add(cat);
                index++;
            }
            query += ")";
        }

        // analyze categories
        if (CollectionUtils.isNotEmpty(search.getAnalyzeTypes())) {
            int index = 0;
            query += " and (";
            Iterator<AnalyzeType> itAnalyzeType = search.getAnalyzeTypes().iterator();
            while (itAnalyzeType.hasNext()) {
                AnalyzeType analyzeType = itAnalyzeType.next();
                query += " (B in (select A.batch from " + BatchAnalyze.class.getName() +
                        " A WHERE A.analyzeType = :analyzeType" + index;
                if (search.getSynthesisMention() != null) {
                    query += " AND A.synthesisMention = :synthesisMention" + index;
                    params.add("synthesisMention" + index);
                    params.add(search.getSynthesisMention());
                }
                query += "))";
                params.add("analyzeType" + index);
                params.add(analyzeType);
                index++;
                if (itAnalyzeType.hasNext()) {
                    query += search.isAnalyzeTypeOrOperator() ? " OR" : " AND";
                }
            }
            query += ")";

        } else if (search.getSynthesisMention() != null) {
            // resultat d'analyze (quand utilisé sans type d'analyse)
            query += " AND B in (select A.batch from " + BatchAnalyze.class.getName() +
                        " A WHERE A.synthesisMention = :synthesisMention)";
            params.add("synthesisMention");
            params.add(search.getSynthesisMention());
        }

        // dates
        if (search.getDateType() != null && (search.getBeginDate() != null || search.getEndDate() != null)) {
            String dateField = null;
            switch (search.getDateType()) {
            case ENTRY_DATE:
                dateField = Batch.PROPERTY_ENTRY_DATE;
                break;
            case DLUO:
                dateField = Batch.PROPERTY_DLUO;
                break;
            case DMES:
                dateField = "(select min(S.date) from " + Expedition.class.getName() + " S where S.presentation.batch = B)";
                break;
            case DMESD:
                dateField = Batch.PROPERTY_DMESD;
                break;
            case DPMES:
                dateField = Batch.PROPERTY_DPMES;
                break;
            case EXPIRED_DATE:
                dateField = Batch.PROPERTY_EXPIRED_DATE;
                break;
            }
            
            if (search.getBeginDate() != null) {
                query += " AND :beginDate <= " + dateField;
                params.add("beginDate");
                params.add(search.getBeginDate());
            }
            if (search.getEndDate() != null) {
                // to fix interval, take date at midnigth
                Date date = DateUtils.addDays(search.getEndDate(), 1);
                date = DateUtils.addMilliseconds(date, -1);
                query += " AND " + dateField + " <= :endDate";
                params.add("endDate");
                params.add(date);
            }
        }

        // lots epuises
        switch (search.getExpired()) {
        case EXPIRED:
            query += " and " + Batch.PROPERTY_EXPIRED_DATE + " != null";
            break;
        case NON_EXPIRED:
            query += " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";
            break;
            default:
                // all
                break;
        }

        // add final suffix to query
        query += querySuffix;

        // perform query
        Object result = null;
        if (offset == -1) {
            result = context.findUnique(query, params.toArray());
        } else if (limit == -1) {
            result = context.findAll(query, params.toArray());
        } else {
            result = context.find(query, offset, offset + limit - 1, params.toArray());
        }
        return result;
    }

    /**
     * Retourne un batch model (batch + meta info) à partir du topia id.
     * 
     * En plus du lot, il y a des informations issue de recherche par sous
     * requettes :
     * <ul>
     * <li>la quantité totale disponible pour l'ensemble des presentations
     * <li>la quantité totale vendue (expediée)
     * <li>la date de mise en service (DMES)
     * </ul>
     * 
     * @return le batch model
     * @throws TopiaException
     */
    public BatchModel findModelByTopiaId(String batchId) throws TopiaException {
        String query = getFindBatchModelQueryPrefix() +
                " from " + Batch.class.getName() + " B" +
                " where B." + Batch.TOPIA_ID + " = :batchId";

        BatchModel result = null;
        Object[] beanAndQt = (Object[])context.findUnique(query, "batchId", batchId);
        if (beanAndQt != null) {
            Double pres = (Double)beanAndQt[1];
            Double sale = (Double)beanAndQt[2];
            Date dmes = (Date)beanAndQt[3];
            result = new BatchModel((Batch)beanAndQt[0], pres == null ? 0 : pres, sale == null ? 0 : sale, dmes);
        }
        return result;
    }

    /**
     * Retourne les lots en attente d'analyse pour lequel il y a déjà une sortie externe.
     * @throws TopiaException 
     */
    public List<Batch> getBatchWithOutputAndPendingAnalysis() throws TopiaException {
        String query = "from " + Batch.class.getName() + " B" +
                " where B in (" +
                "  SELECT E.presentation.batch FROM " + Expedition.class.getName() + " E" +
                " ) AND B in (" +
                "  SELECT A.batch FROM " + BatchAnalyze.class.getName() + " A" +
                "   WHERE A." + BatchAnalyze.PROPERTY_RECEIPT_DATE + " = null)" +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }

    /**
     * Retourne les lots en attente d'analyse pour lequel il y a déjà une sortie d'étiquette interne.
     * @throws TopiaException 
     */
    public List<Batch> getBatchWithInputAndPendingAnalysis() throws TopiaException {
        String query = "from " + Batch.class.getName() + " B" +
                " where B in (" +
                "  SELECT P.presentation.batch FROM " + Production.class.getName() + " P" +
                " ) AND B in (" +
                "  SELECT A.batch FROM " + BatchAnalyze.class.getName() + " A" +
                "   WHERE A." + BatchAnalyze.PROPERTY_RECEIPT_DATE + " = null)" +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }

    /**
     * Retourne les lots qui ont des expeditions supérieures à la quantité entrée.
     * Ces lots sont donc théoriquements épuisé.
     * 
     * @return maybe expired batch
     * @throws TopiaException 
     */
    public List<Batch> getBatchWithMoreSellThanStock() throws TopiaException {
        String query = "SELECT B from " + Batch.class.getName() + " B" +
                " where " + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " AND (select COALESCE(sum(E.quantity),0) from " + Expedition.class.getName() + " E where E.presentation.batch = B)" +
                " - (select COALESCE(sum(E.quantity),0) from " + Expedition.class.getName() + " E where E.correction.batch = B) > B." + Batch.PROPERTY_QUANTITY +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }
    
    /**
     * Retourne les lots qui concernent les clients dédiés et qui sont
     * probablement expiré.
     * 
     * @return maybe expired batch
     * @throws TopiaException 
     */
    public List<Batch> getDedicatedClientExpiredBatch() throws TopiaException {
        String query = "from " + Batch.class.getName() + " B" +
                " where " + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " AND " + Batch.PROPERTY_DEDICATED_CLIENT + " != null" + 
                " AND (SELECT sum(P." + Presentation.PROPERTY_QUANTITY + ") FROM " + Presentation.class.getName() + " P" +
                "  WHERE P.batch = B) <= 0" +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }
    
    /**
     * Retourne les lots qui n'ont plus de stock
     * 
     * @return batch without stock (== 0)
     * @throws TopiaException 
     */
    public List<Batch> getBatchWithNoStock() throws TopiaException {
        String query = "from " + Batch.class.getName() + " B" +
                " where " + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " AND (SELECT sum(P." + Presentation.PROPERTY_QUANTITY + ") FROM " + Presentation.class.getName() + " P" +
                "  WHERE P.batch = B) = 0" +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }

    /**
     * Retourne les lots sur lequels porte une analyze en attente de reception
     * et dont le prestataire est un fournisseur.
     * 
     * @return les lots concernés
     * @throws TopiaException
     */
    public List<Batch> getBatchWithBulletinToAsk() throws TopiaException {
        String query = "from " + Batch.class.getName() + " B" +
                " where " + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " AND B IN (SELECT A.batch FROM " + BatchAnalyze.class.getName() + " A" +
                "  WHERE A.sentDate != null" +
                "  AND A.receiptDate = null" +
                "  AND A.supplier.laboratory = false)" +
                " ORDER BY B." + Batch.PROPERTY_NUMBER;

        List<Batch> result = context.findAll(query);
        return result;
    }

    /**
     * Retoune la date d'entrée produit la plus récente.
     * 
     * @return max entry date
     * @throws TopiaException
     */
    public Date getMaxEntryDate() throws TopiaException {
        String query = "select max(" + Batch.PROPERTY_ENTRY_DATE + ")" +
                " from " + Batch.class.getName() + " B";

        Date result = (Date)context.findUnique(query);
        return result;
    }

} //BatchDAOImpl<E extends Batch>
