/*
 * #%L
 * SGQ :: Business
 * $Id: BatchService.java 118 2012-10-03 08:00:08Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.3/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/BatchService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.nio.charset.Charset;
import java.sql.Blob;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import nl.knaw.dans.common.dbflib.CorruptedTableException;
import nl.knaw.dans.common.dbflib.Record;
import nl.knaw.dans.common.dbflib.Table;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Hibernate;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.csv.AbstractImportErrorInfo;
import org.nuiton.util.csv.Export;
import org.nuiton.util.csv.Import2;
import org.nuiton.util.csv.ImportConf;
import org.nuiton.util.csv.ImportRow;
import org.w3c.dom.Document;
import org.xhtmlrenderer.pdf.ITextRenderer;

import au.com.bytecode.opencsv.CSVWriter;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.business.model.AnalyzeFilter;
import com.herbocailleau.sgq.business.model.BatchModel;
import com.herbocailleau.sgq.business.model.ImportLog;
import com.herbocailleau.sgq.business.model.PresentationModel;
import com.herbocailleau.sgq.business.model.SearchColumn;
import com.herbocailleau.sgq.business.model.SearchModel;
import com.herbocailleau.sgq.business.services.csv.BatchImportModel;
import com.herbocailleau.sgq.business.services.csv.InventoryBean;
import com.herbocailleau.sgq.business.services.csv.InventoryExportModel;
import com.herbocailleau.sgq.entities.Analyze;
import com.herbocailleau.sgq.entities.AnalyzeDAO;
import com.herbocailleau.sgq.entities.AnalyzeFile;
import com.herbocailleau.sgq.entities.AnalyzeFileDAO;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.BatchImpl;
import com.herbocailleau.sgq.entities.Client;
import com.herbocailleau.sgq.entities.ClientDAO;
import com.herbocailleau.sgq.entities.Country;
import com.herbocailleau.sgq.entities.Expedition;
import com.herbocailleau.sgq.entities.ExpeditionDAO;
import com.herbocailleau.sgq.entities.Place;
import com.herbocailleau.sgq.entities.Presentation;
import com.herbocailleau.sgq.entities.PresentationCode;
import com.herbocailleau.sgq.entities.PresentationDAO;
import com.herbocailleau.sgq.entities.Product;
import com.herbocailleau.sgq.entities.ProductDAO;
import com.herbocailleau.sgq.entities.ProductPlace;
import com.herbocailleau.sgq.entities.ProductPlaceDAO;
import com.herbocailleau.sgq.entities.Production;
import com.herbocailleau.sgq.entities.ProductionDAO;
import com.herbocailleau.sgq.entities.Supplier;
import com.herbocailleau.sgq.entities.SupplierDAO;
import com.herbocailleau.sgq.entities.Zone;
import com.itextpdf.text.DocumentException;

import freemarker.cache.ClassTemplateLoader;
import freemarker.ext.beans.BeansWrapper;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

/**
 * Service de gestion des lots et des analyses.
 * 
 * @author echatellier
 */
public class BatchService extends SgqService {

    private static final Log log = LogFactory.getLog(BatchService.class);

    /**
     * Get total batch count.
     * 
     * @return batch count
     */
    public long getBatchCount() {
        long count = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            count = batchDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return count;
    }

    /**
     * Return {@code count} result and total result count in a {@link Pair}
     * elements.
     * 
     * @param search search model
     * @param offset offset
     * @param count count
     */
    public Pair<List<BatchModel>, Long> searchBatch(SearchModel search, int offset, int count) {
        Pair<List<BatchModel>, Long> result = null;
        
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            List<BatchModel> elements = batchDAO.findAllModel(search, offset, count);
            Long totalCount = batchDAO.findAllCount(search);
            result = Pair.of(elements, totalCount);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't search batches", ex);
        }
        
        return result;
    }

    /**
     * Export la recherche en csv. Basé sur l'enumeration de génération
     * dynamique utilisé également dans le rendu web.
     */
    public InputStream searchBatchASCsv(SearchModel search) {
        InputStream result = null;
        try {
            
            // perform search (all elements)
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            List<BatchModel> elements = batchDAO.findAllModel(search, 0, -1);

            File file = File.createTempFile("search", ".csv");
            file.deleteOnExit();

            CSVWriter csvWriter = new CSVWriter(new FileWriter(file));

            // write headers
            List<String> headers = new ArrayList<String>();
            for (SearchColumn searchColumn : search.getSearchColumns()) {
                headers.add(searchColumn.getDescription());
            }
            csvWriter.writeNext(headers.toArray(new String[headers.size()]));

            // write elements
            for (BatchModel batchModel : elements) {
                List<String> data = new ArrayList<String>(headers.size());
                for (SearchColumn searchColumn : search.getSearchColumns()) {
                    String content = searchColumn.getValueFor(batchModel);
                    if (content == null) {
                        data.add("");
                    } else {
                        data.add(content);
                    }
                }
                csvWriter.writeNext(data.toArray(new String[data.size()]));
            }

            csvWriter.close();
            
            result = new FileInputStream(file);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't search batches", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't output csv file", ex);
        }
        return result;
    }

    public List<Batch> findAllBetweenNumbers(int startNumber, int endNumber) {
        List<Batch> result = null;
        
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findAllBetweenNumbers(startNumber, endNumber);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't search batches", ex);
        }
        
        return result;
    }

    /**
     * Get batch instance with batch id.
     * 
     * @param batchId batch topia id
     * @return batch instance
     */
    public Batch getBatchById(String batchId) {
        Batch result = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findByTopiaId(batchId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        
        return result;
    }

    /**
     * Get batch instance with batch id.
     * 
     * @param batchId batch topia id
     * @return batch instance
     */
    public BatchModel getBatchModelById(String batchId) {
        BatchModel result = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findModelByTopiaId(batchId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        
        return result;
    }

    /**
     * Find batch by number.
     * 
     * @param batchNumer batch number
     * @return batch instance
     */
    protected Batch getBatchByNumber(int batchNumer) {
        Batch result = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.findByNumber(batchNumer);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by number", ex);
        }

        return result;
    }

    /**
     * Save batch.
     * 
     * @param batch batch to save
     */
    public void saveBatch(Batch batch) {
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            batch = batchDAO.update(batch);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save batch", ex);
        }
    }

    /**
     * Find all presentation for batch.
     * 
     * Pour la détermination des emplacements:
     * <ul>
     * <li>plante qui arrive par ordre de présence des zones: ZE, ZP, ZC
     * <li>une plante est présente en ZC s'il y a un mouvement pour le client 99997
     * <li>une plante est présente en ZP s'il y a un mouvement pour un client externe depuis le fichier (FIC_HIST.TXT)
     * </ul>
     * 
     * @param batch batch
     * @return batch's presentation
     */
    public List<PresentationModel> findAllPresentationsByBatch(Batch batch) {
        List<PresentationModel> result = null;
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            ProductPlaceDAO productPlaceDAO = daoHelper.getProductPlaceDAO();
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();

            List<Presentation> presentations = presentationDAO.findAllByBatchOrderByPresentation(batch);

            // à ces presentation, il faut recherche dans le referentiel
            // produit les emplacements possibles suivants les informations
            // presente dans la base (ventes, conditionnement, expedition)
            result = new ArrayList<PresentationModel>(presentations.size());
            for (Presentation presentation : presentations) {

                PresentationModel model = new PresentationModel(presentation);

                // par defaut la plante est en ZE
                List<ProductPlace> productPlacesZE = productPlaceDAO.findByPresentationAndZone(presentation, Zone.ZE);
                for (ProductPlace productPlaceZE : productPlacesZE) {
                    if (productPlaceZE.getPresentationCode() == null ||
                            productPlaceZE.getPresentationCode() == presentation.getPresentationCode()) {
                        model.addPlace(productPlaceZE.getPlace());
                    }
                }

                // ensuite ZP (si rien en ZE ou si mouvement pour client externe)
                Expedition expedition = expeditionDAO.findSingleForPresentationAndZone(presentation, Zone.ZP);
                if (expedition != null || model.getPlaces().isEmpty()) {
                    List<ProductPlace> productPlacesZP = productPlaceDAO.findByPresentationAndZone(presentation, Zone.ZP);
                    for (ProductPlace productPlaceZP : productPlacesZP) {
                        if (productPlaceZP.getPresentationCode() == null ||
                                productPlaceZP.getPresentationCode() == presentation.getPresentationCode()) {
                            model.addPlace(productPlaceZP.getPlace());
                        }
                    }
                }

                // enfin ZC (si il y a eu un mouvement pour le client 99997)
                Production production = productionDAO.findSingleForPresentationAndZone(presentation, Zone.ZC);
                if (production != null || model.getPlaces().isEmpty()) {
                    List<ProductPlace> productPlacesZC = productPlaceDAO.findByPresentationAndZone(presentation, Zone.ZC);
                    for (ProductPlace productPlaceZC : productPlacesZC) {
                        if (productPlaceZC.getPresentationCode() == null ||
                                productPlaceZC.getPresentationCode() == presentation.getPresentationCode()) {
                            model.addPlace(productPlaceZC.getPlace());
                        }
                    }
                }
                
                result.add(model);
            }
            
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }

    /**
     * Get presentation by id.
     * 
     * @param presentationId presentation topia id
     * @return presentation
     */
    public Presentation getPresentationById(String presentationId) {
        Presentation result = null;
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            result = presentationDAO.findByTopiaId(presentationId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }

    /**
     * Update presentation.
     * 
     * @param presentation presentation
     */
    public void updatePresentation(Presentation presentation) {
        try {
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            presentationDAO.update(presentation);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't update", ex);
        }
    }

    public List<Analyze> findAllAnalyzesByBatch(Batch batch) {
        List<Analyze> analyzes = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyzes = analyzeDAO.findAllByBatch(batch);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by batch", ex);
        }
        return analyzes;
    }

    public List<AnalyzeFile> findAllAnalyzeFilesByBatch(Batch batch) {
        List<AnalyzeFile> analyzeFiles = null;
        try {
            AnalyzeFileDAO analyzeFileDAO = daoHelper.getAnalyzeFileDAO();
            analyzeFiles = analyzeFileDAO.findAllByBatch(batch);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by batch", ex);
        }
        return analyzeFiles;
    }

    public List<AnalyzeFile> findAllAnalyzeFilesByAnalyze(Analyze analyze) {
        List<AnalyzeFile> analyzeFiles = null;
        try {
            AnalyzeFileDAO analyzeFileDAO = daoHelper.getAnalyzeFileDAO();
            analyzeFiles = analyzeFileDAO.findAllByAnalyze(analyze);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by analyze", ex);
        }
        return analyzeFiles;
    }

    public int getNextBioBatchNumber() {
        int result = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.getNextRangeIdBetween(config.getBatchRangeBioStart(),
                    config.getBatchRangeBioEnd());
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }
    
    public int getNextNonBioBatchNumber() {
        int result = 0;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            result = batchDAO.getNextRangeIdBetween(config.getBatchRangeNonBioStart(),
                    config.getBatchRangeNonBioEnd());
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }

    public Analyze getAnalyzeById(String analyzeId) {
        Analyze result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findByTopiaId(analyzeId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        
        return result;
    }

    public void saveAnalyze(Analyze analyze) {
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyzeDAO.create(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't create analyze", ex);
        }
    }

    /**
     * Retourne un flux de lecture des données d'un fichier d'analyse.
     * 
     * @param analyzeFile analyze file
     * @return in inputstream sur le fichier
     */
    public InputStream getAnalyzeDataStream(AnalyzeFile analyzeFile) {
        InputStream result = null;
        try {
            result = analyzeFile.getData().getBinaryStream();
        } catch (SQLException ex) {
            throw new SgqBusinessException("Can't get analyze input stream", ex);
        }
        return result;
    }

    public List<Analyze> getAnalysisToSend(AnalyzeFilter analyzeFilter) {
        List<Analyze> result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findAllAnalyzeToSend(analyzeFilter);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analysis", ex);
        }
        return result;
    }

    public List<Analyze> getAnalysisToReceive(AnalyzeFilter analyzeFilter) {
        List<Analyze> result = null;
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            result = analyzeDAO.findAllAnalyzeToReceive(analyzeFilter);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analysis", ex);
        }
        return result;
    }

    /**
     * Import du fichier dbf des numéros de lots.
     * 
     * DATE_R : Date d’entrée (1)
     * FOUR_R : Nom fournisseur (1) (2)
     * COMM_R : N° de commande interne (1)
     * COLIS_R : Nombre de colis à la réception (1)
     * POIDS_R : Poids réceptionné (1)
     * PROD_R : Code présentation + Code produit (1)
     * PAYS_R : Pays d’origine de la plante (1) (2)
     * LOT_R : N° de lot (1)
     * DLUO_R : Date Limite d’Utilisation Optimale du produit (1)
     * CLI_R : Client dédié au lot entré (2)
     * ECH_R : N° échantillon utilisé pour valider l’achat du lot
     * REM_R : Observation faite par le responsable des entrées
     * 
     * CODE_FOUR_R : Code fournisseur
     * CODE_PAYS_R : Code pays
     * CODE_CLI_R : Code Client dédié
     * IGNORER : ligne à ne pas intégrer
     * 
     * @param batchFile dbf file
     * @return import log for each line of input file
     */
    public List<ImportLog<Batch>> importBatchFile(File batchFile) {
        
        if (log.isInfoEnabled()) {
            log.info("Importing batch from file " + batchFile);
        }

        Table table = new Table(batchFile);
        List<ImportLog<Batch>> importLogs = new ArrayList<ImportLog<Batch>>();
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ProductDAO productDAO = daoHelper.getProductDAO();
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();

            // get import bounds
            int nonBioStart = getNextNonBioBatchNumber();
            int nonBioEnd = config.getBatchRangeNonBioEnd();
            int bioStart = getNextBioBatchNumber();
            int bioEnd = config.getBatchRangeBioEnd();

            // get date format
            DateFormat df = new SimpleDateFormat("dd/MM/yy");

            // Open dbf file
            table = new Table(batchFile, "UTF-8");
            table.open();

            int line = 0;
            Iterator<Record> recordIterator = table.recordIterator();
            while (recordIterator.hasNext()) {
                line++;
                Record record = recordIterator.next();

                int num = record.getNumberValue("LOT_R").intValue();

                // le fichier contient énormement d'élement que l'on a déja
                // traités, on ne prend pas du tout en compte les lots
                // hors bornes et ne ne génère pas de log pour ca
                if (nonBioStart <= num && num <= nonBioEnd ||
                        bioStart <= num && num <= bioEnd) {

                    // get all data to compute md5
                    Date dateEntree = record.getDateValue("DATE_R");
                    Number commande = record.getNumberValue("COMM_R");
                    Number colis = record.getNumberValue("COLIS_R");
                    Number poids = record.getNumberValue("POIDS_R");
                    String codeProduit = record.getStringValue("PROD_R").trim();
                    String origin = record.getStringValue("PAYS_R").trim();
                    String dluo = record.getStringValue("DLUO_R").trim();
                    String echantillon = record.getStringValue("ECH_R").trim();
                    String remarque = record.getStringValue("REM_R").trim();

                    String hash = SgqUtils.getSHA1Hash(df.format(dateEntree),
                            String.valueOf(commande), String.valueOf(colis), String.valueOf(poids),
                            codeProduit, origin, dluo, echantillon, remarque);

                    // get existing batch and skip it if hash in unchanged
                    Batch batch = getBatchByNumber(num);
                    if (batch == null) {
                        batch = new BatchImpl();
                    } else {
                        if (hash.equals(batch.getImportHash())) {
                            if (log.isDebugEnabled()) {
                                log.debug("Import hashcode unchanged for batch " + num);
                            }
                            // les lots inchangés ne donnent pas lieu à une
                            // lignes de log
                            continue;
                        }
                    }

                    ImportLog<Batch> importLog = new ImportLog<Batch>();
                    importLog.setLine(line);
                    importLog.setCode(String.valueOf(num));
                    importLogs.add(importLog);

                    // commande
                    if (commande == null || commande.intValue() <= 0) {
                        importLog.setError(true);
                        importLog.setMessage(_("Numero de commande interne invalide : %d", commande));
                        continue;
                    }
                    
                    // test presentation code
                    PresentationCode codePres = null;
                    if (codeProduit.length() == 6) {
                        codePres = PresentationCode.getPresentationCodeFor(codeProduit.charAt(0));
                        if (codePres == null) {
                            importLog.setError(true);
                            importLog.setMessage(_("Code présentation inconnu : %s", codeProduit));
                            continue;
                        }

                        // interdiction des code de présentation destiné à la vente
                        if (codePres == PresentationCode.A || codePres == PresentationCode.B
                                || codePres == PresentationCode.D) {
                            importLog.setError(true);
                            importLog.setMessage(_("Code présentation destiné à la vente seulement : %s", codeProduit));
                            continue;
                        }

                        // suppression du premier caractère (présentation)
                        codeProduit = codeProduit.substring(1);

                        // Les lignes ayant un code ayant une structure du type F0XXXX devront être importées sous la forme FXXXX
                        // Les lignes ayant un code ayant une structure du type T0XXXX devront être importées sous la forme TXXXX
                        if ((codePres == PresentationCode.F || codePres == PresentationCode.T) && codeProduit.charAt(0) == '0') {
                            // suppression du premier 0
                            codeProduit = codeProduit.substring(1);
                        }
                    } else {
                        importLog.setError(true);
                        importLog.setMessage(_("Code produit invalide (6 charactères) : %s", codeProduit));
                        continue;
                    }

                    // colis
                    if (colis == null || colis.intValue() <= 0) {
                        importLog.setError(true);
                        importLog.setMessage(_("Nombre de colis invalide : %d", colis));
                        continue;
                    }

                    // colis
                    if (poids == null || poids.doubleValue() <= 0) {
                        importLog.setError(true);
                        importLog.setMessage(_("Poids invalide : %e", poids));
                        continue;
                    }

                    // dluo
                    if (!dluo.matches("\\d\\d(/\\d\\d){2}")) {
                        importLog.setError(true);
                        importLog.setMessage(_("DLUO, format de date invalide : %s", dluo));
                        continue;
                    }

                    // origin
                    // FIXME echatellier 20120921 doit être un code valide
                    if (origin.isEmpty()) {
                        importLog.setError(true);
                        importLog.setMessage(_("Origine invalide : %s", origin));
                        continue;
                    }

                    // supplier
                    // FIXME echatellier 20120921 add supplier exists check

                    // client
                    // FIXME echatellier 20120921 add dedicated client exists check but can be empty

                    // product
                    Product product = productDAO.findByCode(codeProduit);
                    if (product == null) {
                        importLog.setError(true);
                        importLog.setMessage(_("Produit inconnu : %s", codeProduit));
                        continue;
                    }

                    batch.setNumber(num);
                    batch.setOrderNumber(commande.intValue());
                    batch.setProduct(product);
                    batch.setDluo(df.parse(dluo));
                    batch.setEntryDate(dateEntree);
                    batch.setQuantity(poids.doubleValue());
                    batch.setPackageCount(colis.intValue());
                    batch.setOrigin(Country.FR);
                    batch.setSampleCode(echantillon);
                    batch.setComment(remarque);
                    batch.setImportHash(hash);

                    if (batch.getTopiaId() == null) { 
                        batch = batchDAO.create(batch);

                        // la presentation ne peut-être créé que lors de l'import
                        // original (sinon, avec les ventes, etc... ca devient
                        // incohérent
                        Presentation presentation = presentationDAO.create();
                        presentation.setQuantity(poids.doubleValue());
                        presentation.setBatch(batch);
                        presentation.setPresentationCode(codePres);
                        
                        importLog.setMessage(_("Lot importé"));
                    } else {
                        batch = batchDAO.update(batch);
                        
                        importLog.setMessage(_("Lot mis à jour"));
                    }

                    importLog.setBean(batch);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Skipping product " + num + " (out of range)");
                    }
                }
            }

            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't commit modification in database", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't read dfb file", ex);
        } catch (CorruptedTableException ex) {
            throw new SgqBusinessException(_("Format de fichier invalide ou corrompu"), ex);
        } catch (ParseException ex) {
            // can't happen
            throw new SgqBusinessException("Can't parse data", ex);
        } finally {
            if (table != null) {
                try {
                    table.close();
                } catch (IOException ex) {
                    throw new SgqBusinessException("Can't close dbf file", ex);
                }
            }
        }

        return importLogs;
    }

    public void prepareAnalyze(Batch batch, AnalyzeType analyzeType) {

        if (batch == null) {
            throw new NullPointerException("batch can't be null");
        }

        if (analyzeType == null) {
            throw new NullPointerException("analyzeType can't be null");
        }

        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            Analyze analyze = analyzeDAO.create();
            analyze.setAnalyzeType(analyzeType);
            analyze.setBatch(batch);
            analyzeDAO.update(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't prepare analyze", ex);
            }
        }
    }

    public void sendAnalyzeToSupplier(Analyze analyze, Date sentDate, Supplier supplier) {
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyze.setSupplier(supplier);
            analyze.setSentDate(sentDate);
            analyzeDAO.update(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't send analyze to supplier", ex);
            }
        }
    }

    public void receiveAnalyzeFromSupplier(Analyze analyze, Supplier supplier) {
        try {
            AnalyzeDAO analyzeDAO = daoHelper.getAnalyzeDAO();
            analyze.setSupplier(supplier);
            analyzeDAO.update(analyze);
            daoHelper.commit();
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't send analyze to supplier", ex);
            }
        }
    }

    /**
     * Ajoute un fichier d'analyse à un lot.
     * 
     * @param batch batch (can't be null if {@code analyze} is not)
     * @param analyze analyze (can't be null if {@code batch} is not)
     * @param fileName file name
     * @param file file
     * 
     * @see http://i-proving.com/2006/08/23/Blobs-and-Hibernate/ for blob
     */
    public void addAnalyzeFile(Batch batch, Analyze analyze, String fileName, File file) {

        if (batch == null) {
            if (analyze == null) {
                throw new NullPointerException("batch and analyze can't both be null");
            }
        }

        InputStream is = null;
        try {
            is = new FileInputStream(file);

            // create blob object
            Blob blob = Hibernate.createBlob(is);
            is.close();

            AnalyzeFileDAO analyzeFileDAO = daoHelper.getAnalyzeFileDAO();
            AnalyzeFile analyzeFile  = analyzeFileDAO.create();
            analyzeFile.setName(fileName);
            analyzeFile.setData(blob);
            analyzeFile.setBatch(batch);
            analyzeFile.setAnalyze(analyze);

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't read file content", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save file in database", ex);
        } finally {
            IOUtils.closeQuietly(is);
        }
    }

    public AnalyzeFile getAnalyzeFileById(String analyzeFileId) {
        AnalyzeFile result = null;
        try {
            AnalyzeFileDAO analyzeFileDAO = daoHelper.getAnalyzeFileDAO();
            result = analyzeFileDAO.findByTopiaId(analyzeFileId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't find by id", ex);
        }
        return result;
    }

    /**
     * Génere le pdf de controle d'indentification ou botanique suivant
     * le produit du lot.
     * Le fichier géréré est temporaire.
     * 
     * @return le fichier pdf temporaire
     */
    public File generateControlFile(Batch batch) {

        File result = null;
        OutputStream os = null;
        try {
            String out = getControlHtmlContent(batch);

            // get content as w3c document
            Document document = SgqUtils.parseDocument(out.toString());

            // render template output as pdf
            result = File.createTempFile("batchcontrol-", ".pdf");
            //result.deleteOnExit();
            os = new FileOutputStream(result);

            ITextRenderer renderer = new ITextRenderer();
            // recherche d'un des ftl pour avoir la bonne url du bon jar
            URL baseUrl = BatchService.class.getResource("/ftl/botanic.ftl");
            String url = StringUtils.removeEnd(baseUrl.toExternalForm(), "botanic.ftl");
            renderer.setDocument(document, url);
            renderer.layout();
            renderer.createPDF(os);

            os.close();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't generate control file", ex);
        } catch (DocumentException ex) {
            throw new SgqBusinessException("Can't generate control file", ex);
        } finally {
            IOUtils.closeQuietly(os);
        }

        return result;
    }

    /**
     * Extraction de la génération seule du contenu html pour être plus
     * facilement testé.
     * 
     * @return le contenu html prêt à être convertit en pdf
     */
    protected String getControlHtmlContent(Batch batch) {
        // le service ne peut pas avoir d'état à cause du fonctionnement
        // par filtre topia, pas le choix, il faut l'instancier à chaque fois :(
        Configuration freemarkerConfiguration = new Configuration();
        // needed to overwrite "Defaults to default system encoding."
        // fix encoding issue on some systems
        freemarkerConfiguration.setEncoding(Locale.getDefault(), "UTF-8");
        // specific template loader to get template from jars (classpath)
        ClassTemplateLoader templateLoader = new ClassTemplateLoader(BatchService.class, "/ftl");
        freemarkerConfiguration.setTemplateLoader(templateLoader);
        // pour les maps dans les template (entre autre)
        freemarkerConfiguration.setObjectWrapper(new BeansWrapper());

        String result = null;
        try {
            // render freemarker template
            Template mapTemplate = null;
            if (batch.getProduct().isBotanicControl()) {
                mapTemplate = freemarkerConfiguration.getTemplate("botanic.ftl");
            } else if (batch.getProduct().isIdentificationControl()) {
                mapTemplate = freemarkerConfiguration.getTemplate("identification.ftl");
            } else {
                throw new IllegalArgumentException("Batch product doesn't define any control");
            }

            Map<String, Object> root = new HashMap<String, Object>();
            root.put("batch", batch);

            Writer out = new StringWriter();
            mapTemplate.process(root, out);
            out.flush();
            result = out.toString();

        } catch (IOException ex) {
            throw new SgqBusinessException("Can't generate control file", ex);
        } catch (TemplateException ex) {
            throw new SgqBusinessException("Can't generate control file", ex);
        }

        return result;
    }

    /**
     * Retourne le nom du fichier généré lors du télechargement.
     * 
     * @param batch batch
     * @return file name for control
     */
    public String getControlFilename(Batch batch) {

        String result = null;
        if (batch.getProduct().isBotanicControl()) {
            result = "CB";
        } else {
            result = "CI";
        }

        result += "_" + batch.getProduct().getName() + "_" + batch.getNumber() + ".pdf";

        return result;
    }

    /**
     * Reprise du fichier numero lots. Cet import est unique et ne peut
     * plus être utilisé dès que des lots sont présents.
     * 
     * @param file file to import
     */
    public void recoverBatchFile(File file) {
        long before = System.currentTimeMillis();
        if (log.isInfoEnabled()) {
            log.info("Recovering numero lot file " + file.getAbsolutePath());
        }

        Reader reader = null;
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();

            // cache some informations
            ClientDAO clientDAO = daoHelper.getClientDAO();
            Map<String, Client> clients = new HashMap<String, Client>();
            for (Client client : clientDAO.findAll()) {
                clients.put(client.getCode(), client);
            }
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            Map<String, Supplier> suppliers = new HashMap<String, Supplier>();
            for (Supplier supplier : supplierDAO.findAll()) {
                suppliers.put(supplier.getCode(), supplier);
            }
            ProductDAO productDAO = daoHelper.getProductDAO();
            Map<String, Product> products = new HashMap<String, Product>();
            for (Product product : productDAO.findAll()) {
                products.put(product.getCode(), product);
            }

            // check batch count
            if (batchDAO.count() > 0) {
                throw new SgqBusinessException("Can't recover batch file, if batch already exist in database !");
            }

            reader = new InputStreamReader(new FileInputStream(file), config.getImportCsvFileEncoding());
            BatchImportModel model = new BatchImportModel(suppliers, clients, products);
            ImportConf conf = new ImportConf();
            conf.setStrictMode(false);
            Import2<Batch> importCsv = Import2.newImport(conf, model, reader);

            for (ImportRow<Batch> row : importCsv) {

                Batch bean = row.getBean();

                if (row.isValid()) {
                    bean = batchDAO.create(bean);

                    // creation d'une presentation originale
                    // il n'y a l'air de n'avoir que des originales dans le fichier
                    Presentation presentation = presentationDAO.create();
                    presentation.setPresentationCode(PresentationCode._);
                    presentation.setBatch(bean);
                    presentation.setQuantity(bean.getQuantity());
                    presentationDAO.update(presentation);
                } else {
                    Throwable cause = null;
                    for (AbstractImportErrorInfo<Batch> error : row.getErrors()) {
                        if (cause == null) {
                            cause = error.getCause();
                        }
                    }
                    throw new SgqBusinessException("Can't commit import with errors", cause);
                }
            }

            daoHelper.commit();
            reader.close();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't read csv file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't commit modification in database", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }
        
        if (log.isInfoEnabled()) {
            long after = System.currentTimeMillis();
            log.info("File imported in " + (after-before) + " ms");
        }
    }

    /**
     * Export de l'inventaire.
     * 
     * @return le flux de lecture du fichier exporté (fichier temporaire)
     */
    public InputStream exportInventory() {
        InputStream result = null;

        try {
            File file = File.createTempFile("inventory", ".csv");
            file.deleteOnExit();

            BatchDAO batchDAO = daoHelper.getBatchDAO();
            // TODO echatellier 20121001 refaire plus propre
            // pour l'instant un search model vide recherche tout
            List<BatchModel> batchModels = batchDAO.findAllModel(new SearchModel(), 0, -1);

            List<InventoryBean> beans = new ArrayList<InventoryBean>();
            for (BatchModel batchModel : batchModels) {

                List<PresentationModel> presentationModels = findAllPresentationsByBatch(batchModel.getBatch());

                for (PresentationModel model : presentationModels) {

                    List<Place> places = model.getPlaces();
                    if (CollectionUtils.isNotEmpty(places)) {
                        for (Place place : places) {
                            InventoryBean bean = new InventoryBean();
                            bean.setBatch(batchModel.getBatch());
                            bean.setBatchModel(batchModel);
                            bean.setPresentation(model.getPresentation());
                            bean.setPlace(place);
                            beans.add(bean);
                        }
                    } else {
                        // je sais pas si ca peut arriver qu'une plante
                        // soit dans un emplacement non déterminé
                        // en tout cas dans les tests ca arrive
                        InventoryBean bean = new InventoryBean();
                        bean.setBatch(batchModel.getBatch());
                        bean.setPresentation(model.getPresentation());
                        bean.setBatchModel(batchModel);
                        beans.add(bean);
                    }
                }
            }

            InventoryExportModel model = new InventoryExportModel();
            String csv = Export.exportToString(model, beans, Charset.forName(config.getImportCsvFileEncoding()));
            FileUtils.writeStringToFile(file, csv, config.getImportCsvFileEncoding());
            result = new FileInputStream(file);
        } catch (Exception ex) {
            throw new SgqBusinessException("Can't write csv file", ex);
        }
        
        return result;
    }
    
    /**
     * Retourne tous les lots non expiré par emplacements.
     * 
     * @param place
     */
    public Map<Place, List<PresentationModel>> findAllBatchPerPlaces(Place place) {
        
        Map<Place, List<PresentationModel>> result = null;
        
        try {
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            List<Batch> batches = batchDAO.findAll();

            result = new HashMap<Place, List<PresentationModel>>();

            // FIXME echatellier 20121001 : ce code n'est pas du tout optimisé
            // il fait immensément de requete
            for (Batch batch : batches) {
                List<PresentationModel> presentationModels = findAllPresentationsByBatch(batch);

                for (PresentationModel model : presentationModels) {

                    List<Place> places = model.getPlaces();
                    for (Place presPlace : places) {
                        if (place == null || place.equals(presPlace)) {
                            List<PresentationModel> batchList = result.get(presPlace);
                            if (batchList == null) {
                                batchList = new ArrayList<PresentationModel>();
                                result.put(presPlace, batchList);
                            }
                            
                            batchList.add(model);
                        }
                    }
                }
            }
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't read database", ex);
        }
        
        return result;
    }
}
