package com.franciaflex.faxtomail.persistence.entities;

/*
 * #%L
 * FaxToMail :: Persistence
 * $Id: EmailTopiaDao.java 135 2014-06-05 12:42:57Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.2/faxtomail-persistence/src/main/java/com/franciaflex/faxtomail/persistence/entities/EmailTopiaDao.java $
 * %%
 * Copyright (C) 2014 Franciaflex
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

public class EmailTopiaDao extends AbstractEmailTopiaDao<Email> {

    /**
     * Search for email using filter.
     * 
     * @param emailFilter email filter
     * @return email list matching query filters
     */
    public List<Email> search(EmailFilter emailFilter) {

        StringBuilder query = new StringBuilder("FROM " + Email.class.getName() + " E");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = new HashMap<>();

        // email minReceptionDate
        if (emailFilter.getMinReceptionDate() != null) {
            query.append(" AND E." + Email.PROPERTY_RECEPTION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_RECEPTION_DATE);
            args.put(EmailFilter.PROPERTY_MIN_RECEPTION_DATE, emailFilter.getMinReceptionDate());
        }

        // email maxReceptionDate
        if (emailFilter.getMaxReceptionDate() != null) {
            query.append(" AND E." + Email.PROPERTY_RECEPTION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_RECEPTION_DATE);
            args.put(EmailFilter.PROPERTY_MAX_RECEPTION_DATE, emailFilter.getMaxReceptionDate());
        }

        // email taken by
        if (emailFilter.getTakenBy() != null) {
            query.append(" AND E." + Email.PROPERTY_TAKEN_BY + " = :" + EmailFilter.PROPERTY_TAKEN_BY);
            args.put(EmailFilter.PROPERTY_TAKEN_BY, emailFilter.getTakenBy());
        }

        // history minModificationDate && modifiedBy
        if (emailFilter.getMinModificationDate() != null || emailFilter.getModifiedBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :modificationType");
            args.put("modificationType", HistoryType.MODIFICATION);
            if (emailFilter.getMinModificationDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_MODIFICATION_DATE);
                args.put(EmailFilter.PROPERTY_MIN_MODIFICATION_DATE, emailFilter.getMinModificationDate());
            }
            if (emailFilter.getModifiedBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_MODIFIED_BY);
                args.put(EmailFilter.PROPERTY_MODIFIED_BY, emailFilter.getModifiedBy());
            }
            query.append(")");
        }

        // history maxModificationDate && modifiedBy
        if (emailFilter.getMaxModificationDate() != null || emailFilter.getModifiedBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :modificationType");
            args.put("modificationType", HistoryType.MODIFICATION);
            if (emailFilter.getMaxModificationDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_MODIFICATION_DATE);
                args.put(EmailFilter.PROPERTY_MAX_MODIFICATION_DATE, emailFilter.getMaxModificationDate());
            }
            if (emailFilter.getModifiedBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_MODIFIED_BY);
                args.put(EmailFilter.PROPERTY_MODIFIED_BY, emailFilter.getModifiedBy());
            }
            query.append(")");
        }

        // history minTransferDate && transferBy
        if (emailFilter.getMinTransferDate() != null || emailFilter.getTransferBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :transmissionType");
            args.put("transmissionType", HistoryType.TRANSMISSION);
            if (emailFilter.getMinTransferDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_TRANSFER_DATE);
                args.put(EmailFilter.PROPERTY_MIN_TRANSFER_DATE, emailFilter.getMinTransferDate());
            }
            if (emailFilter.getTransferBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_TRANSFER_BY);
                args.put(EmailFilter.PROPERTY_TRANSFER_BY, emailFilter.getTransferBy());
            }
            query.append(")");
        }

        // history maxTransferDate && transferBy
        if (emailFilter.getMaxTransferDate() != null || emailFilter.getTransferBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :transmissionType");
            args.put("transmissionType", HistoryType.TRANSMISSION);
            if (emailFilter.getMaxTransferDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_TRANSFER_DATE);
                args.put(EmailFilter.PROPERTY_MAX_TRANSFER_DATE, emailFilter.getMaxTransferDate());
            }
            if (emailFilter.getTransferBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_TRANSFER_BY);
                args.put(EmailFilter.PROPERTY_TRANSFER_BY, emailFilter.getTransferBy());
            }
            query.append(")");
        }

        // history minArchivedDate && transferBy
        if (emailFilter.getMinArchivedDate() != null || emailFilter.getArchivedBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :archivedType");
            args.put("archivedType", HistoryType.ARCHIVED);
            if (emailFilter.getMinArchivedDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_ARCHIVED_DATE);
                args.put(EmailFilter.PROPERTY_MIN_ARCHIVED_DATE, emailFilter.getMinArchivedDate());
            }
            if (emailFilter.getArchivedBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_ARCHIVED_BY);
                args.put(EmailFilter.PROPERTY_ARCHIVED_BY, emailFilter.getArchivedBy());
            }
            query.append(")");
        }

        // history maxArchivedDate && transferBy
        if (emailFilter.getMaxArchivedDate() != null || emailFilter.getArchivedBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :transmissionType");
            args.put("transmissionType", HistoryType.TRANSMISSION);
            if (emailFilter.getMaxArchivedDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_ARCHIVED_DATE);
                args.put(EmailFilter.PROPERTY_MAX_ARCHIVED_DATE, emailFilter.getMaxArchivedDate());
            }
            if (emailFilter.getArchivedBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_ARCHIVED_BY);
                args.put(EmailFilter.PROPERTY_ARCHIVED_BY, emailFilter.getArchivedBy());
            }
            query.append(")");
        }

        // email minPrintingDate
        if (emailFilter.getMinPrintingDate() != null || emailFilter.getPrintingBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :printingType");
            args.put("printingType", HistoryType.PRINTING);
            if (emailFilter.getMinPrintingDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_PRINTING_DATE);
                args.put(EmailFilter.PROPERTY_MIN_PRINTING_DATE, emailFilter.getMinPrintingDate());
            }
            if (emailFilter.getPrintingBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_PRINTING_BY);
                args.put(EmailFilter.PROPERTY_PRINTING_BY, emailFilter.getPrintingBy());
            }
            query.append(")");
        }

        // email maxPrintingDate
        if (emailFilter.getMaxPrintingDate() != null || emailFilter.getPrintingBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :printingType");
            args.put("printingType", HistoryType.PRINTING);
            if (emailFilter.getMaxPrintingDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_PRINTING_DATE);
                args.put(EmailFilter.PROPERTY_MAX_PRINTING_DATE, emailFilter.getMaxPrintingDate());
            }
            if (emailFilter.getPrintingBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_PRINTING_BY);
                args.put(EmailFilter.PROPERTY_PRINTING_BY, emailFilter.getPrintingBy());
            }
            query.append(")");
        }

        // email minReplyDate
        if (emailFilter.getMinReplyDate() != null || emailFilter.getReplyBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :replyType");
            args.put("replyType", HistoryType.REPLY);
            if (emailFilter.getMinReplyDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " >= :" + EmailFilter.PROPERTY_MIN_REPLY_DATE);
                args.put(EmailFilter.PROPERTY_MIN_REPLY_DATE, emailFilter.getMinReplyDate());
            }
            if (emailFilter.getReplyBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_REPLY_BY);
                args.put(EmailFilter.PROPERTY_REPLY_BY, emailFilter.getReplyBy());
            }
            query.append(")");
        }

        // email maxReplyDate
        if (emailFilter.getMaxReplyDate() != null || emailFilter.getReplyBy() != null) {
            query.append(" AND EXISTS (FROM " + History.class.getName() + " H WHERE H in elements(E." + Email.PROPERTY_HISTORY + ")");
            query.append("   AND H." + History.PROPERTY_TYPE + " = :replyType");
            args.put("replyType", HistoryType.REPLY);
            if (emailFilter.getMaxReplyDate() != null) {
                query.append("   AND H." + History.PROPERTY_MODIFICATION_DATE + " <= :" + EmailFilter.PROPERTY_MAX_REPLY_DATE);
                args.put(EmailFilter.PROPERTY_MAX_REPLY_DATE, emailFilter.getMaxReplyDate());
            }
            if (emailFilter.getReplyBy() != null) {
                query.append("   AND H." + History.PROPERTY_FAX_TO_MAIL_USER + " = :" + EmailFilter.PROPERTY_REPLY_BY);
                args.put(EmailFilter.PROPERTY_REPLY_BY, emailFilter.getReplyBy());
            }
            query.append(")");
        }

        // email sender
        if (StringUtils.isNotBlank(emailFilter.getSender())) {
            query.append(" AND lower(E." + Email.PROPERTY_SENDER + ") LIKE lower(:" + EmailFilter.PROPERTY_SENDER + ")");
            args.put(EmailFilter.PROPERTY_SENDER, "%" + emailFilter.getSender() + "%");
        }

        // client code
        if (StringUtils.isNoneBlank(emailFilter.getClientCode())) {
            query.append(" AND lower(E." + Email.PROPERTY_CLIENT + "." + Client.PROPERTY_CODE + ") LIKE lower(:" + EmailFilter.PROPERTY_CLIENT_CODE + ")");
            args.put(EmailFilter.PROPERTY_CLIENT_CODE, "%" + emailFilter.getClientCode() + "%");
        }

        // email subject
        if (StringUtils.isNotBlank(emailFilter.getDemandObject())) {
            query.append(" AND lower(E." + Email.PROPERTY_OBJECT + ") LIKE lower(:" + EmailFilter.PROPERTY_DEMAND_OBJECT + ")");
            args.put(EmailFilter.PROPERTY_DEMAND_OBJECT, "%" + emailFilter.getDemandObject() + "%");
        }

        // email demand type
        List<DemandType> demandType = emailFilter.getDemandType();
        if (CollectionUtils.isNotEmpty(demandType)) {
            query.append(" AND (E." + Email.PROPERTY_DEMAND_TYPE + " IN (:" + EmailFilter.PROPERTY_DEMAND_TYPE + ")");
            args.put(EmailFilter.PROPERTY_DEMAND_TYPE, demandType);

            if (demandType.contains(null)) {
                query.append(" OR E." + Email.PROPERTY_DEMAND_TYPE + " IS NULL");
            }
            query.append(")");
        }

        // email ediCodeNumber
        if (StringUtils.isNotBlank(emailFilter.getEdiCodeNumber())) {
            query.append(" AND lower(E." + Email.PROPERTY_EDI_CODE_NUMBER + ") LIKE lower(:" + EmailFilter.PROPERTY_EDI_CODE_NUMBER + ")");
            args.put(EmailFilter.PROPERTY_EDI_CODE_NUMBER, "%" + emailFilter.getEdiCodeNumber() + "%");
        }

        // email projectReference
        if (StringUtils.isNotBlank(emailFilter.getProjectReference())) {
            query.append(" AND lower(E." + Email.PROPERTY_PROJECT_REFERENCE + ") LIKE lower(:" + EmailFilter.PROPERTY_PROJECT_REFERENCE + ")");
            args.put(EmailFilter.PROPERTY_PROJECT_REFERENCE, "%" + emailFilter.getProjectReference() + "%");
        }

        // email priority
        List<Priority> priority = emailFilter.getPriority();
        if (CollectionUtils.isNotEmpty(priority)) {
            query.append(" AND (E." + Email.PROPERTY_PRIORITY + " IN (:" + EmailFilter.PROPERTY_PRIORITY + ")");
            args.put(EmailFilter.PROPERTY_PRIORITY, priority);

            if (priority.contains(null)) {
                query.append(" OR E." + Email.PROPERTY_PRIORITY + " IS NULL");
            }
            query.append(")");
        }

        // email demand status
        List<DemandStatus> demandStatus = emailFilter.getDemandStatus();
        if (CollectionUtils.isNotEmpty(demandStatus)) {
            query.append(" AND (E." + Email.PROPERTY_DEMAND_STATUS + " IN (:" + EmailFilter.PROPERTY_DEMAND_STATUS + ")");
            args.put(EmailFilter.PROPERTY_DEMAND_STATUS, demandStatus);

            if (demandStatus.contains(null)) {
                query.append(" OR E." + Email.PROPERTY_DEMAND_STATUS + " IS NULL");
            }
            query.append(")");
        }

        // email recipient
        /*if (StringUtils.isNotBlank(emailFilter.getRecipient())) {
            query.append(" AND lower(E." + Email.PROPERTY_RECIPIENT + ") LIKE lower(:" + EmailFilter.PROPERTY_RECIPIENT + ")");
            args.put(EmailFilter.PROPERTY_RECIPIENT, "%" + emailFilter.getRecipient() + "%");
        }*/

        // email etat attente
        List<EtatAttente> etatAttente = emailFilter.getEtatAttente();
        if (CollectionUtils.isNotEmpty(etatAttente)) {
            query.append(" AND (E." + Email.PROPERTY_ETAT_ATTENTE + " IN (:" + EmailFilter.PROPERTY_ETAT_ATTENTE + ")");
            args.put(EmailFilter.PROPERTY_ETAT_ATTENTE, etatAttente);

            if (etatAttente.contains(null)) {
                query.append(" OR E." + Email.PROPERTY_ETAT_ATTENTE + " IS NULL");
            }
            query.append(")");
        }

        // email comment
        if (StringUtils.isNotBlank(emailFilter.getMessage())) {
            query.append(" AND lower(E." + Email.PROPERTY_COMMENT + ") LIKE lower(:" + EmailFilter.PROPERTY_MESSAGE + ")");
            args.put(EmailFilter.PROPERTY_MESSAGE, "%" + emailFilter.getMessage() + "%");
        }

        // email gamme
        List<Range> gamme = emailFilter.getGamme();
        if (CollectionUtils.isNotEmpty(gamme)) {
            query.append(" AND ((EXISTS (FROM " + RangeRow.class.getName() + " RR WHERE RR in elements(E." + Email.PROPERTY_RANGE_ROW + ")");
            query.append("   AND RR." + RangeRow.PROPERTY_RANGE + " IN (:" + EmailFilter.PROPERTY_GAMME + "))");
            args.put(EmailFilter.PROPERTY_GAMME, gamme);

            if (gamme.contains(null)) {
                query.append(" OR NOT EXISTS (FROM " + RangeRow.class.getName() + " RR WHERE RR in elements(E." + Email.PROPERTY_RANGE_ROW + "))");
            }
            query.append("))");
        }

//        // email comment
//        if (StringUtils.isNotBlank(emailFilter.getBody())) {
//            query.append(" AND lower(E." + Email.PROPERTY_HTML_CONTENT + ") LIKE lower(:" + EmailFilter.PROPERTY_BODY + ")");
//            args.put(EmailFilter.PROPERTY_BODY, "%" + emailFilter.getBody() + "%");
//        }

        // email localReference
        if (StringUtils.isNotBlank(emailFilter.getLocalReference())) {
            query.append(" AND (EXISTS (FROM " + RangeRow.class.getName() + " RR WHERE RR in elements(E." + Email.PROPERTY_RANGE_ROW + ")");
            query.append("   AND lower(RR." + RangeRow.PROPERTY_COMMAND_NUMBER + ") LIKE lower(:" + EmailFilter.PROPERTY_LOCAL_REFERENCE + "))");
            query.append(" OR lower(E." + Email.PROPERTY_COMPANY_REFERENCE + ") LIKE lower(:" + EmailFilter.PROPERTY_LOCAL_REFERENCE + ")");
            args.put(EmailFilter.PROPERTY_LOCAL_REFERENCE, "%" + emailFilter.getLocalReference() + "%");
            query.append(")");
        }

        return findAll(query.toString(), args);
    }

    /**
     * Retounre le nombre d'email par dossier.
     * 
     * @return le nombre d'email par dossier
     */
    public Map<String, Long> getMailCountByFolder() {
        String query = "SELECT mailFolder.topiaId, count(*) FROM " + Email.class.getName() + " group by mailFolder.topiaId";
        
        Map<String, Long> result = new HashMap<>();
        
        List<Object[]> queryResuts = findAll(query);
        for (Object[] queryResut : queryResuts) {
            String mailFolder = (String)queryResut[0];
            Long count = (Long)queryResut[1];
            result.put(mailFolder, count);
        }
        return result;
    }
}
