package org.planx.xmlstore.stores;

import java.io.*;
import java.util.*;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nameserver.LocalNameServer;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import org.planx.xmlstore.regions.*;

/**
 * A local <code>XMLStore</code> that only operates on location dependent
 * <code>Reference</code>s, that is, <code>LocalLocator</code>s. Thus, it is
 * fast but unable to resolve <code>ValueReference</code>s. In order to do this,
 * wrap the <code>LocalXMLStore</code> in a {@link TranslatorXMLStore}.
 * <p>
 * A <code>LocalXMLStore</code> will run a <code>Sharer</code> in the background
 * that compacts redundant on-disk data and performs garbage collection.
 *
 * @author Thomas Ambus
 */
public class LocalXMLStore extends AbstractXMLStore {
    private String name;
    private RegionManager manager;
    private NameServer nameServer;

    /**
     * Creates a new <code>LocalXMLStore</code> with the specified on-disk name.
     *
     * @param name         the on-disk file name to use as base for persistent data
     * @throws IOException if an error occurred initializing the <code>XMLStore</code>
     *                     using the specified <code>name</code>
     */
    public LocalXMLStore(String name) throws IOException {
        this(name, new RegionConfiguration());
    }

    /**
     * Creates a new <code>LocalXMLStore</code> with the specified on-disk name.
     *
     * @param name         the on-disk file name to use as base for persistent data
     * @throws IOException if an error occurred initializing the <code>XMLStore</code>
     *                     using the specified <code>name</code>
     */
    public LocalXMLStore(String name, RegionConfiguration conf) throws IOException {
        this.name = name;
        manager = new RegionManager(this, conf);
        nameServer = new LocalNameServer(this);
    }

    /**
     * Saves the specified <code>Node</code> and returns a <code>LocalLocator</code> to it.
     */
    public Reference save(Node node) throws IOException {
        checkClosed();
        return manager.save(DVMNodeFactory.convert(node));
    }

    /**
     * Only accepts <code>LocalLocator</code>s.
     */
    protected Node resolvedLoad(Reference vref) throws IOException,
                                         UnknownReferenceException {
        checkClosed();
        LocalLocator loc = (LocalLocator) vref;
        return manager.load(loc);
    }

    /**
     * Returns a <code>NameServer</code> for this store.
     */
    public NameServer getNameServer() {
        return nameServer;
    }

    /**
     * Access to the <code>RegionManager</code> is provided merely for testing
     * purposes and for accessing the managers statistics.
     */
    public RegionManager getRegionManager() {
        return manager;
    }

    /**
     * Informs the <code>XMLStore</code> that the data referenced by the
     * specified <code>Reference</code> should be kept permanently.
     * Implementations may assume that the <code>Reference</code> is one
     * previously obtained from this <code>XMLStore</code>.
     * When the data referenced is no longer needed the <code>XMLStore</code>
     * should be informed of this be calling {@link #release(Reference)}.
     *
     * @param ref the <code>Reference</code> to retain
     */
    public void retain(Reference ref) throws UnknownReferenceException {
        manager.retain((LocalLocator) ref);
    }

    /**
     * Informs the <code>XMLStore</code> that the data referenced by the
     * specified <code>Reference</code> is no longer needed. The
     * <code>XMLStore</code> should previously have been informed to retain
     * the data using the {@link #retain(Reference)} method.
     * More specifically, implementations may assume that the argument of this
     * method, or one <code>equal</code> to it, has previously been an argument
     * to <code>retain</code>.
     *
     * @param ref the <code>Reference</code> to release
     */
    public void release(Reference ref) throws UnknownReferenceException {
        manager.release((LocalLocator) ref);
    }

    /**
     * Adds the specified <code>ReferenceListener</code> for receiving
     * <code>Reference</code> events.
     *
     * @param listener the listener to add
     */
    public void addReferenceListener(ReferenceListener listener) {
        manager.addRootListener(listener);
    }

    /**
     * Removes the specified <code>ReferenceListener</code> previously added
     * with <code>addReferenceListener</code>.
     *
     * @param listener the listener to remove
     */
    public void removeReferenceListener(ReferenceListener listener) {
        manager.removeRootListener(listener);
    }

    /**
     * Returns the disk size of this <code>LocalXMLStore</code>.
     */
    public long size() throws IOException {
        return manager.size();
    }

    public void close() throws IOException {
        manager.close();
        manager = null;
    }

    public String toString() {
        return name;
    }

    protected void checkClosed() {
        if (manager == null) throw new IllegalStateException("XMLStore is closed");
    }
}
