package org.planx.xmlstore.docnodes;

import java.util.ArrayList;
import java.util.List;
import org.planx.util.Array;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Attribute;

/**
 * DocNode is an occurrence of a {@link Node} in a document.
 * DocNode provides the ability to locate the parent, root,
 * and siblings of a node.
 * A DocNode object is unique in an instance of a document
 * with respect to the {@link Object#equals(Object)} and
 * {@link Object#hashCode()}
 * methods. Note, however, that if a document is created multiple
 * times using the same root {@link Node}, the documents are
 * considered to be distinct and each document will
 * have its own set of DocNode objects.
 *
 * @author Thomas Ambus
 */
public class DocNode implements Node {
    private Node node;
    private DocNode parent;
    private int childIndex; // -1 indicates root
    private DocNode root;

    private List<DocNode> docChildren = null;
    private List<DocAttribute> docAttributes = null;

    /**
     * Create a DocNode which is the root of
     * a document.
     *
     * @param node       the DVM Node
     */
    public DocNode(Node node) {
        this.node = node;
        this.parent = null;
        this.childIndex = -1;
        this.root = this;
    }


    /**
     * Create a DocNode which is an internal
     * node of a document.
     */
    DocNode(Node node, DocNode parent, int childIndex, DocNode root) {
        this.node = node;
        this.parent = parent;
        this.childIndex = childIndex;
        this.root = root;
    }

    /**
     * Returns an integer array containing the path from the root to this node.
     * The integers represent the child indexes of each node from the root to this node.
     */
    public int[] getPath() {
        List path = new ArrayList();
        makePath(path);
        int length = path.size();
        int[] a = new int[length];
        for (int i=0; i<length; i++) {
            a[i] = ((Integer) path.get(i)).intValue();
        }
        return a;
    }

    private void makePath(List path) {
        if (parent != null) {
            parent.makePath(path);
            path.add(new Integer(childIndex));
        }
    }

    // Navigation methods

    /**
     * Returns the root DocNode of the document
     * that contains this DocNode.
     */
    public DocNode getRoot() {
        return root;
    }

    /**
     * Returns the parent DocNode of this DocNode.
     * If there is no parent, i.e. this is the root
     * then <code>null</code> is returned.
     */
    public DocNode getParent() {
        return parent;
    }

    /**
     * Returns the index of this node in the parent's
     * list of children. If this node does not have a
     * parent, i.e. this is the root, then -1 is returned.
     */
    public int getIndex() {
        return childIndex;
    }

    /**
     * Returns the child with the specified index.
     */
    public DocNode getChild(int index) {
        return getDocChildren().get(index);
    }

    /**
     * Returns the number of children of this node.
     */
    public int childCount() {
        return getDocChildren().size();
    }

    /**
     * Returns a list that contains the children of this node
     * as <code>DocNode</code>s.
     */
    public List<DocNode> getDocChildren() {
        if (docChildren == null) {
            List<? extends Node> children = node.getChildren();
            DocNode[] dn = new DocNode[children.size()];
            for (int i=0, max=children.size(); i<max; i++) {
                dn[i] = new DocNode(children.get(i), this, i, root);
            }
            docChildren = Array.asUnmodifiableList(dn);
        }
        return docChildren;
    }

    /**
     * Returns the sibling node immediately following this node.
     * If there is no such node, this returns null.
     */
    public DocNode nextSibling() {
        if (parent == null) return null;
        List<DocNode> siblings = parent.getDocChildren();
        if (siblings.size() > childIndex + 1) {
            return siblings.get(childIndex + 1);
        } else {
            return null;
        }
    }

    /**
     * Returns the sibling node immediately preceding this node.
     * If there is no such node, this returns null.
     */
    public DocNode previousSibling() {
        if (parent == null) return null;
        List<DocNode> siblings = parent.getDocChildren();
        if (childIndex - 1 >= 0) {
            return siblings.get(childIndex - 1);
        } else {
            return null;
        }
    }

    // Attribute methods

    /**
     * Returns the number of attributes of this node.
     */
    public int attributeCount() {
        return getDocAttributes().size();
    }

    /**
     * Returns the attribute with the specified index.
     * The indexing is merely for convenience, it does
     * not imply a specific ordering of the attributes.
     */
    public DocAttribute getAttribute(int index) {
        return getDocAttributes().get(index);
    }

    public List<DocAttribute> getDocAttributes() {
        if (docAttributes == null) {
            List<Attribute> attributes = node.getAttributes();
            DocAttribute[] dn = new DocAttribute[attributes.size()];
            for (int i=0, max=attributes.size(); i<max; i++) {
                Attribute attr = attributes.get(i);
                dn[i] = new DocAttribute(attr, this);
            }
            docAttributes = Array.asUnmodifiableList(dn);
        }
        return docAttributes;
    }

    // Delegate methods specified by Node interface

    public boolean isMutable() {
        return node.isMutable();
    }

    public byte getType() {
        return node.getType();
    }

    public String getNodeValue() {
        return node.getNodeValue();
    }

    public String getAttribute(String attrName) {
        return node.getAttribute(attrName);
    }

    public String[] getAttributeNames() {
        return node.getAttributeNames();
    }

    public List<? extends Node> getChildren() {
        return node.getChildren();
    }

    public List<Attribute> getAttributes() {
        return node.getAttributes();
    }

    public String toString() {
        return node.toString();
    }
}
