/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2012 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.query;

import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.BusinessEntityImpl;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.query.conditions.Aggregate;
import org.nuiton.wikitty.query.conditions.And;
import org.nuiton.wikitty.query.conditions.Between;
import org.nuiton.wikitty.query.conditions.ConditionValue;
import org.nuiton.wikitty.query.conditions.Condition;
import org.nuiton.wikitty.query.conditions.ConditionValueString;
import org.nuiton.wikitty.query.conditions.ContainsAll;
import org.nuiton.wikitty.query.conditions.ContainsOne;
import org.nuiton.wikitty.entities.Element;
import org.nuiton.wikitty.entities.ElementField;
import org.nuiton.wikitty.query.conditions.Equals;
import org.nuiton.wikitty.query.conditions.False;
import org.nuiton.wikitty.query.conditions.Greater;
import org.nuiton.wikitty.query.conditions.GreaterOrEquals;
import org.nuiton.wikitty.query.conditions.Keyword;
import org.nuiton.wikitty.query.conditions.Less;
import org.nuiton.wikitty.query.conditions.LessOrEquals;
import org.nuiton.wikitty.query.conditions.Like;
import org.nuiton.wikitty.query.conditions.Not;
import org.nuiton.wikitty.query.conditions.NotEquals;
import org.nuiton.wikitty.query.conditions.NotNull;
import org.nuiton.wikitty.query.conditions.Null;
import org.nuiton.wikitty.query.conditions.Or;
import org.nuiton.wikitty.query.conditions.Select;
import org.nuiton.wikitty.query.conditions.True;
import org.nuiton.wikitty.query.conditions.Unlike;

/**
 * Cette objet sert a construire une condition a la facon d'un flux.
 * <p>
 * Condition c = new WikittyQueryMaker().and().eq("ext.field", "toto").eq("ext.field2", 10).getCondition();
 * <p>
 * On peut aussi vouloir continuer avec un WikittyQuery pour cela on peut faire.
 * <p>
 * WikittyQuery q = new WikittyQueryMaker().and().[other condition].end();
 * <p>
 * Si un {@link WikittyQuery} est passé en parametre du constructeur et que la
 * method {@link #end()} est appeler alors la condition creee est envoyee dans
 * le WikittyQuery et le constructeur est fermer (on ne peut plus ajouter de
 * condition)
 * <p>
 * Le WikittyQuery lie avec cet objet lorsqu'on le recupere via la method
 * {@link #getQuery()} a en interne comme condition la valuer courante de la
 * condition en cours d'ecriture
 *
 * @author poussin
 * @version $Revision$
 * @since 3.3
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyQueryMaker {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyQueryMaker.class);

    /** query where to send condition when end() method called */
    protected WikittyQuery query;

    /** query condition */
    protected Condition condition;
    /** stack des conditions non terminales ouvertes */
    protected Deque<Condition> openStack = new LinkedList<Condition>();

    public WikittyQueryMaker() {
    }

    public WikittyQueryMaker(WikittyQuery query) {
        this.query = query;
    }

    public Condition getCondition() {
        return condition;
    }

    /**
     * La query passee dans le constructeur ou une nouvelle query si aucune
     * query n'avait ete passee dans le constructeur
     * 
     * @return
     */
    public WikittyQuery getQuery() {
        if (query == null) {
            query = new WikittyQuery();
        }
        // la condition de la query doit toujours refleter la valeur courante
        // de la condition en cours de creation
        query.setCondition(getCondition());
        return query;
    }

    /**
     * Retourne le stack courant des conditions, si le stack est null cela
     * veut dire que le maker ne permet plus de modification
     * @return
     */
    protected Deque<Condition> getOpenStack() {
        if (openStack == null) {
            throw new WikittyException(
                    "You can't create condition if you have used setCondition method"
                    + " or close last condition");
        }
        return openStack;
    }

    /**
     * Ajout une condition
     *
     */
    protected void addCondition(Condition c) {
        addCondition(c, false);
    }
    /**
     * Ajout une condition. Si terminal est true, alors quelque soit le type de
     * cette condition la stack restera inchange apres l'ajout. Cela permet
     * d'ajouter des conditions non terminale comme des terminales
     */
    protected void addCondition(Condition c, boolean terminal) {
        Condition parent = getOpenStack().peek();
        if (!terminal) {
            // on ne met les conditions toujours dans le stack (meme les terminales)
            // si c'est la premiere cela permet d'indiquer a l'utilisateur
            // qu'il faut commencer par un or, and, not car une exception sera
            // levee car non supprimer par le while dans lequel on ne passe pas
            // car on passe dans le if et non le else (parent == null)
            getOpenStack().push(c);
        }

        if (parent == null) {
            // il n'y a rien dans la stack donc rien dans condition, il faut
            // mettre cette condition dedans
            condition = c;
            // et on ne depile pas le stack
        } else {
            // se add peut lever une exception si parent n'accepte pas cet enfant
            parent.addCondition(c);
            // on depile toutes les conditions qui n'ont plus besoin de renseignement
            // sauf si l'ajout a ete force en terminal
            if (!terminal) {
                closeIfNecessary();
            }
        }
    }

    protected void closeIfNecessary() {
        // on depile toutes les conditions qui n'ont plus besoin de renseignement
        while (getOpenStack().peek() != null
                && !getOpenStack().peek().waitCondition()) {
            getOpenStack().poll();
        }
        if (getOpenStack().size() == 0) {
            // we just close last open condition, set stack to null, to prevent
            // next condition add that is mistake
            openStack = null;
        }
    }
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // Q U E R  Y   F L O W   C R E A T I O N
    //
    ///////////////////////////////////////////////////////////////////////////

    // eq(Wikitty|Date|Number|Boolean|String)

    static protected ConditionValue convertToConditionValue(Object o) {
        ConditionValue result;
        if (o instanceof ConditionValue) {
            result = (ConditionValue)o;
        } else {
            String s = WikittyUtil.toString(o);
            result = new ConditionValueString(s);
        }
        return result;
    }

    public WikittyQueryMaker value(Object value) {
        ConditionValue v = convertToConditionValue(value);
        addCondition(v);
        return this;
    }

    /**
     * Ajoute une condition, cette condition est prise comme une condition terminal
     * Si l'on veut continuer a construire la requete, il faut avoir ajouter
     * avant une {@link #and()}, {@link #or()}, {@link #not()}, {@link #in()}
     * @param c la condition a ajouter
     * @return {@code this} with the {@code c} restriction added.
     */
    public WikittyQueryMaker condition(Condition c) {
        addCondition(c, true);
        return this;
    }

    /**
     * Ajoute une contrainte qui cree les conditions en prenant comme exemple
     * l'objet passer en parametre. Seuls les champs non null sont utilises ainsi
     * que la liste des extensions de l'objet
     *
     * @param w le wikitty a prendre comme exemple
     * @return {@code this} with the {@code w} restriction added.
     */
    public WikittyQueryMaker wikitty(Wikitty w) {
        WikittyQueryMaker result = new WikittyQueryMaker().and();

        // result object must have same extension that wikitty example
        result.extContainsAll(w.getExtensionNames());
        
        for (String fqfieldName : w.fieldNames()) {
            Object value = w.getFqField(fqfieldName);
            if (value != null) {
                FieldType type = w.getFieldType(fqfieldName);
                if (type.isCollection()) {
                    result.containsAll(fqfieldName, (Collection<?>)value);
                } else {
                    result.eq(fqfieldName, value);
                }
            }
        }
        addCondition(result.getCondition());

        return this;
    }

    /**
     * Ajoute une contrainte qui cree les conditions en prenant comme exemple
     * l'objet passer en parametre. Seuls les champs non null sont utilises ainsi
     * que la liste des extensions de l'objet
     *
     * @param e l'objet a prendre comme exemple
     * @return {@code this} with the {@code e} restriction added.
     */
    public WikittyQueryMaker wikitty(BusinessEntityImpl e) {
        Wikitty w = e.getWikitty();
        return wikitty(w);
    }

    /**
     * @see {@link ContainsAll}
     */
    public WikittyQueryMaker containsAll(Element element) {
        addCondition(new ContainsAll(element));
        return this;
    }

    /**
     * @see {@link ContainsAll}
     */
    public WikittyQueryMaker containsAll(String element) {
        return containsAll(new ElementField(element));
    }

    /**
     * Contains.
     *
     * Search on lists (multivalued fields) that a field contains all the values
     * of the list given in parameter.
     *
     * Ex : The field with value [toto,titi,tutu] contains [titi,tutu] but not
     * [titi,tutu,tata]
     *
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @param element the element on which the restriction is put
     * @param values the values to search in the element
     * @return {@code this} with the {@code contains} restriction added.
     * @see {@link ContainsAll}
     */
    public <E> WikittyQueryMaker containsAll(String fqfield, Collection<E> values) {
        return containsAll(new ElementField(fqfield), values);
    }

    /**
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @see {@link ContainsAll}
     */
    public <E> WikittyQueryMaker containsAll(Element element, Collection<E> values) {
        containsAll(element);
        for (E e : values) {
            value(e);
        }
        close();
        return this;
    }

    /**
     * Search on lists (multivalued fields) that a field contains all the values
     * given in parameter.
     *
     * Ex : The field with value [toto,titi,tutu] contains [titi,tutu] but not
     * [titi,tutu,tata]
     *
     * Ps : Use wildcards if you search for substrings.
     *
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @param element the element on which the restriction is put
     * @param value1 first value to search in the field
     * @param values list of values to search in the field
     * @return {@code this} with the {@code contains} restriction added.
     * @see {@link ContainsAll}
     */
    public <E> WikittyQueryMaker containsAll(String fqfield, E value1, E ... values) {
        List<E> l = new LinkedList<E>();
        l.add(value1);
        l.addAll(Arrays.asList(values));
        return containsAll(fqfield, l);
    }

    /**
     * @see {@link ContainsOne}
     */
    public WikittyQueryMaker containsOne(Element element) {
        addCondition(new ContainsOne(element));
        return this;
    }

    /**
     * @see {@link ContainsOne}
     */
    public WikittyQueryMaker containsOne(String element) {
        return containsOne(new ElementField(element));
    }

    /**
     * Search if a field is contained in the list of values in parameter
     *
     * Ex : The field with value titi is in [titi,tutu] but not in
     * [tutu,tata]
     *
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @param element the element on which the restriction is put
     * @param values list of values the field must be in
     * @return {@code this} with the {@code in} restriction added.
     * @see {@link ContainsOne}
     */
    public <E> WikittyQueryMaker containsOne(String fqfield, Collection<E> values) {
        return containsOne(new ElementField(fqfield), values);
    }

    /**
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @see {@link ContainsOne}
     */
    public <E> WikittyQueryMaker containsOne(Element element, Collection<E> values) {
        containsOne(element);
        for (E e : values) {
            value(e);
        }
        close();
        return this;
    }

    /**
     * Search if a field is contained in the list of values in parameter
     *
     * Ex : The field with value titi is in [titi,tutu] but not in
     * [tutu,tata]
     *
     * Ps : Use wildcards in the values if you search for substrings.
     *
     * Force l'ajout du containsAll en terminal (il n'y a pas besoin de faire
     * de {@link #close()}
     *
     * @param element the element on which the restriction is put
     * @param value1 first value the field must be in
     * @param values list of values the field must be in
     * @return {@code this} with the {@code in} restriction added.
     * @see {@link ContainsOne}
     */
    public <E> WikittyQueryMaker containsOne(String fqfield, E value1, E ... values) {
        List<E> l = new LinkedList<E>();
        l.add(value1);
        l.addAll(Arrays.asList(values));
        return containsOne(fqfield, l);
    }

    /**
     * @see {@link Equals}
     */
    public WikittyQueryMaker eq(Element element) {
        addCondition(new Equals(element));
        return this;
    }

    /**
     * Equals.
     *
     * Restrict search so that the field value equals the parameter.
     *
     * You might use patterns in your equality.
     *
     * @param element the field on which the search is made
     * @param value the value the element must be equals to
     * @return {@code this}
     * @see {@link Equals}
     */
    public WikittyQueryMaker eq(String fqfield, Object value) {
        return eq(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Equals}
     */
    public WikittyQueryMaker eq(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new Equals(element, s));
        return this;
    }

    /**
     * Extension equals.
     *
     * Restrict search to wikitties that got the extension in parameter.
     *
     * @param s the extension to restrict the results to
     * @return {@code this} with the {@code exteq} restriction added.
     * @see {@link Equals}
     */
    public WikittyQueryMaker exteq(String extensionName) {
        return eq(Element.EXTENSION, extensionName);
    }

    /**
     * Id equals.
     *
     * Restrict search to wikitties that got the id in parameter.
     *
     * @param value the id or wikitty to restrict the results to
     * @return {@code this} with the {@code ideq} restriction added.
     * @see {@link Equals}
     */
    public WikittyQueryMaker ideq(Object idOrWikitty) {
        return eq(Element.ID, idOrWikitty);
    }

    /**
     * Extension equals.
     *
     * Restrict search to wikitties that got all the extensions in parameter.
     *
     * @param extensionNames list of the extension to restrict the results to
     * @return {@code this} with the {@code exteq} restriction added.
     * @see {@link ContainsAll}
     */
    public WikittyQueryMaker extContainsAll(Collection<String> extensionNames) {
        return containsAll(Element.EXTENSION, extensionNames);
    }

    /**
     * @see {@link ContainsAll}
     */
    public WikittyQueryMaker extContainsAll(String ext1, String ... exts) {
        List<String> l = new LinkedList<String>();
        l.add(ext1);
        l.addAll(Arrays.asList(exts));
        return containsAll(Element.EXTENSION, l);
    }

    /**
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker ne(Element element) {
        addCondition(new NotEquals(element));
        return this;
    }

    /**
     * Not equals.
     *
     * Restrict search to elements that are not equals to the value given in
     * parameter.
     *
     * @param fqfield the element on which the restriction is put
     * @param value the value the element must not be equals to.
     * @return {@code this} with the {@code neq} restriction added.
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker ne(String fqfield, Object value) {
        return ne(new ElementField(fqfield), value);
    }

    /**
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker ne(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new NotEquals(element, s));
        return this;
    }

    /**
     * Extension not equals.
     *
     * Restrict search to wikitties that do not get the extension given in
     * parameter.
     *
     * @param extensionName the extension that the wikitties must not have.
     * @return {@code this} with the {@code extneq} restriction added.
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker extne(String extensionName) {
        return ne(Element.EXTENSION, extensionName);
    }

    /**
     * Id not equals.
     *
     * Restrict search to wikitties that do not have the id given in parameter.
     *
     * @param idOrWikitty the id the wikitties must not have.
     * @return {@code this} with the {@code idne} restriction added.
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker idne(Object idOrWikitty) {
        return ne(Element.ID, idOrWikitty);
    }

    /**
     * @see {@link Greater}
     */
    public WikittyQueryMaker gt(Element element) {
        addCondition(new Greater(element));
        return this;
    }

    /**
     * Greater than.
     *
     * Search if an element value is greater than the parameter.
     *
     * @param fqfield the element on which the restriction is put
     * @param value the value to be compared to
     * @return {@code this} with the {@code gt} restriction added.
     * @see {@link Greater}
     */
    public WikittyQueryMaker gt(String fqfield, Object value) {
        return gt(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Greater}
     */
    public WikittyQueryMaker gt(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new Greater(element, s));
        return this;
    }

    /**
     * @see {@link GreaterOrEquals}
     */
    public WikittyQueryMaker ge(Element element) {
        addCondition(new GreaterOrEquals(element));
        return this;
    }

    /**
     * Greater than or equals.
     *
     * Search if an element value is greater than or equals to the parameter.
     *
     * @param fqfield the field on which the search is made
     * @param value the value to be compared to
     * @return {@code this} with the {@code ge} restriction added.
     * @see {@link GreaterOrEquals}
     */
    public WikittyQueryMaker ge(String fqfield, Object value) {
        return ge(new ElementField(fqfield), value);
    }

    /**
     * @see {@link GreaterOrEquals}
     */
    public WikittyQueryMaker ge(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new GreaterOrEquals(element, s));
        return this;
    }

    /**
     * @see {@link Less}
     */
    public WikittyQueryMaker lt(Element element) {
        addCondition(new Less(element));
        return this;
    }

    /**
     * Less than.
     *
     * Search if an element value is less than the parameter.
     *
     * @param fqfield the element on which the restriction is put
     * @param value the value to be compared to
     * @return {@code this} with the {@code lt} restriction added.
     * @see {@link Less}
     */
    public WikittyQueryMaker lt(String fqfield, Object value) {
        return lt(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Less}
     */
    public WikittyQueryMaker lt(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new Less(element, s));
        return this;
    }

    /**
     * @see {@link LessOrEquals}
     */
    public WikittyQueryMaker le(Element element) {
        addCondition(new LessOrEquals(element));
        return this;
    }

    /**
     * Less than or equals.
     *
     * Search if an element value is less than or equals to the parameter.
     *
     * @param fqfield the element on which the restriction is put.
     * @param value the value to be compared to.
     * @return {@code this} with the {@code le} restriction added.
     * @see {@link LessOrEquals}
     */
    public WikittyQueryMaker le(String fqfield, Object value) {
        return le(new ElementField(fqfield), value);
    }

    /**
     * @see {@link LessOrEquals}
     */
    public WikittyQueryMaker le(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new LessOrEquals(element, s));
        return this;
    }

    /**
     * @see {@link Between}
     */
    public WikittyQueryMaker bw(Element element) {
        addCondition(new Between(element));
        return this;
    }

    /**
     * Between.
     *
     * Restrict search so that the element value is between the lower and upper
     * values (it can also be equals).
     *
     * @param fqfield the element on which the restriction is put.
     * @param lowerValue the lower bound.
     * @param upperValue the upper bound.
     * @return {@code this} with the {@code le} restriction added.
     * @see {@link Between}
     */
    public WikittyQueryMaker bw(String fqfield, Object lowerValue, Object upperValue) {
        return bw(new ElementField(fqfield), lowerValue, upperValue);
    }

    /**
     * @see {@link Between}
     */
    public WikittyQueryMaker bw(Element element, Object lowerValue, Object upperValue) {
        ConditionValue min = convertToConditionValue(lowerValue);
        ConditionValue max = convertToConditionValue(upperValue);
        addCondition(new Between(element, min, max));
        return this;
    }

    /**
     * Starts with.
     *
     * Search if an element starts with the value in parameter.
     *
     * @param fqfield the element on which the restriction is put.
     * @param value the value the element must start with.
     * @return {@code this} with the {@code sw} restriction added.
     * @see {@link Equals}
     */
    public WikittyQueryMaker sw(String fqfield, String value) {
        return sw(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Equals}
     */
    public WikittyQueryMaker sw(Element element, String value) {
        addCondition(new Equals(element, value + "*"));
        return this;
    }

    /**
     * Not starts with.
     *
     * Search if an element does not starts with the value in parameter.
     *
     * @param fqfield the element on which the restriction is put.
     * @param value the value the element must not start with.
     * @return {@code this} with the {@code nsw} restriction added.
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker notsw(String fqfield, String value) {
        return notsw(new ElementField(fqfield), value);
    }

    /**
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker notsw(Element element, String value) {
        addCondition(new NotEquals(element, value + "*"));
        return this;
    }

    /**
     * Ends with.
     *
     * Search if an element ends with the value in parameter.
     *
     * @param fqfield the element on which the restriction is put
     * @param value the value the element must ends with.
     * @return {@code this} with the {@code ew} restriction added.
     * @see {@link Equals}
     */
    public WikittyQueryMaker ew(String fqfield, Object value) {
        return ew(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Equals}
     */
    public WikittyQueryMaker ew(Element element, Object value) {
        addCondition(new Equals(element, "*" + value));
        return this;
    }

    /**
     * Not ends with.
     *
     * Search if an element does not ends with the value in parameter.
     *
     * @param fqfield the element on which the restriction is put
     * @param value the value the element must not ends with.
     * @return {@code this} with the {@code notew} restriction added.
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker notew(String fqfield, Object value) {
        return notew(new ElementField(fqfield), value);
    }

    /**
     * @see {@link NotEquals}
     */
    public WikittyQueryMaker notew(Element element, Object value) {
        addCondition(new NotEquals(element, "*" + value));
        return this;
    }

    /**
     * Keyword.
     *
     * Search if the value in parameter is present in any field of any
     * extension.
     *
     * @param value the value to find.
     * @return {@code this} with the {@code keyword} restriction added.
     * @see {@link Keyword}
     */
    public WikittyQueryMaker keyword() {
        addCondition(new Keyword());
        return this;
    }

    /**
     * Keyword.
     *
     * Search if the value in parameter is present in any field of any
     * extension.
     *
     * @param value the value to find.
     * @return {@code this} with the {@code keyword} restriction added.
     * @see {@link Keyword}
     */
    public WikittyQueryMaker keyword(Object value) {
        ConditionValue s = convertToConditionValue(value);
        addCondition(new Keyword().addCondition(s));
        return this;
    }

    /**
     * Is null.
     *
     * Check that a field is null.
     *
     * @param fqfield the field that must be null.
     * @return {@code this} with the {@code isNull} restriction added.
     * @see {@link Null}
     */
    public WikittyQueryMaker isNull(String fqfield) {
        return isNull(new ElementField(fqfield));
    }

    /**
     * @see {@link Null}
     */
    public WikittyQueryMaker isNull(Element element) {
        addCondition(new Null(element));
        return this;
    }

    /**
     * Is not null.
     *
     * Check that a field is not null.
     *
     * @param fqfield the field that must not be null.
     * @return {@code this} with the {@code isNotNull} restriction added.
     * @see {@link NotNull}
     */
    public WikittyQueryMaker isNotNull(String fqfield) {
        return isNotNull(new ElementField(fqfield));
    }

    /**
     * @see {@link NotNull}
     */
    public WikittyQueryMaker isNotNull(Element element) {
        addCondition(new NotNull(element));
        return this;
    }

    /**
     * False.
     *
     * Add a restriction that always return false.
     *
     * @return {@code this} with the {@code rFalse} restriction added.
     * @see {@link False}
     */
    public WikittyQueryMaker rFalse() {
        addCondition(new False());
        return this;
    }

    /**
     * True.
     *
     * Add a restriction that always return true.
     *
     * @return {@code this} with the {@code rTrue} restriction added.
     * @see {@link True}
     */
    public WikittyQueryMaker rTrue() {
        addCondition(new True());
        return this;
    }

    /**
     * @see {@link Like}
     */
    public WikittyQueryMaker like(Element element) {
        addCondition(new Like(element));
        return this;
    }

    /**
     * Like.
     *
     * Check that a string is present in a field. For example "tric" is present
     * in "Restriction".
     *
     * @param fqfield the element on which the restriction is put
     * @param value
     * @return {@code this}
     * @see {@link Like}
     */
    public WikittyQueryMaker like(String fqfield, Object value) {
        return like(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Like}
     */
    public WikittyQueryMaker like(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        Like c = new Like(element, s);
        addCondition(c);
        return this;
    }

    /**
     * @see {@link Unlike}
     */
    public WikittyQueryMaker unlike(Element element) {
        addCondition(new Unlike(element));
        return this;
    }

    /**
     * Unlike.
     *
     * @param fqfield the element on which the restriction is put
     * @param value
     * @param searchAs
     * @return {@code this}
     * @see {@link Unlike}
     */
    public WikittyQueryMaker unlike(String fqfield, Object value) {
        return unlike(new ElementField(fqfield), value);
    }

    /**
     * @see {@link Unlike}
     */
    public WikittyQueryMaker unlike(Element element, Object value) {
        ConditionValue s = convertToConditionValue(value);
        Unlike c = new Unlike(element, s);
        addCondition(c);
        return this;
    }

    /**
     * Not (sub query). To close this sub query you must used {@link #close()}
     * <li>ex: WikittyQueryMaker().not().rTrue().close().and().rTrue().rFalse().close().or().rTrue().rFalse().close();
     *
     * @see {@link Not}
     */
    public WikittyQueryMaker not() {
        Condition child = new Not();
        addCondition(child);
        return this;
    }

    /**
     * Or (sub query). To close this sub query you must used {@link #close()}
     * <li>ex: WikittyQueryMaker().not().rTrue().close().and().rTrue().rFalse().close().or().rTrue().rFalse().close();
     *
     * @see {@link Or}
     */
    public WikittyQueryMaker or() {
        Condition child = new Or();
        addCondition(child);
        return this;
    }

    /**
     * And (sub query). To close this sub query you must used {@link #close()}
     * <li>ex: WikittyQueryMaker().not().rTrue().close().and().rTrue().rFalse().close().or().rTrue().rFalse().close();
     *
     * @see {@link And}
     */
    public WikittyQueryMaker and() {
        Condition child = new And();
        addCondition(child);

        return this;
    }

    /**
     * Add {@link Select}, this condition must be first or
     * @param element le champs dont il faut extraire les donnees
     * @return {@code this}
     * @see {@link Select}
     */
    public WikittyQueryMaker select(String element) {
        return select(new ElementField(element));
    }

    /**
     * Add {@link Select}, this condition must be first or
     * @param element le champs dont il faut extraire les donnees
     * @return {@code this}
     * @see {@link Select}
     */
    public WikittyQueryMaker select(String element, Aggregate aggregate) {
        return select(new ElementField(element), aggregate);
    }

    /*
     * @see {@link Select}
     */
    public WikittyQueryMaker select(Element element) {
        return select(element, null);
    }

    /*
     * @see {@link Select}
     */
    public WikittyQueryMaker select(Element element, Aggregate aggregate) {
        Condition child = new Select(element, aggregate);
        addCondition(child);
        return this;
    }

//    /**
//     * Add {@link In} to allow search on association (like sql join).
//     * To close this sub query you must used {@link #close()}
//     * @param foreignFieldName association fieldName
//     * @return {@code this}
//     * @see {@link In}
//     */
//    public WikittyQueryMaker in(String foreignFieldName) {
//        return in(new ElementField(foreignFieldName));
//    }
//
//    /*
//     * @see {@link In}
//     */
//    public WikittyQueryMaker in(Element element) {
//        Condition child = new In(element);
//        addCondition(child);
//        return this;
//    }
//
    /**
     * Close last non terminal condition (or, and, not, in).
     * <li>ex: WikittyQueryMaker().not().rTrue().close().and().rTrue().rFalse().close().or().rTrue().rFalse().close();
     * @return
     */
    public WikittyQueryMaker close() {
        getOpenStack().pop(); // on en ferme 1 obligatoirement
        // on cherche a en fermer plus
        closeIfNecessary();
        return this;
    }

    /**
     * Ferme la construction de la condition et la met en place dans la
     * WikittyQuery qui sera retournee
     * 
     * @return un objet WikittyQuery soit un nouveau soit celui passe dans le 
     * constructeur
     */
    public WikittyQuery end() {
        WikittyQuery result = getQuery();
        result.setCondition(getCondition());
        // on interdit les modifications futur
        openStack = null;
        
        return result;
    }
}
