/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2012 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.query;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.query.conditions.Aggregate;
import org.nuiton.wikitty.query.conditions.And;
import org.nuiton.wikitty.query.conditions.Between;
import org.nuiton.wikitty.query.conditions.Condition;
import org.nuiton.wikitty.query.conditions.ConditionValue;
import org.nuiton.wikitty.query.conditions.ConditionValueString;
import org.nuiton.wikitty.query.conditions.ContainsAll;
import org.nuiton.wikitty.query.conditions.ContainsOne;
import org.nuiton.wikitty.query.conditions.Element;
import org.nuiton.wikitty.query.conditions.ElementField;
import org.nuiton.wikitty.query.conditions.Equals;
import org.nuiton.wikitty.query.conditions.False;
import org.nuiton.wikitty.query.conditions.Greater;
import org.nuiton.wikitty.query.conditions.GreaterOrEquals;
import org.nuiton.wikitty.query.conditions.Keyword;
import org.nuiton.wikitty.query.conditions.Less;
import org.nuiton.wikitty.query.conditions.LessOrEquals;
import org.nuiton.wikitty.query.conditions.Like;
import org.nuiton.wikitty.query.conditions.Not;
import org.nuiton.wikitty.query.conditions.NotEquals;
import org.nuiton.wikitty.query.conditions.Or;
import org.nuiton.wikitty.query.conditions.NotNull;
import org.nuiton.wikitty.query.conditions.Null;
import org.nuiton.wikitty.query.conditions.Select;
import org.nuiton.wikitty.query.conditions.True;
import org.nuiton.wikitty.query.conditions.Unlike;
import org.parboiled.BaseParser;
import org.parboiled.Context;
import org.parboiled.Parboiled;
import org.parboiled.Rule;
import org.parboiled.annotations.BuildParseTree;
import org.parboiled.errors.ErrorUtils;
import org.parboiled.parserunners.RecoveringParseRunner;
import org.parboiled.parserunners.ReportingParseRunner;
import org.parboiled.parserunners.TracingParseRunner;
import org.parboiled.support.ParseTreeUtils;
import org.parboiled.support.ParsingResult;
import org.parboiled.support.Var;

/**
 * Cette classe permet d'interpreter une requete faite textuellement en la
 * convertisant en sa representation objet.
 *
 * Pour plus d'information reportez-vous à la
 * <a href="http://maven-site.nuiton.org/wikitty/user/query.html">documentation</a>
 *
 * @author poussin
 * @version $Revision$
 * @since 3.3
 *
 * Last update: $Date$
 * by : $Author$
 */
@BuildParseTree
public class WikittyQueryParser extends BaseParser<Object> {

    public static final String AVG = "AVG";
    public static final String COUNT = "COUNT";
    public static final String MAX = "MAX";
    public static final String MIN = "MIN";
    public static final String SUM = "SUM";
    public static final String SELECT = "SELECT";
    public static final String WHERE = "WHERE";
    public static final String IN = "IN";
    public static final String LITERAL_CLOSE = "\"";
    public static final String LITERAL_OPEN = "\"";
    public static final String NULL = "NULL";
    public static final String TO = "TO";
    public static final String FALSE = "FALSE";
    public static final String TRUE = "TRUE";
    public static final String UNLIKE = "UNLIKE";
    public static final String AND = "AND";
    public static final String COMMA = ",";
    public static final String CURLY_BRACKET_CLOSE = "}";
    public static final String CURLY_BRACKET_OPEN = "{";
    public static final String EQUALS = "=";
    public static final String GREATER = ">";
    public static final String GREATER_OR_EQUALS = ">=";
    public static final String LESS = "<";
    public static final String LESS_OR_EQUALS = "<=";
    public static final String LIKE = "LIKE";
    public static final String NOT = "NOT";
    public static final String NOT_EQUALS = "!=";
    public static final String OR = "OR";
    public static final String BRACKET_CLOSE = ")";
    public static final String BRACKET_OPEN = "(";
    public static final String SQUARE_BRACKET_CLOSE = "]";
    public static final String SQUARE_BRACKET_OPEN = "[";

    public Rule icEXTENSION = IgnoreCase(Element.EXTENSION.getValue());
    public Rule icID = IgnoreCase(Element.ID.getValue());
    public Rule icNOT = IgnoreCase(NOT);
    public Rule icAND = IgnoreCase(AND);
    public Rule icOR = IgnoreCase(OR);
    public Rule icAVG = IgnoreCase(AVG);
    public Rule icCOUNT = IgnoreCase(COUNT);
    public Rule icMAX = IgnoreCase(MAX);
    public Rule icMIN = IgnoreCase(MIN);
    public Rule icSUM = IgnoreCase(SUM);
    public Rule icSELECT = IgnoreCase(SELECT);
    public Rule icWHERE = IgnoreCase(WHERE);
    public Rule icIN = IgnoreCase(IN);
    public Rule icTO = IgnoreCase(TO);
    public Rule icFALSE = IgnoreCase(FALSE);
    public Rule icTRUE = IgnoreCase(TRUE);
    public Rule icLIKE = IgnoreCase(LIKE);
    public Rule icUNLIKE = IgnoreCase(UNLIKE);
    public Rule icNULL = IgnoreCase(NULL);

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyQueryParser.class);

    protected Map<String, String> alias = new LinkedHashMap<String, String>();

    public WikittyQueryParser() {
    }

    boolean debug(String text, Context context) {
        System.out.println("DEBUG("+context.getCurrentIndex()+"):" + text + "  nodes:" +context.getSubNodes());
        return true;
    }

    public Map<String, String> getAlias() {
        return alias;
    }

    public WikittyQueryParser setAlias(Map<String, String> alias) {
        this.alias = alias;
        return this;
    }

    public WikittyQueryParser addAlias(String aliasName, String aliasValue) {
        alias.put(aliasName, aliasValue);
        return this;
    }

    /**
     * Parse query and use alias added with {@link #addAlias} or {@link #setAlias}
     *
     * @param queryString query to parse
     * @return
     */
    public WikittyQuery parseQuery(String queryString) {
        WikittyQuery result = parse(queryString, alias);
        return result;
    }

    /**
     * Parse query without alias
     *
     * @param queryString query to parse
     * @return
     */
    static public WikittyQuery parse(String queryString) {
        WikittyQuery result = parse(queryString, null);
        return result;
    }

    /**
     * Parse query and use alias in argument
     *
     * @param queryString query to parse
     * @param alias alias to used to change query
     * @return
     */
    static public WikittyQuery parse(String queryString, Map<String, String> alias) {
        if (alias != null) {
            String queryStringInit = queryString;
            // first replace alias in queryString
            for (Map.Entry<String, String> a : alias.entrySet()) {
                queryString = queryString.replaceAll(a.getKey(), a.getValue());
            }
            if (log.isDebugEnabled()) {
                log.debug(String.format("QueryString \n'%s' become after alias \n'%s'\naliases are %s",
                        queryStringInit, queryString, alias));
            }
        }

        WikittyQueryParser parser = Parboiled.createParser(WikittyQueryParser.class);

       ParsingResult<?> result = new ReportingParseRunner(parser.start()).run(queryString);
//        ParsingResult<?> result = new TracingParseRunner(parser.start()).run(queryString);
//        ParsingResult<?> result = new RecoveringParseRunner(parser.start()).run(queryString);

        if (result.hasErrors() || !result.matched) {
            System.out.println("\nParse Errors:\n" + ErrorUtils.printParseErrors(result));
        }

        if (log.isDebugEnabled()) {
            log.debug("\nParse Tree:\n" + ParseTreeUtils.printNodeTree(result) + '\n');
        }
//        System.out.println("\nParse Tree:\n" + ParseTreeUtils.printNodeTree(result) + '\n');
        
        Condition c = (Condition)result.resultValue;

        WikittyQuery query = new WikittyQuery(c);
        return query;
    }

    /**
     * can be field, extension name or id element
     * @param v
     * @return
     */
    protected Element toElement(String v) {
        Element result = Element.get(v);
        return result;
    }

    /**
     * Remove quote at beginning and ending of String in parameter if necessary
     *
     * <li>"toto" return toto
     * <li>"toto return "toto
     * <li> toto return toto"
     * <li> to"to return to"to
     * 
     * @param s
     * @return
     */
    protected String removeQuote(String s) {
        String result = s;
        if (StringUtils.startsWith(s, LITERAL_OPEN)
                && StringUtils.startsWith(s, LITERAL_CLOSE)) {
            result = StringUtils.substring(s, 1, -1);
        }
        return result;
    }

   Rule start() {
       return Sequence(or(), EOI);
   }

   Rule or() {
        return Sequence(and(), ZeroOrMore(space(), icOR, space(), and(),
                push(new Or((Condition)pop(1), (Condition)pop()))));
    }

    Rule and() {
        return Sequence(term(), ZeroOrMore(
                // when no AND or OR is used, AND is default
                // don't change order of FirstOf, this order is important
                FirstOf(Sequence(space(), icAND, space()), Sequence(space(), TestNot(icOR))),
                term(),
                push(new And((Condition)pop(1), (Condition)pop()))));
    }

    Rule term() {
        return FirstOf(condition(), Parens());
    }

    Rule Parens() {
        return Sequence(space(), BRACKET_OPEN, space(), or(), space(), BRACKET_CLOSE, space());
    }

    Rule condition() {
        return FirstOf(
                not(), isNull(), isNotNull(), select(),
                eq(), neq(), less(), lesseq(), greater(), greatereq(), like(), notlike(),
                between(), containsAll(), containsOne(),
                rTrue(), rFalse(), keyword()
                );
    }

    Rule not() {
        return Sequence(space(), icNOT, space(), term(),
                push(new Not((Condition)pop())));
    }

    /**
     * gere eq, startsWith, endsWith, isNull
     * @return
     */
    Rule isNull() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), icNULL,
                push(new Null(toElement(pop().toString()))));
    }

    /**
     * gere eq, isNull
     * @return
     */
    Rule isNotNull() {
        return Sequence(field(), push(match()), space(), NOT_EQUALS, space(), icNULL,
                push(new NotNull(toElement(pop().toString()))));
    }

    /**
     * gere eq, startsWith, endsWith, isNull
     * @return
     */
    Rule eq() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), value(),
                push(new Equals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }

    /**
     * gere eq, isNull
     * @return
     */
    Rule neq() {
        return Sequence(field(), push(match()), space(), NOT_EQUALS, space(), value(),
                push(new NotEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }

    Rule less() {
        return Sequence(field(), push(match()), space(), LESS, space(), value(),
                push(new Less(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule lesseq() {
        return Sequence(field(), push(match()), space(), LESS_OR_EQUALS, space(), value(),
                push(new LessOrEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule greater() {
        return Sequence(field(), push(match()), space(), GREATER, space(), value(),
                push(new Greater(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule greatereq() {
        return Sequence(field(), push(match()), space(), GREATER_OR_EQUALS, space(), value(),
                push(new GreaterOrEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule like() {
        return Sequence(field(), push(match()), space(), icLIKE, space(), value(),
                push(new Like(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule notlike() {
        return Sequence(field(), push(match()), space(), icUNLIKE, space(), value(),
                push(new Unlike(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule between() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), SQUARE_BRACKET_OPEN, space(),
                value(), space(), icTO, space(),
                value(), space(), SQUARE_BRACKET_CLOSE,
                push(new Between(toElement(pop(2).toString()), (ConditionValue)pop(1), (ConditionValue)pop())));
    }
    Rule containsAll() {
        Var<List<ConditionValue>> elems = new Var<List<ConditionValue>>(new LinkedList<ConditionValue>());
        return Sequence(field(), push(match()), space(), EQUALS, space(), SQUARE_BRACKET_OPEN, space(),
                value(), elems.get().add((ConditionValue)pop()), space(), ZeroOrMore(COMMA, space(),
                value(), elems.get().add((ConditionValue)pop()), space()), SQUARE_BRACKET_CLOSE,
                push(new ContainsAll(toElement(pop().toString()), elems.get())));
    }
    Rule containsOne() {
        Var<List<ConditionValue>> elems = new Var<List<ConditionValue>>(new LinkedList<ConditionValue>());
        return Sequence(field(), push(match()), space(), EQUALS, space(), CURLY_BRACKET_OPEN, space(),
                value(), elems.get().add((ConditionValue)pop()), space(), ZeroOrMore(COMMA, space(),
                value(), elems.get().add((ConditionValue)pop()), space()), CURLY_BRACKET_CLOSE,
                push(new ContainsOne(toElement(pop().toString()), elems.get())));
    }
    Rule select() {
        Var<Aggregate> aggregate = new Var<Aggregate>();
        return Sequence(icSELECT, space(), Optional(aggregate(aggregate)), space(), field(), push(match()), space(), icWHERE, space(), term(),
                push(new Select(toElement(pop(1).toString()), aggregate.get(), (Condition)pop())));
    }
    Rule aggregate(Var<Aggregate> aggregate) {
        return FirstOf(avg(aggregate), count(aggregate), max(aggregate), min(aggregate), sum(aggregate));
    }

    Rule avg(Var<Aggregate> aggregate) {
        return Sequence(icAVG, aggregate.set(Aggregate.AVG));
    }

    Rule count(Var<Aggregate> aggregate) {
        return Sequence(icCOUNT, aggregate.set(Aggregate.COUNT));
    }

    Rule max(Var<Aggregate> aggregate) {
        return Sequence(icMAX, aggregate.set(Aggregate.MAX));
    }

    Rule min(Var<Aggregate> aggregate) {
        return Sequence(icMIN, aggregate.set(Aggregate.MIN));
    }

    Rule sum(Var<Aggregate> aggregate) {
        return Sequence(icSUM, aggregate.set(Aggregate.SUM));
    }

    Rule keyword() {
        return Sequence(value(), push(new Keyword().addCondition((ConditionValue)pop())));
    }

    Rule field() {
        return OneOrMore(FirstOf(CharRange('0', '9'), CharRange('a', 'z'), CharRange('A', 'Z'), AnyOf("_.*")));
    }

    Rule rTrue() {
        return Sequence(icTRUE, push(new True()));
    }

    Rule rFalse() {
        return Sequence(icFALSE, push(new False()));
    }

    Rule value() {
        return FirstOf(select(), valueText());
    }

    Rule valueText() {
        return Sequence(FirstOf(field(), StringLiteral()),
                push(new ConditionValueString(removeQuote(match()))));
    }

    Rule StringLiteral() {
        return Sequence(
                LITERAL_OPEN,
                ZeroOrMore(
                        FirstOf(
                                Escape(),
                                Sequence(TestNot(AnyOf("\r\n\\"+LITERAL_OPEN+LITERAL_CLOSE)), ANY)
                        )
                ).suppressSubnodes(),
                LITERAL_CLOSE
                );
    }

    Rule Escape() {
        return Sequence('\\', FirstOf(AnyOf("btnfr\'\\"+LITERAL_OPEN+LITERAL_CLOSE), OctalEscape(), UnicodeEscape()));
    }

    Rule OctalEscape() {
        return FirstOf(
                Sequence(CharRange('0', '3'), CharRange('0', '7'), CharRange('0', '7')),
                Sequence(CharRange('0', '7'), CharRange('0', '7')),
                CharRange('0', '7')
        );
    }

    Rule UnicodeEscape() {
        return Sequence(OneOrMore('u'), HexDigit(), HexDigit(), HexDigit(), HexDigit());
    }

    Rule HexDigit() {
        return FirstOf(CharRange('a', 'f'), CharRange('A', 'F'), CharRange('0', '9'));
    }

    Rule space() {
        return ZeroOrMore(AnyOf(" \t\f"));
    }

}
