/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyService.java 648 2010-12-22 17:59:58Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.3/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyService.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.services.WikittyListener;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.nuiton.wikitty.entities.WikittyTreeNode;

/**
 * Wikitty service.
 *
 * The main implementation for this interface is {@link WikittyServiceStorage}.
 * It can be used alone but this implementation doesn't deal with all 
 * stuffs described in this interface. Thus, other functionalities are added
 * to the implementation through objects that decorate WikittyServiceStorage :
 *
 * <dl>
 *   <dt>{@link WikittyServiceCached}</dt>
 *     <dd>add a cache for wikitties</dd>
 *   <dt>{@link WikittyServiceSecurity}</dt>
 *     <dd>add user authentication support and right management</dd>
 *   <dt>{@link WikittyServiceNotifier}</dt>
 *     <dd>add notifications between client of the same wikitty service</dd>
 * </dl>
 * 
 * @author poussin
 * @version $Revision: 648 $
 *
 * Last update: $Date: 2010-12-22 18:59:58 +0100 (mer., 22 déc. 2010) $
 * by : $Author: bpoussin $
 */
public interface WikittyService {

    /*
     * listeners 
     */
    
    /** Event listener type. */
    public enum ServiceListenerType {
        ALL, LOCAL, REMOTE
    }

    /**
     * Add new wikitty service listener.
     * 
     * Warning, {@code listener} is referenced as WeakReference, but sure to
     * another reference to work.
     * 
     * @param listener listener to add
     * @param type type of event to listen
     * 
     * @see ServiceListenerType
     */
    public void addWikittyServiceListener(WikittyListener listener, ServiceListenerType type);
    
    /**
     * Remove wikitty service listener.
     * 
     * Warning, {@code listener} is referenced as WeakReference, but sure to
     * another reference to work.
     * 
     * @param listener listener to remove
     * @param type type of event to listen
     * 
     * @see ServiceListenerType
     */
    public void removeWikittyServiceListener(WikittyListener listener, ServiceListenerType type);

    /*
     * security
     */

    /**
     * Authenticate someone on WikittyService. securityToken returned must be
     * used to call others methods
     *
     * @param login can be application specifique login, but best practice is
     * to use email user
     * @param password
     * @return return token securityToken
     */
    public String login(String login, String password);

    /**
     * Unanthenticate someone by disabled securityToken
     * 
     * @param securityToken security token previously returned by login. If
     * securityToken is not valid, this method do nothing
     */
    public void logout(String securityToken);

    /*
     * Storage
     */

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     * 
     * @param securityToken security token
     */
    public WikittyEvent clear(String securityToken);

    /**
     * Verifie si l'utilisateur lie au securityToken a le droit d'ecrire
     * le Wikitty passe en argument.
     *
     * On ne peut pas passer seulement l'id du wikitty en parametre car de
     * nouvelles extensions ont peut lui etre ajouter depuis la derniere
     * sauvegarde
     *
     * @param securityToken le token de securite qui permet de retrouver
     * l'utilisateur et ainsi verifier les droits
     * @param wikitty le wikitty a sauver
     * @return vrai si l'utilisateur peut sauver l'objet
     */
    public boolean canWrite(String securityToken, Wikitty wikitty);

    /**
     * Verifie que l'utilisateur associe au securityToken peut supprimer le
     * wikitty dont on passe l'identifiant. Seul le propriétaire de l'objet
     * ou un admin peut supprimer un objet.
     *
     * Si l'id de l'objet est invalide, la methode retourne true, car la
     * suppression d'un id invalide ne fait rien
     *
     * @param securityToken security token
     * @param wikittyId wikitty id
     * @return vrai le la suppression ne posera pas de probleme.
     */
    public boolean canDelete(String securityToken, String wikittyId);

    /**
     * Un utilisateur peu lire un objet, s'il est Reader ou a defaut:
     * - owner
     * - AppAdmin
     * - Admin
     * - Writer
     *
     * @param securityToken security token
     * @param wikittyId wikitty id
     * @return vrai si l'utilisateur peut lire l'obbjet
     */
    public boolean canRead(String securityToken, String wikittyId);

    /**
     *  true if wikitty with id exists, even wikitty is deleted
     */
    public boolean exists(String securityToken, String wikittyId);

    /**
     * true if wikitty is deleted, throw an exception if id don't exist
     * @param securityToken
     * @param wikittyId
     * @return
     */
    public boolean isDeleted(String securityToken, String wikittyId);

    /**
     * Replay all events in argument on this WikittyService
     * 
     * @param securityToken security token
     * @param events event to replay
     * @param force for to not change wikitty version (use version in wikitty
     * present in event)
     * @return new event that represent all event passed in argument.
     * if arguement have: store, store, delete, clear, store. Return event
     * resume all by only one clear + store, because all action before clear is
     * not necessary. Similarly for store + delete for the same object.
     * (note: perhaps this broke history, when history are implanted and
     * two serveur must have same history ?)
     */
    public WikittyEvent replay(
            String securityToken, List<WikittyEvent> events, boolean force);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param wikitties list of wikitty to be persisted
     * @param force boolean force non version version increment on saved wikitty
     *              or force version on wikitty creation (version 0.0)
     * @return update response
     */
    public WikittyEvent store(
            String securityToken, Collection<Wikitty> wikitties, boolean force);

    /**
     * Return all extension id (ex: "extName[version])").
     * 
     * @param securityToken security token
     * @return extension ids list
     */
    public List<String> getAllExtensionIds(String securityToken);
    
    /**
     * Return all extension id (ex: "extName[version])") where extensionName is
     * required.
     * 
     * @param securityToken security token
     * @param extensionName extension name
     * @return extension id list
     */
    public List<String> getAllExtensionsRequires(String securityToken, String extensionName);

    /**
     * Manage Update and creation
     *
     * @param securityToken security token
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public WikittyEvent storeExtension(
            String securityToken, Collection<WikittyExtension> exts);

   /**
     * Delete all extension if id exists and no wikitty used this extension.
     * extension name must be just the name (extName)
     *
     * @param securityToken security token
     * @param ids extension's ids to remove
     */
    public WikittyEvent deleteExtension(
            String securityToken, Collection<String> extNames);

    /**
     * Load extension from id. Id is 'name[version]'.
     *
     * @param securityToken security token
     * @param extensionId
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(
            String securityToken, String extensionId);

    /**
     * Search extension with name in last version.
     * 
     * @param securityToken security token
     * @param name extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name);

    /**
     * Restore wikitty
     * 
     * @param securityToken security token
     * @param id list of wikitty ids to restore
     * @return list of corresponding wikitty, if one id is not valid (no object
     * or deleted or no authorisation) this id return null and result list can
     * have null elements
     */
   public List<Wikitty> restore(String securityToken, List<String> id);

    /**
     * Delete all object if id exists.
     * 
     * @param securityToken security token
     * @param ids object's ids to remove
     */
    public WikittyEvent delete(String securityToken, Collection<String> ids);

    /**
     * 
     * @param securityToken security token
     * @param criteria
     * @return
     */
    public PagedResult<String> findAllByCriteria(
            String securityToken, Criteria criteria);

    /**
     * First lonely (or first one) wikitty object that match criteria, if no
     * wikitty found or first retrived is not authorized for the user return
     * null
     *
     * @param securityToken security token
     * @param criteria
     * @return wikitty id object or null
     */
    public String findByCriteria(String securityToken, Criteria criteria);

    /*
     * Classification
     * Most of classification purpose is handle by extension mechanisms
     */

    /**
     * Retrieve all wikitties children (recursively) of an other one
     * Wikitty reference by wikittyId MUST include the 'Node' extension
     *
     * @param securityToken security token
     * @param wikittyId
     * @return
     *
     * @deprecated cette methode retourne des wikitty or seul la methode restore
     * doit le faire (centralisation de la restoration). Il faut donc que cette
     * methode n'existe que sur le proxy et utilise les autres methodes
     */
    @Deprecated
    public WikittyTree restoreTree(String securityToken, String wikittyId);

    /**
     * Delete specified tree node and all sub nodes.
     * 
     * @param securityToken security token
     * @param treeNodeId tree node id to delete
     * @return delete wikitty ids
     */
    public WikittyEvent deleteTree(String securityToken, String treeNodeId);

    /**
     * Retrieve wikitty node with count. Wikitty reference by wikittyId MUST
     * include the 'WikittyTreeNode' extension.
     *
     * Count is number of attachment in subtree. If filter is not null only
     * attachments that satisfy filter are counted
     *
     * @param securityToken security token
     * @param wikittyId
     * @param filter
     * @return
     */
    public Map.Entry<String, Integer> restoreNode(
            String securityToken, String wikittyId, Criteria filter);

    /**
     * Retrieve all wikitties children (no recursively) with count of an other one
     * Wikitty reference by wikittyId MUST include the 'WikittyTreeNode' extension
     *
     * Count is number of attachment in subtree (recursively). If filter is not
     * null only attachments that satisfy filter are counted
     *
     * @param securityToken security token
     * @param wikittyId
     * @param filter 
     * @return
     */
    public Map<String, Integer> restoreChildren(
            String securityToken, String wikittyId, Criteria filter);


    /*
     * history
     */

    /**
     * Restore wikitty in specifique version.
     * Authorisation is checked on last version even for previous wikitty version
     * 
     * @param securityToken security token
     */
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version);

    /*
     * admin
     */

    /**
     * Synchronise search engine with wikitty storage engine, i.e. clear and
     * reindex all wikitties.
     * 
     * @param securityToken security token
     */
    public void syncSearchEngine(String securityToken);

}
