/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.AbstractMap.SimpleEntry;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.search.Search;

/**
 * WikittyService is main service
 *
 * @author poussin
 * @version $Revision: 357 $
 *
 * Last update: $Date: 2010-09-28 15:44:36 +0200 (mar., 28 sept. 2010) $
 * by : $Author: echatellier $
 */
public class WikittyServiceImpl implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceImpl.class);

    /** Default migration use to migrate a wikitty in last extension version */
    protected WikittyExtensionMigration defaultExtensionMigration =
            new WikittyExtensionMigrationRename();

    protected WikittySearchEngin searchEngin;
    protected WikittyExtensionStorage extensionStorage;
    protected WikittyStorage wikittyStorage;

    /**
     * Used by specific child
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceJDBC}
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceJPA}
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceHbase}
     */
    protected WikittyServiceImpl() {
        
    }

    public WikittyServiceImpl(WikittyExtensionStorage extensionStorage,
            WikittyStorage wikittyStorage,
            WikittySearchEngin searchEngin) {
        this.extensionStorage = extensionStorage;
        this.wikittyStorage = wikittyStorage;
        this.searchEngin = searchEngin;
    }

    protected WikittySearchEngin getSearchEngin() {
        return searchEngin;
    }

    protected WikittyExtensionStorage getExtensionStorage() {
        return extensionStorage;
    }

    protected WikittyStorage getWikittyStorage() {
        return wikittyStorage;
    }

    /*
     * @see org.nuiton.wikitty.WikittyService#addWikittyServiceListener(org.nuiton.wikitty.WikittyServiceListener, org.nuiton.wikitty.WikittyService.ServiceListenerType)
     */
    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        throw new UnsupportedOperationException("Can't add listener on " + WikittyServiceImpl.class.getName());
    }

    /*
     * @see org.nuiton.wikitty.WikittyService#removeWikittyServiceListener(org.nuiton.wikitty.WikittyServiceListener, org.nuiton.wikitty.WikittyService.ServiceListenerType)
     */
    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        throw new UnsupportedOperationException("Can't remove listener on " + WikittyServiceImpl.class.getName());
    }

    @Override
    public String login(String login, String password) {
        log.warn("login asked, but there is no security service");
        return null;
    }

    @Override
    public void logout(String securityToken) {
        log.warn("logout asked, but there is no security service");
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        return true;
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        return true;
    }

    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        return true;
    }

    protected void checkConstraint(Collection<Wikitty> wikitties) {
        for(Wikitty w : wikitties) {
            for(WikittyExtension ext : w.getExtensions()) {
                for (String fieldName : ext.getFieldNames()) {
                    FieldType type = ext.getFieldType(fieldName);
                    if (type.isNotNull()) {
                        if (null == w.getFieldAsObject(ext.getName(), fieldName)) {
                            throw new WikittyException(String.format(
                                    "Field %s must not be null", fieldName));
                        }
                    }
                }
            }
        }
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean force) {
        checkConstraint(wikitties);

        // update/store extension if necessary
        Set<WikittyExtension> allExtensions = new HashSet<WikittyExtension>();
        for (Wikitty w : wikitties) {
            // collect all extensions used by all wikitties
            allExtensions.addAll(w.getExtensions());
        }

        // try to commit command
        UpdateResponse extUpdate = getExtensionStorage().store(transaction, allExtensions);
        UpdateResponse wikUpdate = getWikittyStorage().store(transaction, wikitties, force);
        UpdateResponse indexUpdate = getSearchEngin().store(transaction, wikitties);

        UpdateResponse result = new UpdateResponse();
        // prepare update client response
        result.add(extUpdate);
        result.add(wikUpdate);
        result.add(indexUpdate);

        return result;
    }

    /**
     * Store and index wikitty object
     * @param wikitty
     */
    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        if (wikitty != null) {
            WikittyTransaction transaction = new WikittyTransaction();
            try {
                transaction.begin();

                List<Wikitty> wikitties = Arrays.asList(wikitty);
                UpdateResponse result = store(
                        securityToken, transaction, wikitties, false);
                
                transaction.commit();
                return result;
            } catch (WikittyException ex) {
                transaction.rollback();
                throw ex;
            } catch (Exception eee) {
                transaction.rollback();
                throw new WikittyException(eee);
            }
        } else {
            throw new WikittyException("You can't store null wikitty object");
        }
    }

    /**
     * Store and index wikitties object.
     * 
     * @param wikitties
     */
    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = store(
                    securityToken, transaction, wikitties, false);

            transaction.commit();
            return result;
        } catch (WikittyException ex) {
            transaction.rollback();
            throw ex;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Store and index wikitties object.
     * 
     * @param wikitties
     * @param force boolean force non version version increment on saved wikitty
     *              or force version on wikitty creation (version 0.0)
     */
    @Override
    public UpdateResponse store(String securityToken,
            Collection<Wikitty> wikitties, boolean force) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            UpdateResponse result = store(securityToken,
                    transaction, wikitties, force);

            transaction.commit();
            return result;
        } catch (WikittyException ex) {
            transaction.rollback();
            throw ex;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> result = getExtensionStorage().getAllExtensionIds(transaction);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> result = getExtensionStorage()
                    .getAllExtensionsRequires(transaction, extensionName);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        UpdateResponse result = getExtensionStorage().store(transaction, exts);
        return result;
    }

    /**
     * Save several extension.
     * 
     * @param exts
     */
    @Override
    public UpdateResponse storeExtension(String securityToken,
            Collection<WikittyExtension> exts) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = storeExtension(
                    securityToken, transaction, exts);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = storeExtension(
                    securityToken, transaction, Arrays.asList(ext));

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public WikittyExtension restoreExtension(
            String securityToken, WikittyTransaction transaction, String extensionId) {
        //split the id to ensure that version is normalized
        String name = WikittyExtension.computeName(extensionId);
        String version = WikittyExtension.computeVersion(extensionId);

        WikittyExtension result = getExtensionStorage().restore(transaction, name, version);
        return result;
    }

    /**
     * Load extension from id. Id is 'name[version]'.
     * 
     * @param extensionId extension id to restore
     * @return extension
     */
    @Override
    public WikittyExtension restoreExtension(String securityToken, String extensionId) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            WikittyExtension result = restoreExtension(
                    securityToken, transaction, extensionId);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }

    }

    public WikittyExtension restoreExtensionLastVersion(String securityToken,
            WikittyTransaction transaction, String name) {
        String version = getExtensionStorage().getLastVersion(transaction, name);
        if(version == null) {
            return null;
        }

        WikittyExtension result = getExtensionStorage().restore(transaction, name, version);
        return result;
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            WikittyExtension result = restoreExtensionLastVersion(
                    securityToken, transaction, name);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected Wikitty restore(String securityToken,
            WikittyTransaction transaction, String id) {
        if (!getWikittyStorage().exists(transaction, id)) {
            // object doesn't exist, we return null
            return null;
        }

        if (getWikittyStorage().isDeleted(transaction, id)) {
            // object deleted, we return null
            return null;
        }
        Wikitty result = getWikittyStorage().restore(transaction, id);
        if(result != null) {
            result = upgradeData(securityToken, transaction, result);
        }
        return result;
    }

    @Override
    public List<Wikitty> restore(String securityToken,
            WikittyTransaction transaction, List<String> ids) {
        List<Wikitty> result = new ArrayList<Wikitty>();
        for(String id : ids) {
            Wikitty w = restore(securityToken, transaction, id);
            if (w != null) {
                result.add(w);
            }
        }
        return result;
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<Wikitty> result = restore(securityToken, transaction, ids);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't restore wikitty collection", eee);
            }
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Wikitty restore(String securityToken, String id) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty result = restore(securityToken, transaction, id);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected Wikitty upgradeData(String securityToken,
            WikittyTransaction transaction, Wikitty wikitty) {
        Wikitty result = wikitty;

        Collection<WikittyExtension> extensions = wikitty.getExtensions();
        for (WikittyExtension extension : extensions) {
            String extensionName = extension.getName();
            log.debug("extensionName="  + extensionName);
            
            WikittyExtension currentExtension = extension;
            String currentExtensionVersion = currentExtension.getVersion();

            WikittyExtension lastExtension = restoreExtensionLastVersion(
                    securityToken, transaction, extensionName);
            String lastExtensionVersion = lastExtension.getVersion();
            log.debug("lastExtensionVersion="  + lastExtensionVersion);

            WikittyExtensionMigration migration = WikittyExtensionMigration.migrationRegistry.get(extensionName);
            if (migration == null) {
                migration = defaultExtensionMigration;
            }

            // Loop on between extension in wikitty and last version
            while(WikittyUtil.versionGreaterThan(lastExtensionVersion, currentExtensionVersion)) {

                // Get extension after the current version
                String nextExtensionVersion = WikittyUtil.incrementMajorRevision(currentExtensionVersion);
                String nextExtensionId = WikittyExtension.computeId(extensionName, nextExtensionVersion);
                WikittyExtension nextExtension = restoreExtension(
                        securityToken, transaction, nextExtensionId);

                log.debug("currentExtensionVersion="  + currentExtensionVersion);
                log.debug("nextExtensionVersion="  + nextExtensionVersion);

                // Test if extension is never use in this version
                if(nextExtension != null) {
                    result = migration.migrate(this, transaction, result,
                            currentExtension, nextExtension);
                    currentExtension = nextExtension;
                }

                // Follow
                currentExtensionVersion = nextExtensionVersion;
            }
        }
        
        return result;
    }

    protected void delete(String securityToken, WikittyTransaction transaction,
            Collection<String> ids) throws WikittyException {
        // work only on valid id
        Collection<Wikitty> storedWikitties = new ArrayList<Wikitty>();
        List<String> idList = new LinkedList<String>(ids);
        for (Iterator<String> i = idList.iterator(); i.hasNext();) {
            String id = i.next();
            // test if wikitty exists
            if (!getWikittyStorage().exists(transaction, id)) {
                // don't exist, remove this id in id list
                i.remove();
            }
            if (getWikittyStorage().isDeleted(transaction, id)) {
                // already deleted, remove this id in id list
                i.remove();
            }

            // Store node with have deleted node as parent
            Criteria criteria =
                    Search.query().eq(TreeNode.FQ_FIELD_TREENODE_PARENT, id).criteria();
            List<String> wikittyNodesId = findAllByCriteria(
                    securityToken, transaction, criteria).getAll();
            for (String wikittyNodeId : wikittyNodesId) {
                if(!ids.contains(wikittyNodeId)) {
                    Wikitty treeNode = restore(
                            securityToken, transaction, wikittyNodeId);
                    TreeNodeHelper.setParent(treeNode, null);
                    storedWikitties.add(treeNode);
                }
            }

            // Store node with have deleted child
            criteria = Search.query().eq(TreeNode.FQ_FIELD_TREENODE_ATTACHMENT, id).criteria();
            wikittyNodesId = findAllByCriteria(
                    securityToken, transaction, criteria).getAll();
            for (String wikittyNodeId : wikittyNodesId) {
                if(!ids.contains(wikittyNodeId)) {
                    Wikitty treeNode = restore(
                            securityToken, transaction, wikittyNodeId);
                    TreeNodeHelper.removeAttachment(treeNode, id);
                    storedWikitties.add(treeNode);
                }
            }
        }

        getWikittyStorage().delete(transaction, idList);
        getSearchEngin().delete(transaction, idList);

        store(securityToken, transaction, storedWikitties, false);
    }

    @Override
    public void delete(String securityToken, String id) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> ids = Arrays.asList(id);
            delete(securityToken, transaction, ids);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public void delete(String securityToken, Collection<String> ids){
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            delete(securityToken, transaction, ids);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    @Override
    public void clear(String securityToken) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            getSearchEngin().clear(transaction);
            getWikittyStorage().clear(transaction);
            getExtensionStorage().clear(transaction);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Assume that this PagedResult contains wikitty id as result and
     * return new PagedResult with Wikitty instance
     */
    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        PagedResult<String> result =
                getSearchEngin().findAllByCriteria(transaction, criteria);
        return result;
    }
    
    @Override
    public PagedResult<String> findAllByCriteria(
            String securityToken, Criteria criteria) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            PagedResult<String> result = findAllByCriteria(
                    securityToken, transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    public Wikitty findByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        criteria.setFirstIndex(0).setEndIndex(1);
        PagedResult<String> pages = findAllByCriteria(
                securityToken, transaction, criteria);

        Wikitty result = null;
        if (pages.size() > 0) {
            String id = pages.getFirst();
            result = restore(securityToken, transaction, id);
        }

        return result;
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty result = findByCriteria(
                    securityToken, transaction, criteria);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }
    
    @Override
    public void addLabel(String securityToken, String wikittyId, String label) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(securityToken, transaction, wikittyId);
            w.addExtension(LabelImpl.extensions);
            LabelImpl l = new LabelImpl(w);
            l.addLabels(label);
            store(securityToken, transaction, Arrays.asList(w), false);
            
            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public PagedResult<String> findAllByLabel(String securityToken,
            String label, int firstIndex, int endIndex) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            LabelImpl l = new LabelImpl();
            l.addLabels(label);
            Criteria criteria = Search.query(l.getWikitty()).criteria()
                    .setFirstIndex(firstIndex).setEndIndex(endIndex);
            PagedResult<String> result = findAllByCriteria(
                    securityToken, transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public Wikitty findByLabel(String securityToken, String label) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            LabelImpl l = new LabelImpl();
            l.addLabels(label);
            Criteria criteria = Search.query(l.getWikitty()).criteria();
            Wikitty result = findByCriteria(securityToken, transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public Set<String> findAllAppliedLabels(String securityToken, String wikittyId) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(securityToken, transaction, wikittyId);
            Set<String> result = LabelHelper.getLabels(w);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    protected Tree restoreTree(String securityToken,
            WikittyTransaction transaction, String wikittyId) {
        Wikitty w = restore(securityToken, transaction, wikittyId);
        if(w == null) {
            return null;
        }

        if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
            throw new WikittyException(String.format(
                    "Wikitty '%s' do not handle extension %s",
                    wikittyId, TreeNode.EXT_TREENODE ));
        }
        Tree tree = new Tree();
        TreeNode node = new TreeNodeImpl(w);
        tree.setNode(node);

        TreeNodeImpl exempleNode = new TreeNodeImpl();
        exempleNode.setParent(wikittyId);

        Criteria criteria = Search.query(exempleNode.getWikitty()).criteria()
                .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS);
        PagedResult<String> childNodesId = findAllByCriteria(
                securityToken, transaction, criteria);
        for (String childNodeId : childNodesId.getAll()) {
            tree.addChild(restoreTree(securityToken, transaction, childNodeId));
        }

        return tree;
    }

    @Override
    public Tree restoreTree(String securityToken, String wikittyId) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            Tree tree = restoreTree(securityToken, transaction, wikittyId);

            transaction.commit();
            return tree;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public List<String> deleteTree(String securityToken, String thesaurusId) {
        List<String> deletedIds = null;
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            deletedIds = deleteTree(securityToken, transaction, thesaurusId);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
        
        return deletedIds;
    }

    /**
     * Delete specified treenode and sub treenode.
     * 
     * @param securityToken security token
     * @param transaction transaction
     * @param treeNodeId thesaurusId to delete
     * @return all id of delete nodes
     */
    public List<String> deleteTree(String securityToken, WikittyTransaction transaction, String treeNodeId) {
        List<String> allTreeNodeId = getRecursiveTreeNodeId(securityToken, transaction, treeNodeId);
        delete(securityToken, transaction, allTreeNodeId);
        return allTreeNodeId;
    }

    /**
     * Get recursive id of tree node children of {@code treeNodeId}.
     * 
     * @param securityToken security token
     * @param transaction transaction
     * @param treeNodeId tree node id
     * @return all id of {@code treeNodeId}'s children 
     */
    protected List<String> getRecursiveTreeNodeId(String securityToken, WikittyTransaction transaction, String treeNodeId) {

        Search search = Search.query();
        search = search.eq(TreeNode.FQ_FIELD_TREENODE_PARENT, treeNodeId);
        Criteria criteria = search.criteria();

        PagedResult<String> childTreeNodeIds = findAllByCriteria(securityToken, transaction, criteria);
        List<String> treeNodeIds = new ArrayList<String>();
        treeNodeIds.add(treeNodeId);
        for (String childTreeNodeId : childTreeNodeIds.getAll()) {
            List<String> subTreeNodeIds = getRecursiveTreeNodeId(securityToken, transaction, childTreeNodeId);
            treeNodeIds.addAll(subTreeNodeIds);
        }
        return treeNodeIds;
    }

    @Override
    public Map.Entry<TreeNode, Integer> restoreNode(String securityToken,
            String wikittyId, Criteria filter) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(securityToken, transaction, wikittyId);
            if(w == null) {
                transaction.commit();
                return null;
            }

            if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
                throw new WikittyException(String.format(
                        "Wikitty '%s' do not handle extension %s",
                        wikittyId, TreeNode.EXT_TREENODE ));
            }

            TreeNode node = new TreeNodeImpl(w);
            Integer count = getSearchEngin().findNodeCount(transaction, w, filter);

            HashMap.SimpleEntry<TreeNode, Integer> result =
                    new SimpleEntry<TreeNode, Integer>(node, count);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Map<TreeNode, Integer> restoreChildren(String securityToken,
            String wikittyId, Criteria filter) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(securityToken, transaction, wikittyId);
            if(w == null) {
                transaction.commit();
                return null;
            }

            if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
                throw new WikittyException(String.format(
                        "Wikitty '%s' do not handle extension %s",
                        wikittyId, TreeNode.EXT_TREENODE ));
            }

            Map<TreeNode, Integer> result = new LinkedHashMap<TreeNode, Integer>();

            Map<String, Integer> search = getSearchEngin().findAllChildrenCount(
                    transaction, w, filter);
            Set<Entry<String, Integer>> children = search.entrySet();
            for (Entry<String, Integer> child : children) {
                Integer count = child.getValue();

                String id = child.getKey();
                Wikitty wikitty = restore(securityToken, transaction, id);
                TreeNode node = new TreeNodeImpl(wikitty);

                result.put(node, count);
            }

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Wikitty restoreVersion(String securityToken,
            String wikittyId, String version) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public UpdateResponse syncEngin(final String securityToken) {
        final WikittyTransaction transaction = new WikittyTransaction();
        try {
            final int numberForCommit = 1000;
            final WikittySearchEngin searchEngin = getSearchEngin();
            final UpdateResponse result = new UpdateResponse();
            final List<Wikitty> wikitties = new ArrayList<Wikitty>(numberForCommit);

            transaction.begin();
            searchEngin.clear(transaction);
            transaction.commit();
            transaction.begin();

            getWikittyStorage().scanWikitties(transaction, new WikittyStorage.Scanner() {
                int count = 0;

                @Override
                public void scan(String wikittyId) {
                    Wikitty wikitty = restore(securityToken, transaction, wikittyId);
                    Date deleteDate = wikitty.getDeleteDate();
                    if(deleteDate == null) {
                        count ++;
                        wikitties.add(wikitty);

                        if(count == numberForCommit) {
                            // Reindex
                            UpdateResponse response = searchEngin.store(transaction, wikitties);
                            result.add(response);
                            transaction.commit();
                            // Reinit
                            count = 0;
                            wikitties.clear();
                            transaction.begin();
                        }
                    }
                }
            });

            // Last wikitties
            UpdateResponse response = searchEngin.store(transaction, wikitties);
            result.add(response);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

}
