/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyServiceNotifier.java 436 2010-10-20 14:16:17Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyServiceNotifier.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.commons.beanutils.ConstructorUtils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ListenerSet;

/**
 * Wikitty service notifier.
 * 
 * Currently based on jgroups.
 * 
 * @author chatellier
 * @version $Revision: 436 $
 * 
 * Last update : $Date: 2010-10-20 16:16:17 +0200 (mer., 20 oct. 2010) $
 * By : $Author: bpoussin $
 */
public class WikittyServiceNotifier implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceNotifier.class);

    /** Service to delegate. */
    protected WikittyService ws;

    /** Wikitty service listener (all event). */
    protected ListenerSet<WikittyServiceListener> allWikittyServiceListeners;

    /** Wikitty service listener (only for local event). */
    protected ListenerSet<WikittyServiceListener> localWikittyServiceListeners;

    /** Wikitty service listener (only for remote event). */
    protected ListenerSet<WikittyServiceListener> remoteWikittyServiceListeners;

    /** notifier */
    protected WikittyServiceListener notifier;

    /**
     * Tous les events en attentent d'etre envoyer aux listeners
     */
    protected LinkedBlockingQueue<WikittyServiceEvent> eventToSend;

    /** thread utilise pour evoyer les events */
    protected EventThread eventThread;

    /**
     * Default constructor.
     * 
     * @param ws delegate service
     */
    public WikittyServiceNotifier(WikittyService ws) {
        this(ws, null);
    }

    /**
     * Constructor with configuration.
     * 
     * @param ws delegate service
     * @param props properties (can be null)
     */
    public WikittyServiceNotifier(WikittyService ws, Properties props) {
        // service
        this.ws = ws;

        // listeners
        allWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();
        localWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();
        remoteWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();

        eventToSend = new LinkedBlockingQueue<WikittyServiceEvent>();
        
        eventThread = new EventThread(eventToSend,
                allWikittyServiceListeners, localWikittyServiceListeners,
                remoteWikittyServiceListeners);

        // can be null according to default constructor
        if (props != null) {
            notifier = new RemoteNotifier(this, props);
        }
    }

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        // not delegated
        switch (type) {
            case ALL :
                synchronized(allWikittyServiceListeners) {
                    allWikittyServiceListeners.add(listener);
                }
                break;
            case LOCAL :
                synchronized(localWikittyServiceListeners) {
                    localWikittyServiceListeners.add(listener);
                }
                break;
            case REMOTE :
                synchronized(remoteWikittyServiceListeners) {
                    remoteWikittyServiceListeners.add(listener);
                }
                break;
        }
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        // not delegated
        switch (type) {
            case ALL :
                synchronized(allWikittyServiceListeners) {
                    allWikittyServiceListeners.remove(listener);
                }
                break;
            case LOCAL :
                synchronized (localWikittyServiceListeners) {
                    localWikittyServiceListeners.remove(listener);
                }
                break;
            case REMOTE :
                synchronized(remoteWikittyServiceListeners) {
                    remoteWikittyServiceListeners.remove(listener);
                }
                break;
        }
    }

    @Override
    public String login(String login, String password) {
        return ws.login(login, password);
    }

    @Override
    public void logout(String securityToken) {
        ws.logout(securityToken);
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        return ws.canWrite(securityToken, wikitty);
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        return ws.canDelete(securityToken, wikittyId);
    }

    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        return ws.canRead(securityToken, wikittyId);
    }

    @Override
    public void clear(String securityToken) {
        // FIXME poussin 20101015 pourquoi n'y a t il pas d'event pour un clear total ?
        // il faut prévenir les clients de synchro de ce vider totalement aussi
        ws.clear(securityToken);
    }

    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        UpdateResponse updateResponse = ws.store(securityToken, wikitty);
        // update wikitty
        updateResponse.update(wikitty);
        // notify listeners
        firePutWikitty(wikitty);
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        UpdateResponse updateResponse = ws.store(securityToken, wikitties);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties,
            boolean force) {
        UpdateResponse updateResponse = ws.store(
                securityToken, wikitties, force);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean force) {
        UpdateResponse updateResponse = ws.store(securityToken, transaction,
                wikitties, force);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // no notification
        return ws.getAllExtensionIds(securityToken);
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // no notification
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext) {
        UpdateResponse updateResponse = ws.storeExtension(securityToken, ext);
        // notify listeners
        firePutExtension(ext);
        return updateResponse;
    }

    @Override
    public UpdateResponse storeExtension(
            String securityToken, Collection<WikittyExtension> exts) {
        UpdateResponse updateResponse = ws.storeExtension(securityToken, exts);
        // notify listeners
        firePutExtension(exts.toArray(new WikittyExtension[exts.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, 
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        UpdateResponse updateResponse = ws.storeExtension(
                securityToken, transaction, exts);
        // no notification called by #storeExtension(Collection<WikittyExtension>)
        return updateResponse;
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // no notification
        return ws.restoreExtension(securityToken, id);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, 
            WikittyTransaction transaction, String id) {
        // no notification
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        // no notification
        return ws.restoreExtensionLastVersion(securityToken, name);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // no notification
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    @Override
    public Wikitty restore(String securityToken, String id) {
        // no notification
        return ws.restore(securityToken, id);
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        // no notification
        return ws.restore(securityToken, ids);
    }

    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        // no notification
        return ws.restore(securityToken, transaction, ids);
    }

    @Override
    public void delete(String securityToken, String id) {
        ws.delete(securityToken, id);
        // notify listeners
        fireRemoveWikitty(id);
    }

    @Override
    public void delete(String securityToken, Collection<String> ids) {
        ws.delete(securityToken, ids);
        // notify listeners
        fireRemoveWikitty(ids.toArray(new String[ids.size()]));
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken, Criteria criteria) {
        // no notification
        return ws.findAllByCriteria(securityToken, criteria);
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // no notification
        return ws.findAllByCriteria(securityToken, transaction, criteria);
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        // no notification
        return ws.findByCriteria(securityToken, criteria);
    }
    
    @Override
    public Wikitty findByCriteria(String securityToken, WikittyTransaction transaction, Criteria criteria) {
        // no notification
        return ws.findByCriteria(securityToken, transaction, criteria);
    }

    @Override
    public WikittyTree restoreTree(String securityToken, String wikittyId) {
        // no notification
        return ws.restoreTree(securityToken, wikittyId);
    }

    @Override
    public List<String> deleteTree(String securityToken, String wikittyId) {
        // no notification
        List<String> result = ws.deleteTree(securityToken, wikittyId);
        fireRemoveWikitty(result.toArray(new String[result.size()]));
        return result;
    }

    @Override
    public Entry<WikittyTreeNode, Integer> restoreNode(String securityToken, String wikittyId,
            Criteria filter) {
        // no notification
        return ws.restoreNode(securityToken, wikittyId, filter);
    }

    @Override
    public Map<WikittyTreeNode, Integer> restoreChildren(String securityToken,
            String wikittyId, Criteria filter) {
        // no notification
        return ws.restoreChildren(securityToken, wikittyId, filter);
    }

    @Override
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version) {
        // no notification
        return ws.restoreVersion(securityToken, wikittyId, version);
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        // no notification
        return ws.syncEngin(securityToken);
    }

    /**
     * Build event to fire and call {@link #firePutWikitty(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void firePutWikitty(Wikitty... ws) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.PUT_WIKITTY);
        Set<String> ids = new HashSet<String>();
        Map<String, Set<String>> idsExtension = new HashMap<String, Set<String>>();
        Map<String, String> idsVersion = new HashMap<String, String>();
        
        for (Wikitty w : ws) {
            // ids
            ids.add(w.getId());
            // extension
            Set<String> extension = new HashSet<String>();
            extension.addAll(w.getExtensionNames());
            idsExtension.put(w.getId(), extension);
            // version
            idsVersion.put(w.getId(), w.getVersion());
        }
        
        event.setIds(ids);
        event.setIdExtensions(idsExtension);
        event.setIdVersions(idsVersion);
        fireEvent(event);
    }

    /**
     * Build event to fire and call {@link #fireRemoveWikitty(WikittyServiceEvent)}.
     *
     * @param wikittyIds wikitty ids
     */
    protected void fireRemoveWikitty(String... wikittyIds) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.REMOVE_WIKITTY);
        Set<String> ids = new HashSet<String>();
        
        for (String wikittyId : wikittyIds) {
            // ids
            ids.add(wikittyId);
        }
        
        event.setIds(ids);
        fireEvent(event);
    }

    /**
     * Build event to fire and call {@link #fireClearWikitty(WikittyServiceEvent)}.
     */
    protected void fireClearWikitty() {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.CLEAR_WIKITTY);
        fireEvent(event);
    }

    /**
     * Build event to fire and call {@link #firePutExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void firePutExtension(WikittyExtension... exts) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.PUT_EXTENSION);
        Set<String> ids = new HashSet<String>();
        Map<String, Set<String>> idsExtension = new HashMap<String, Set<String>>();

        for (WikittyExtension ext : exts) {
            // ids
            ids.add(ext.getId());
            // extension
            Set<String> extension = new HashSet<String>();
            extension.add(ext.getName());

            // also add requires fields
            extension.add(ext.requires);

            idsExtension.put(ext.getId(), extension);
        }

        event.setIds(ids);
        event.setIdExtensions(idsExtension);
        fireEvent(event);
    }

    /**
     * Build event to fire and call {@link #fireRemoveExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void fireRemoveExtension(WikittyExtension... exts) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.REMOVE_EXTENSION);
        Set<String> ids = new HashSet<String>();
        
        for (WikittyExtension ext : exts) {
            // ids
            ids.add(ext.getId());
        }
        
        event.setIds(ids);
        fireEvent(event);
    }

    /**
     * Build event to fire and call {@link #fireClearExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void fireClearExtension() {
        WikittyServiceEvent event = new WikittyServiceEvent(ws,
                WikittyServiceEvent.WikittyEventType.CLEAR_EXTENSION);
        fireEvent(event);
    }

    /**
     * Fire event to all registred listener.
     *
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void fireEvent(final WikittyServiceEvent event) {
        // ajout d'un thread, car si les listener doit
        // ouvrir une transaction WikittyTransaction
        // alors que celui qui lance l'event en a une ouverte
        // cela cause une exception JTA
        EventThread thread = getEventThread();

        // si le thread de notification est en cours d'arret on leve une exception
        if (thread.stopAsked()) {
            throw new WikittyException(
                    "Event thread dispatcher is stopped, no more event can be send");
        } else {
            eventToSend.offer(event);

            if (!thread.isAlive()) {
                // on demarre le thread que lorsqu'il y a le premier event d'arrive
                thread.start();
            }
        }
    }

    /**
     * fire event passed in argument. Before fire, change source to current
     * WikittyServiceNotifier and set remote event to true.
     */
    public void processRemoteEvent(WikittyServiceEvent event) {
         //source is transient, add it here :
        event.setSource(this);
        event.setRemote(true); // received event became remote
        fireEvent(event);
    }

    /**
     * Retourne le dernier thread utiliser pour envoyer les events.
     *
     * @return
     */
    public EventThread getEventThread() {
        return eventThread;
    }

    @Override
    protected void finalize() throws Throwable {
        getEventThread().askStop();
        super.finalize();
    }

    /**
     * Thread utilise pour envoyer les events. On rend accessible ce thread
     * pour pouvoir y acceder depuis l'exterieur (pour l'instant pour les tests
     * mais peut-etre plus tard du monitoring). Il permet a un thread d'attendre
     * qu'un evenement leve a une certaine heure est bien ete dispatchee grace a
     * la methode waitfor
     */
    static public class EventThread extends Thread {

        protected boolean mustBeRunning = true;
        
        protected SortedMap<Long, Object> waiter = new TreeMap<Long, Object>();
        /**
         * reference vers la collection qui contient les events a envoyer
         */
        protected LinkedBlockingQueue<WikittyServiceEvent> eventToSend;

        /** Wikitty service listener (all event). */
        protected ListenerSet<WikittyServiceListener> allWikittyServiceListeners;

        /** Wikitty service listener (only for local event). */
        protected ListenerSet<WikittyServiceListener> localWikittyServiceListeners;

        /** Wikitty service listener (only for remote event). */
        protected ListenerSet<WikittyServiceListener> remoteWikittyServiceListeners;

        /** heure du dernier event envoye */
        protected long lastEventTime = 0;

        public EventThread(LinkedBlockingQueue<WikittyServiceEvent> eventToSend,
                ListenerSet<WikittyServiceListener> allWikittyServiceListeners,
                ListenerSet<WikittyServiceListener> localWikittyServiceListeners,
                ListenerSet<WikittyServiceListener> remoteWikittyServiceListeners) {
            super("wikitty-event-thread");
            this.eventToSend = eventToSend;
            this.allWikittyServiceListeners = allWikittyServiceListeners;
            this.localWikittyServiceListeners = localWikittyServiceListeners;
            this.remoteWikittyServiceListeners = remoteWikittyServiceListeners;
        }
        
        /**
         * demande l'arret du thread, ne doit être appeler que par le finalize
         * du WikittyServiceNotifier
         */
        protected void askStop() {
            this.mustBeRunning = false;
        }

        /**
         * retourne vrai si on a demande l'arret du thread
         * @return
         */
        public boolean stopAsked() {
            return !mustBeRunning;
        }

        /**
         * thread that want wait for particulare event to be processed, can be 
         * call this method with event time in argument. Used only in unit test
         * but this is necessary for test.
         */
        public void waitFor(long eventTime) throws InterruptedException {
            Object mutex = null;
            sleep(1); // sleep 1 millis to prevent problem with 2 events in same millis
            synchronized (waiter) {
                if (eventTime <= lastEventTime) {
                    // le thread demande a attendre un event deja passe
                    // on le met donc pas en attente
                    if (log.isDebugEnabled()) {
                        log.debug("event deja passe " + eventTime + " <= " + lastEventTime);
                    }
                    return;
                }
                mutex = waiter.get(eventTime);
                if (mutex == null) {
                    mutex = new Object();
                    waiter.put(eventTime, mutex);
                }
            }
            synchronized(mutex) {
                mutex.wait();
            }
        }

        @Override
        public void run() {
            while(mustBeRunning) {
                processEventQueue();
            }
            // le thread est arrete, force l'envoi de tous les events pour
            // liberer correctement tous les threads en attente
            // plus aucun event ne doit etre accepte dans la queue (voir method fireEvent)
            processEventQueue();
        }

        protected void processEventQueue() {
            try {
                WikittyServiceEvent event;
                // on attend pas indefiniment un event, car il faut verifier
                // aussi que personne n'a arrete le thread
                while (null != (event = eventToSend.poll(5, TimeUnit.SECONDS))) {
                    try {
                        synchronized (allWikittyServiceListeners) {
                            allWikittyServiceListeners.fire(
                                    event.getType().listenerMethodName, event);
                        }
                    } catch (Exception eee) {
                        log.error("Can't notify listener", eee);
                    }
                    try {
                        if (event.isRemote()) {
                            synchronized (remoteWikittyServiceListeners) {
                                remoteWikittyServiceListeners.fire(
                                        event.getType().listenerMethodName, event);
                            }
                        } else {
                            synchronized (localWikittyServiceListeners) {
                                localWikittyServiceListeners.fire(
                                        event.getType().listenerMethodName, event);
                            }
                        }
                    } catch (Exception eee) {
                        log.error("Can't notify listener", eee);
                    }
                    synchronized (waiter) {
                        // on met a jour l'heure du dernier event envoye
                        lastEventTime = event.getTime();

                        // on previent les threads en attente si besoin

                        // dans un premier temps on ne recupere que ceux
                        // inferieur a event.getTime()
                        SortedMap<Long, Object> subwaiter =
                                waiter.headMap(event.getTime());
                        for (Iterator<Map.Entry<Long, Object>> i = subwaiter.entrySet().iterator(); i.hasNext();) {
                            Object mutex = i.next().getValue();
                            i.remove();
                            synchronized (mutex) {
                                mutex.notifyAll();
                            }
                        }
                        // dans un second temps on verifie si le suivant ne
                        // serait pas egal a event.getTime()
                        if (!waiter.isEmpty()) {
                            Long time = waiter.firstKey();
                            // il pourrait y avoir plusieurs event avec la meme heure
                            // il faut bien tous les liberer
                            while (time.equals(event.getTime())) {
                                // il est bien egal on l'enleve aussi
                                Object mutex = waiter.remove(time);
                                synchronized (mutex) {
                                    mutex.notifyAll();
                                }

                                if (!waiter.isEmpty()) {
                                    time = waiter.firstKey();
                                } else {
                                    break;
                                }
                            }
                        }
                    }
                }
            } catch (InterruptedException eee) {
                log.error("Notification thread error", eee);
            }
        }
    };

    /**
     * This interface must be implemented to send and received remote message.
     * Only sendMessage method is in interface but you must write receive
     * method too, but this method is protocol specific and can't appear in
     * interface
     */
    static public interface RemoteNotifierTransporter {

        /**
         * Send a jgroup message to all other channel member.
         *
         * @param event message to send
         */
        public void sendMessage(WikittyServiceEvent event) throws Exception;
    }

    /**
     * Class used to notify remote listener. This class is realy activate
     * only if wikitty.notifier.transporter.class configuration is found and
     * wikitty.service.event.propagateEvent is true
     */
    static public class RemoteNotifier implements WikittyServiceListener {

        /** to use log facility, just put in your code: log.info(\"...\"); */
        static private Log log = LogFactory.getLog(RemoteNotifier.class);
        /**
         * Indique si les objects sont propages (true) vers les autres caches ou
         * simplement supprimes des autres caches (false). Default to {@code false}.
         */
        static public final String WIKITTY_EVENT_PROPAGATE_OPTION = "wikitty.service.event.propagateEvent";
        /** notifier class name in configuration that this service must used */
        static final public String WIKITTY_NOTIFIER_TRANSPORTER_CLASS = "wikitty.notifier.transporter.class";

        /** Notifier service reference reference. */
        protected WikittyServiceNotifier ws;

        /**
         * Indique si les objects sont propages (true) vers les autres caches ou
         * simplement supprimes des autres caches (false).
         *
         * @see WikittyServiceCached#WIKITTY_PROPAGATE_CACHE_OPTION
         */
        protected boolean propagateEvent = false;
        protected RemoteNotifierTransporter transporter;

        public RemoteNotifier(WikittyServiceNotifier ws, Properties props) {
            // can be null according to default constructor
            if (props != null) {
                this.ws = ws;

                if (!props.containsKey(WIKITTY_NOTIFIER_TRANSPORTER_CLASS)) {
                    props.setProperty(WIKITTY_NOTIFIER_TRANSPORTER_CLASS, JGroupsNotifierTransporter.class.getName());
                }
                ApplicationConfig config = new ApplicationConfig(props);
                
                propagateEvent = config.getOptionAsBoolean(WIKITTY_EVENT_PROPAGATE_OPTION);
                if (log.isDebugEnabled()) {
                    log.debug("Set propagateEvent option to " + propagateEvent);
                }

                Class transporterClass = config.getOptionAsClass(
                        WIKITTY_NOTIFIER_TRANSPORTER_CLASS);
                try {
                    transporter = (RemoteNotifierTransporter) ConstructorUtils.invokeConstructor(transporterClass,
                            new Object[]{ws, props},
                            new Class[]{WikittyServiceNotifier.class, Properties.class});
                } catch (Exception eee) {
                    throw new WikittyException("Can't create notifier: "
                            + transporterClass.getName(), eee);
                }

                // add this as listener when transporter is created without error
                ws.addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL); // weak reference
            }
            if (log.isInfoEnabled()) {
                if (transporter == null) {
                    log.info("RemoteNotifier synchronisation not used ");
                } else {
                    log.info("RemoteNotifier transporter: " + transporter.getClass().getName());
                }
            }
        }

        /**
         * Send a jgroup message to all other channel member.
         *
         * @param event message to send
         */
        protected void sendMessage(WikittyServiceEvent event) {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Try to send message : " + event);
                }

                transporter.sendMessage(event);

                if (log.isDebugEnabled()) {
                    log.debug("Message is sent : " + event);
                }
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't send message", eee);
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#putWikitty(org.nuiton.wikitty.Wikitty[])
         */
        @Override
        public void putWikitty(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate putWikitty event");
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#removeWikitty(java.lang.String[])
         */
        @Override
        public void removeWikitty(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate removeWikitty event");
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#clearWikitty()
         */
        @Override
        public void clearWikitty(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate clearWikitty event");
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#putExtension(org.nuiton.wikitty.WikittyExtension[])
         */
        @Override
        public void putExtension(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate putExtension event");
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#removeExtension(java.lang.String[])
         */
        @Override
        public void removeExtension(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate removeExtension event");
                }
            }
        }

        /*
         * @see org.nuiton.wikitty.WikittyServiceListener#clearExtension()
         */
        @Override
        public void clearExtension(WikittyServiceEvent event) {
            if (propagateEvent) {
                sendMessage(event);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Not master cache, do not propagate clearExtension event");
                }
            }
        }
    }
}
