/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyService.java 441 2010-10-21 10:35:22Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyService.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Wikitty service.
 *
 * The main implementation for this interface is {@link WikittyServiceImpl}.
 * It can be used alone but this implementation doesn't deal with all 
 * stuffs described in this interface. Thus, other functionalities are added
 * to the implementation through objects that decorate WikittyServiceImpl :
 *
 * <dl>
 *   <dt>{@link WikittyServiceCached}</dt>
 *     <dd>add a cache for wikitties</dd>
 *   <dt>{@link WikittyServiceSecurity}</dt>
 *     <dd>add user authentication support and right management</dd>
 *   <dt>{@link WikittyServiceNotifier}</dt>
 *     <dd>add notifications between client of the same wikitty service</dd>
 * </dl>
 * 
 * @author poussin
 * @version $Revision: 441 $
 *
 * Last update: $Date: 2010-10-21 12:35:22 +0200 (jeu., 21 oct. 2010) $
 * by : $Author: bpoussin $
 */
public interface WikittyService {

    /*
     * listeners 
     */
    
    /** Event listener type. */
    public enum ServiceListenerType {
        ALL, LOCAL, REMOTE
    }

    /**
     * Add new wikitty service listener.
     * 
     * Warning, {@code listener} is referenced as WeakReference, but sure to
     * another reference to work.
     * 
     * @param listener listener to add
     * @param type type of event to listen
     * 
     * @see ServiceListenerType
     */
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type);
    
    /**
     * Remove wikitty service listener.
     * 
     * Warning, {@code listener} is referenced as WeakReference, but sure to
     * another reference to work.
     * 
     * @param listener listener to remove
     * @param type type of event to listen
     * 
     * @see ServiceListenerType
     */
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type);

    /*
     * security
     */

    /**
     * Authenticate someone on WikittyService. securityToken returned must be
     * used to call others methods
     *
     * @param login can be application specifique login, but best practice is
     * to use email user
     * @param password
     * @return return token securityToken
     */
    public String login(String login, String password);

    /**
     * Unanthenticate someone by disabled securityToken
     * 
     * @param securityToken security token previously returned by login. If
     * securityToken is not valid, this method do nothing
     */
    public void logout(String securityToken);

    /*
     * Storage
     */

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     * 
     * @param securityToken security token
     */
    public void clear(String securityToken);

    /**
     * Verifie si l'utilisateur lie au securityToken a le droit d'ecrire
     * le Wikitty passe en argument
     *
     * @param securityToken le token de securite qui permet de retrouver
     * l'utilisateur et ainsi verifier les droits
     * @param wikitty le wikitty a sauver
     * @return vrai si l'utilisateur peut sauver l'objet
     */
    public boolean canWrite(String securityToken, Wikitty wikitty);

    /**
     * Verifie que l'utilisateur associe au securityToken peut supprimer le
     * wikitty dont on passe l'identifiant. Seul le propriétaire de l'objet
     * ou un admin peut supprimer un objet.
     *
     * Si l'id de l'objet est invalide, la methode retourne true, car la
     * suppression d'un id invalide ne fait rien
     *
     * @param securityToken security token
     * @param wikittyId wikitty id
     * @return vrai le la suppression ne posera pas de probleme.
     */
    public boolean canDelete(String securityToken, String wikittyId);

    /**
     * Un utilisateur peu lire un objet, s'il est Reader ou a defaut:
     * - owner
     * - AppAdmin
     * - Admin
     * - Writer
     *
     * @param securityToken security token
     * @param wikittyId wikitty id
     * @return vrai si l'utilisateur peut lire l'obbjet
     */
    public boolean canRead(String securityToken, String wikittyId);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param wikitty contains an id all the time.
     * @return update response containing version and id of stored wikitty
     */
    public UpdateResponse store(String securityToken, Wikitty wikitty);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param wikitties list of wikitty to be persisted
     * @return update response containing version and id of stored wikitty
     */
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param wikitties list of wikitty to be persisted
     * @param force boolean force non version version increment on saved wikitty
     *              or force version on wikitty creation (version 0.0)
     * @return update response
     */
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties,
            boolean force);


    /**
     * Manage Update and creation, action is done in transaction passed in
     * argument.
     *
     * @param securityToken security token
     * @param transaction transaction used to do the action (must not be null)
     * @param wikitties list of wikitty to be persisted
     * @param force boolean force non version version increment on saved wikitty
     *              or force version on wikitty creation (version 0.0)
     * @return update response
     */
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean force);

    /**
     * Return all extension id (ex: "extName[version])").
     * 
     * @param securityToken security token
     * @return extension ids list
     */
    public List<String> getAllExtensionIds(String securityToken);
    
    /**
     * Return all extension id (ex: "extName[version])") where extensionName is
     * required.
     * 
     * @param securityToken security token
     * @param extensionName extension name
     * @return extension id list
     */
    public List<String> getAllExtensionsRequires(String securityToken, String extensionName);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param ext extension to be persisted
     * @return update response
     */
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext);

    /**
     * Manage Update and creation.
     *
     * @param securityToken security token
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public UpdateResponse storeExtension(String securityToken,
            Collection<WikittyExtension> exts);

    /**
     * Manage Update and creation, action is done in transaction passed in
     * argument
     *
     * @param securityToken security token
     * @param transaction transaction used to do the action (must not be null)
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts);

    /**
     * Load extension from id. Id is 'name[version]'.
     * 
     * @param securityToken security token
     * @param extensionId extension id to restore
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(String securityToken, String extensionId);

    /**
     * Action is done in transaction passed in argument.
     *
     * @param securityToken security token
     * @param transaction transaction used to do the action (must not be null)
     * @param extensionId
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(String securityToken, 
            WikittyTransaction transaction, String extensionId);

    /**
     * Search extension with name in last version.
     * 
     * @param securityToken security token
     * @param name extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name);
    
    /**
     * Search extension with name in last version.
     * 
     * @param securityToken security token
     * @param transaction transaction used to do the action (must not be null)
     * @param name extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name);

    /**
     *
     * @param securityToken security token
     * @param id object id to restore
     * @return the corresponding object, or null if object doesn't exist, is
     * deleted or you don't have authorisation (you can check authorisation
     * before call restore with {@link #canRead(java.lang.String, java.lang.String)}
     */
    public Wikitty restore(String securityToken, String id);

    /**
     *
     * @param securityToken security token
     * @param id list of wikitty ids to restore
     * @return list of corresponding wikitty, if one id is not valid (no object
     * or deleted or no authorisation) this id is skip and result list have less
     * elements than collection argument
     */
    public List<Wikitty> restore(String securityToken, List<String> id);
    
    /**
     * Restore wikitty using opened transaction.
     * 
     * @param securityToken security token
     * @param transaction transaction to use
     * @param id list of wikitty ids to restore
     * @return list of corresponding wikitty, if one id is not valid (no object
     * or deleted or no authorisation) this id is skip and result list have less
     * elements than collection argument
     */
   public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> id);

    /**
     * Delete object if id exists.
     * 
     * @param securityToken security token
     * @param id object's id to remove
     */
    public void delete(String securityToken, String id);

    /**
     * Delete all object if id exists.
     * 
     * @param securityToken security token
     * @param ids object's ids to remove
     */
    public void delete(String securityToken, Collection<String> ids);

    /**
     *
     * @param securityToken security token
     * @param criteria
     * @return
     */
    public PagedResult<String> findAllByCriteria(
            String securityToken, Criteria criteria);

    /**
     * 
     * @param securityToken security token
     * @param transaction
     * @param criteria
     * @return
     */
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria);

    /**
     * Fist lonely (or first one) wikitty object that match criteria, if no
     * wikitty found or first retrived is not authorized for the user return
     * null
     *
     * @param securityToken security token
     * @param criteria
     * @return wikitty object or null
     */
    public Wikitty findByCriteria(String securityToken, Criteria criteria);
    
    /**
     * Fist lonely (or first one) wikitty object that match criteria, if no
     * wikitty found or first retrived is not authorized for the user return
     * null
     *
     * @param securityToken security token
     * @param criteria
     * @param transaction
     * @return wikitty object or null
     */
    public Wikitty findByCriteria(String securityToken, WikittyTransaction transaction, Criteria criteria);

    /*
     * Classification
     * Most of classification purpose is handle by extension mechanisms
     */

    /**
     * Retrieve all wikitties children (recursively) of an other one
     * Wikitty reference by wikittyId MUST include the 'Node' extension
     *
     * @param securityToken security token
     * @param wikittyId
     * @return
     */
    public WikittyTree restoreTree(String securityToken, String wikittyId);

    /**
     * Delete specified tree node and all sub nodes.
     * 
     * @param securityToken security token
     * @param treeNodeId tree node id to delete
     * @return delete wikitty ids
     */
    public List<String> deleteTree(String securityToken, String treeNodeId);

    /**
     * Retrieve wikitty node with count. Wikitty reference by wikittyId MUST
     * include the 'Node' extension.
     * 
     * @param securityToken security token
     * @param wikittyId
     * @return
     */
    public Map.Entry<WikittyTreeNode, Integer> restoreNode(String securityToken,
            String wikittyId, Criteria filter);

    /**
     * Retrieve all wikitties children with count (no recursively) of an other one
     * Wikitty reference by wikittyId MUST include the 'Node' extension
     *
     * @param securityToken security token
     * @param wikittyId
     * @return
     */
    public Map<WikittyTreeNode, Integer> restoreChildren(String securityToken,
            String wikittyId, Criteria filter);


    /*
     * history
     */

    /**
     * Restore wikitty in specifique version.
     * Authorisation is checked on last version even for previous wikitty version
     * 
     * @param securityToken security token
     */
    public Wikitty restoreVersion(String securityToken,
            String wikittyId, String version);

    /*
     * admin
     */

    /**
     * Synchronise search engine with wikitty storage engine, i.e. clear and
     * reindex all wikitties.
     * 
     * @param securityToken security token
     * @return update response
     */
    public UpdateResponse syncEngin(String securityToken);

}
