/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyProxy.java 444 2010-10-21 14:32:50Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyProxy.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

/**
 * Wikitty proxy is used to transform wikitty object used by {@link WikittyService}
 * into business objects used by applications.
 * 
 * It also manage {@link #securityToken} for {@link WikittyServiceSecurity}.
 * 
 * @author poussin
 * @version $Revision: 444 $
 *
 * Last update: $Date: 2010-10-21 16:32:50 +0200 (jeu., 21 oct. 2010) $
 * by : $Author: bpoussin $
 */
public class WikittyProxy {

    /** Delegated wikitty service. */
    protected WikittyService wikittyService;

    /**
     * Security token.
     * 
     * @see WikittyServiceSecurity#login(String, String)
     */
    protected String securityToken;

    /**
     * Empty constructor (uninitialized wikittyService).
     */
    public WikittyProxy() {
        
    }

    /**
     * Constructor with wikittyService.
     * 
     * @param wikittyService wikitty service
     */
    public WikittyProxy(WikittyService wikittyService) {
        this();
        this.wikittyService = wikittyService;
    }

    public String login(String login, String password) {
        String result = wikittyService.login(login, password);
        setSecurityToken(result);
        return result;
    }

    public void logout() {
        wikittyService.logout(securityToken);
    }

    public String getSecurityToken() {
        return securityToken;
    }

    public void setSecurityToken(String securityToken) {
        this.securityToken = securityToken;
    }

    public WikittyService getWikittyService() {
        return wikittyService;
    }

    public void setWikittyService(WikittyService wikittyService) {
        this.wikittyService = wikittyService;
    }

    /**
     * Cast Business object to another Business Object
     * If source don't have target extension, this extension is added
     *
     * @param source business entity source
     * @param target business entity class wanted
     * @return new instance of object wanted
     */
    public <E extends BusinessEntity> E cast(BusinessEntity source, Class<E> target) {
        E result = WikittyUtil.newInstance(
                securityToken, wikittyService, target, ((BusinessEntityWikitty)source).getWikitty());
        return result;
    }

    public <E extends BusinessEntity> E store(E e) {
        Wikitty w = ((BusinessEntityWikitty)e).getWikitty();
        UpdateResponse resp = wikittyService.store(securityToken, w);
        resp.update(w);
        return e;
    }

    /**
     * Store to WikittyService objects.
     * 
     * @param <E> object type
     * @param objets list
     * @return updated objects list
     */
    public <E extends BusinessEntity> List<E> store(List<E> objets) {
        // prepare data to send to service
        List<Wikitty> wikitties = new ArrayList<Wikitty>(objets.size());
        for (E e : objets) {
            Wikitty w = ((BusinessEntityWikitty)e).getWikitty();
            wikitties.add(w);
        }

        // call the service with Wikitty
        UpdateResponse resp = wikittyService.store(securityToken, wikitties);

        // update object
        for (Wikitty w : wikitties) {
            resp.update(w);
        }
        return objets;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @param checkExtension if true check that Wikitty result has all extension
     * declared in clazz
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id, boolean checkExtension) {
        if (id == null) {
            return null;
        }
        E result = null;
        HashSet<String> extNames = null;
        try {
            Wikitty wikitty = wikittyService.restore(securityToken, id);
            if(wikitty == null) {
                return null;
            }
            
            if (checkExtension) {
                extNames = new HashSet<String>(wikitty.getExtensionNames());
            }
            result = WikittyUtil.newInstance(
                    securityToken, wikittyService, clazz, wikitty);
        } catch (SecurityException eee) {
            throw eee;
        } catch (Exception eee) {
            throw new WikittyException("Can't restore wikitty", eee);
        }
        if (checkExtension) {
            // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
            BusinessEntityWikitty b = (BusinessEntityWikitty)result;
            Collection<WikittyExtension> BusinessEntityStaticExtensions = b.getStaticExtensions();
            for (WikittyExtension ext : BusinessEntityStaticExtensions) {
                String extensionName = ext.getName();
                if (!extNames.contains(extensionName)) {
                    // extension wanted by BusinessEntity (clazz)
                    // is not in wikitty, then wikitty is not good type
                    // for business
                    return null;
                }
            }
        }
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public Wikitty restore(String id) {
        if (id == null) {
            return null;
        }

    	Wikitty wikitty = wikittyService.restore(securityToken, id);
    	return wikitty;
    }
    
    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id) {
        E result = restore(clazz, id, false);
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be be found.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @param checkExtension if true check that Wikitty result has all extension
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> List<E> restore(
            Class<E> clazz, List<String> id, boolean checkExtension) {

        List<Wikitty> wikitties = wikittyService.restore(securityToken, id);
        List<E> result = new ArrayList<E>();
        for (Wikitty w : wikitties) {
            HashSet<String> extNames = null;
            if (checkExtension) {
                extNames = new HashSet<String>(w.getExtensionNames());
            }

            E dto = WikittyUtil.newInstance(
                    securityToken, wikittyService, clazz, w);
            if (checkExtension) {
                // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                BusinessEntityWikitty b = (BusinessEntityWikitty) dto;
                for (WikittyExtension ext : b.getStaticExtensions()) {
                    if (!extNames.contains(ext.getName())) {
                        // extension wanted by BusinessEntity (clazz)
                        // is not in wikitty, then wikitty is not good type
                        // for business
                        return null;
                    }
                }
            }
            result.add(dto);
        }
        return result;
    }

    public <E extends BusinessEntity> List<E> restore(Class<E> clazz, List<String> id) {
        List<E> result = restore(clazz, id, false);
        return result;
    }

    public void delete(String id) {
        wikittyService.delete(securityToken, id);
    }

    public void delete(Collection<String> ids) {
        wikittyService.delete(securityToken, ids);
    }

    public <E extends BusinessEntityWikitty> PagedResult<E> findAllByExample(E e,
            int firstIndex, int endIndex, String ... fieldFacet ) {

        Criteria criteria = Search.query(e.getWikitty()).criteria()
                .setFirstIndex(firstIndex).setEndIndex(endIndex)
                .setFacetField(fieldFacet);

        PagedResult<String> pagedResult = wikittyService.findAllByCriteria(
                securityToken, criteria);
        // we can use autoconvert = true because search by example add automaticaly
        // restriction on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(
                this, e.getClass(), true);
        return result;
    }

    public <E extends BusinessEntityWikitty> E findByExample(E e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();

        Wikitty w = wikittyService.findByCriteria(securityToken, criteria);
        E result = null;
        if (w != null) {
            result = (E) WikittyUtil.newInstance(
                    securityToken, wikittyService, e.getClass(), w);
        }
        return result;
    }

    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz). If criteria is null, find all extensions
     * else if criteria is empty return nothing.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param criteria criteria
     * @return paged result
     */
    public <E extends BusinessEntity> PagedResult<E> findAllByCriteria(
            Class<E> clazz, Criteria criteria) {

        // newInstance only return BusinessEntityWikittyImpl
        BusinessEntityWikitty sample =
                (BusinessEntityWikitty)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();

        Criteria serviceCriteria;
        if(criteria == null) {
            Search search = Search.query();
            search = search.eq(Element.ELT_EXTENSION, extensions);

            serviceCriteria = search.criteria();
        } else {
            Search search = Search.query(criteria);
            search = search.eq(Element.ELT_EXTENSION, extensions);

            serviceCriteria = search.criteria(criteria.getName());
            serviceCriteria.setFirstIndex(criteria.getFirstIndex());
            serviceCriteria.setEndIndex(criteria.getEndIndex());

            List<String> facetField = criteria.getFacetField();
            if(facetField != null) {
                String[] facetFieldTab = new String[facetField.size()];
                serviceCriteria.setFacetField(facetField.toArray(facetFieldTab));
            }

            List<Criteria> facetCriteria = criteria.getFacetCriteria();
            if(facetCriteria != null) {
                Criteria[] facetCriteriaTab = new Criteria[facetCriteria.size()];
                serviceCriteria.setFacetCriteria(facetCriteria.toArray(facetCriteriaTab));
            }

            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                String[] sortAscendingTab = new String[sortAscending.size()];
                serviceCriteria.setSortAscending(sortAscending.toArray(sortAscendingTab));
            }

            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                String[] sortDescendingTab = new String[sortDescending.size()];
                serviceCriteria.setSortDescending(sortDescending.toArray(sortDescendingTab));
            }
        }

        PagedResult<String> pagedResult = wikittyService.findAllByCriteria(
                securityToken, serviceCriteria);
        
        // we can cast with autoconvert true because we have add restriction
        // on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(
                this, sample.getClass(), true);
        return result;
    }

    public PagedResult<Wikitty> findAllByCriteria(Criteria criteria) {
    	PagedResult<String> resultId = wikittyService.findAllByCriteria(securityToken, criteria);
    	PagedResult<Wikitty> result = resultId.cast(securityToken, wikittyService);
    	return result;
    }
    
    public <E extends BusinessEntity> E findByCriteria(Class<E> clazz, Criteria criteria) {
        BusinessEntityWikitty sample =
                (BusinessEntityWikitty)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();
        Search search = Search.query(criteria);
        search = search.eq(Element.ELT_EXTENSION, extensions);

        criteria = search.criteria(criteria.getName());

        Wikitty w = wikittyService.findByCriteria(securityToken, criteria);
        E result = WikittyUtil.newInstance(
                securityToken, wikittyService, clazz, w);
        return result;
    }

    public Wikitty findByCriteria(Criteria criteria) {
    	Wikitty wikitty = wikittyService.findByCriteria(securityToken, criteria);
    	return wikitty;
    }
    
    public WikittyTree restoreTree(String wikittyId) {
        WikittyTree result = wikittyService.restoreTree(securityToken, wikittyId);
        return result;
    }

    /**
     * Delete specified tree node and all sub nodes.
     * 
     * @param treeNodeId tree node id to delete
     * @return {@true} if at least one node has been deleted
     */
    public List<String> deleteTree(String treeNodeId) {
        List<String> result = wikittyService.deleteTree(securityToken, treeNodeId);
        return result;
    }

    public <E extends BusinessEntity> Map.Entry<E, Integer> restoreNode(
            Class<E> clazz, String wikittyId, Criteria filter) {
        Entry<E, Integer> result = restoreNode(clazz, wikittyId, filter, false);
        return result;
    }

    public <E extends BusinessEntity> Map.Entry<E, Integer> restoreNode(
            Class<E> clazz, String wikittyId, Criteria filter, boolean checkExtension) {
        Map.Entry<WikittyTreeNode, Integer> node = wikittyService.restoreNode(
                securityToken, wikittyId, filter);
        if(node == null) {
            return null;
        }

        E bean = restore(clazz, wikittyId, checkExtension);
        if(bean == null) {
            return null;
        }

        Map.Entry<E, Integer> result =
                new HashMap.SimpleEntry<E, Integer>(bean, node.getValue());
        return result;
    }

    public <E extends BusinessEntity> Map<E, Integer> restoreChildren(
            Class<E> clazz, String wikittyId, Criteria filter) {
        Map<E, Integer> result = restoreChildren(clazz, wikittyId, filter, false);
        return result;
    }

    public <E extends BusinessEntity> Map<E, Integer> restoreChildren(
            Class<E> clazz, String wikittyId, Criteria filter, boolean checkExtension) {
        
        Map<WikittyTreeNode, Integer> result = wikittyService.restoreChildren(
                securityToken, wikittyId, filter);
        if(result == null) {
            return null;
        }

        Map<E, Integer> convertedResult = new LinkedHashMap<E, Integer>();
        Set<Entry<WikittyTreeNode, Integer>> entries = result.entrySet();
        for (Entry<WikittyTreeNode, Integer> entry : entries) {

            WikittyTreeNode node = entry.getKey();
            
            //TODO EC-20100420 was : String nodeId = node.getId();
            String nodeId = node.getWikittyId();

            E businessEntity = null;
            HashSet<String> extNames = null;
            try {
                Wikitty wikitty = wikittyService.restore(securityToken, nodeId);
                if(checkExtension) {
                    extNames = new HashSet<String>(wikitty.getExtensionNames());
                }
                businessEntity = WikittyUtil.newInstance(
                        securityToken, wikittyService, clazz, wikitty);
            } catch (SecurityException eee) {
                throw eee;
            } catch (Exception eee) {
                throw new WikittyException("Can't restore children", eee);
            }

            boolean goodType = true;
            if(checkExtension) {
                // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                BusinessEntityWikitty businessEntityImpl =
                        (BusinessEntityWikitty) businessEntity;
                Collection<WikittyExtension> staticExtensions =
                        businessEntityImpl.getStaticExtensions();

                for (Iterator<WikittyExtension> iterator =
                        staticExtensions.iterator(); iterator.hasNext() && goodType;) {
                    WikittyExtension ext = iterator.next();
                    if (!extNames.contains(ext.getName())) {
                        // extension wanted by BusinessEntity (clazz)
                        // is not in wikitty, then wikitty is not good type
                        // for business
                        goodType = false;
                    }
                }
            }

            if(goodType) {
                Integer nodeCount = entry.getValue();
                convertedResult.put(businessEntity, nodeCount);
            }
        }
        return convertedResult;
    }

    public Wikitty restoreVersion(String wikittyId, String version) {
        Wikitty result = wikittyService.restoreVersion(
                securityToken, wikittyId, version);
        return result;
    }

    public <E extends BusinessEntity> boolean hasType(Class<E> clazz, String wikittyId) {
        E result = null;
        HashSet<String> extNames = null;
        try {
            Wikitty wikitty = wikittyService.restore(securityToken, wikittyId);
            if (wikitty == null) {
                return false;
            }
            extNames = new HashSet<String>(wikitty.getExtensionNames());
            result = WikittyUtil.newInstance(
                    securityToken, wikittyService, clazz, wikitty);
        } catch (SecurityException eee) {
            throw eee;
        } catch (Exception eee) {
            throw new WikittyException(
                    "Can't retrieve wikitty needed for hasType test", eee);
        }
        // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
        BusinessEntityWikitty b = (BusinessEntityWikitty)result;
        for (WikittyExtension ext : b.getStaticExtensions()) {
            if (!extNames.contains(ext.getName())) {
                // extension wanted by BusinessEntity (clazz)
                // is not in wikitty, then wikitty is not good type
                // for business
                return false;
            }
        }
        return true;
    }

    /**
     * Manage Update and creation.
     *
     * @param ext extension to be persisted
     * @return update response
     */
    public UpdateResponse storeExtension(WikittyExtension ext) {
        UpdateResponse response = wikittyService.storeExtension(securityToken, ext);
        return response;
    }

    /**
     * Manage Update and creation.
     *
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public UpdateResponse storeExtension(Collection<WikittyExtension> exts) {
        UpdateResponse response = wikittyService.storeExtension(securityToken, exts);
        return response;
    }

    /**
     * Load extension from id. Id is 'name[version]'.
     * 
     * @param extensionId extension id to restore
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(String extensionId) {
        WikittyExtension extension = wikittyService.restoreExtension(securityToken, extensionId);
        return extension;
    }

    /**
     * Search extension with name in last version.
     * 
     * @param extensionName extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(String extensionName) {
        WikittyExtension extension = wikittyService.restoreExtensionLastVersion(securityToken, extensionName);
        return extension;
    }
    
    /**
     * Return all extension id (ex: "extName[version])").
     * 
     * @return extension id list
     */
    public List<String> getAllExtensionIds() {
        List<String> result = wikittyService.getAllExtensionIds(securityToken);
        return result;
    }
    
    /**
     * Return all extension id (ex: "extName[version])") where
     * {@code extensionName} is required.
     * 
     * @param extensionName extension name
     * @return extensions
     */
    public List<String> getAllExtensionsRequires(String extensionName) {
        List<String> result = wikittyService.getAllExtensionsRequires(securityToken, extensionName);
        return result;
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    public void clear() {
        wikittyService.clear(securityToken);
    }

    /**
     * Synchronize search engine with wikitty storage engine, i.e. clear and
     * reindex all wikitties.
     * 
     * @return update response
     */
    public UpdateResponse syncEngin() {
        UpdateResponse result = wikittyService.syncEngin(securityToken);
        return result;
    }
    
}
