/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: PagedResult.java 417 2010-10-15 15:24:44Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/PagedResult.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import static org.nuiton.i18n.I18n._;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Paged result containing result collection and facet topics.
 * 
 * @param <T> paged result value type
 * 
 * @author poussin
 * @version $Revision: 417 $
 *
 * Last update: $Date: 2010-10-15 17:24:44 +0200 (ven., 15 oct. 2010) $
 * by : $Author: sletellier $
 */
public class PagedResult<T> implements Serializable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(PagedResult.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = 8518774558223121855L;

    /** indice element in global search result */
    protected int firstIndice;
    /** total number of result if we call the query for all possible result */
    protected int numFound;
    /** query really executed */
    protected String queryString;
    /** list of result in the wanted interval */
    protected List<T> results;
    /** facet asked or null if no facet */
    protected Map<String, List<FacetTopic>> facets;

    /**
     * we don't initialize securityToken
     *
     * @param firstIndice
     * @param numFound
     * @param queryString
     * @param facets
     * @param results
     */
    public PagedResult(int firstIndice, int numFound, String queryString,
            Map<String, List<FacetTopic>> facets, List<T> results) {
        this.firstIndice = firstIndice;
        this.numFound = numFound;
        this.queryString = queryString;
        this.facets = facets;
        this.results = Collections.unmodifiableList(results);
    }

    /**
     * Call {@link #cast(WikittyProxy, Class, boolean)} with
     * autoconvert = true
     *
     * @param proxy used to retrieve securityToken and WikittyService
     * @param clazz target PagedResult type
     * @return new PagedResult, this result can have less elements that original
     * for some reason (security, ...)
     */
    public <E extends BusinessEntityWikitty> PagedResult<E> cast(
            WikittyProxy proxy, Class<E> clazz) {
        return cast(proxy, clazz, true);
    }
    
    /**
     * Convert all result to the wikitty type and return new PagedResult with
     * this new result list.
     *
     * @param securityToken security token
     * @param  ws wikitty service
     * @return new PagedResult, this result can have less elements that original
     * for some reason (security, ...)
     */
    public PagedResult<Wikitty> cast(String securityToken, WikittyService ws) {
        List<Wikitty> castedResult;

        if (results.size() == 0) {
            castedResult = new ArrayList<Wikitty>();
        } else {
            if (results.get(0) instanceof String) {
                // le pagedresult courant contient des Ids
                // Si ce n'est pas le cas, ca veut dire que le developpeur utilisant
                // ce PagedResult ne sait pas ce qu'il fait :)
                List<String> ids = (List<String>) results;
                castedResult = ws.restore(securityToken, ids);
            } else {
                throw new ClassCastException("PagedResult don't contains" +
                        " wikitty String id but " + results.get(0).getClass());
            }
        }
        PagedResult<Wikitty> result = new PagedResult<Wikitty>(
                firstIndice, numFound, queryString, facets, castedResult);
        return result;
    }

    /**
     * Convert all result to the wikitty type and return new PagedResult with
     * this new result list.
     * 
     * @param securityToken security token
     * @param ws wikitty service
     * @param transaction transaction
     * 
     * @return new PagedResult, this result can have less elements that original
     * for some reason (security, ...)
     */
    public PagedResult<Wikitty> cast(String securityToken, WikittyService ws, WikittyTransaction transaction) {
        List<Wikitty> castedResult;

        if (results.size() == 0) {
            castedResult = new ArrayList<Wikitty>();
        } else {
            if (results.get(0) instanceof String) {
                // le pagedresult courant contient des Ids
                // Si ce n'est pas le cas, ca veut dire que le developpeur utilisant
                // ce PagedResult ne sait pas ce qu'il fait :)
                List<String> ids = (List<String>) results;
                castedResult = ws.restore(securityToken, transaction, ids);
            } else {
                throw new ClassCastException("PagedResult don't contains" +
                        " wikitty String id but " + results.get(0).getClass());
            }
        }
        PagedResult<Wikitty> result = new PagedResult<Wikitty>(
                firstIndice, numFound, queryString, facets, castedResult);
        return result;
    }

    /**
     * Convert all result to the wanted type and return new PagedResult with
     * this new result list. If some result don't have the right extension (clazz)
     * this extension is automatically added if autoconvert is true. Else
     * an exception is thrown when result without extension is found.
     *
     * When you used autoconvert = false, you have a potentially problem when
     * you have modified a BusinessEntity to have new extension and all your
     * wikitty object are not uptodate in database.
     * 
     * @param <E> class to cast into
     * @param proxy used to retrieve securityToken and WikittyService
     * @param autoconvert if autoconvert is false and object don't all needed
     * extension, object is not put in the result
     * @return new PagedResult, this result can have less elements that original
     * for some reason (security, ...)
     */
    public <E extends BusinessEntityWikitty> PagedResult<E> cast(
            WikittyProxy proxy, Class<E> clazz, boolean autoconvert) {
        List<E> castedResult;

        if (results.size() > 0 && results.get(0) instanceof String) {
            // le pagedresult courant contient des Ids
            // Si ce n'est pas le cas, ca veut dire que le developpeur utilisant
            // ce PagedResult ne sait pas ce qu'il fait :)
            List<String> ids = (List<String>)results;
            castedResult = proxy.restore(clazz, ids, !autoconvert);
        } else {
            castedResult = new ArrayList<E>(results.size());
            E sample = WikittyUtil.newInstance(clazz);
            Collection<WikittyExtension> wantedExtension = sample.getStaticExtensions();
            for (T t : results) {
                if (t == null) {
                    castedResult.add(null);
                } else {
                    Wikitty w = null;
                    if (t instanceof Wikitty) {
                        w = (Wikitty) t;
                    } else if (t instanceof BusinessEntityWikitty) {
                        w = ((BusinessEntityWikitty) t).getWikitty();
                    } else {
                        throw new WikittyException(String.format(
                                "Illegal object result class '%s' can't convert it to wikitty",
                                t.getClass().getName(), clazz.getName()));
                    }

                    Collection<WikittyExtension> wikittyExtension = w.getExtensions();
                    if (autoconvert || wikittyExtension.containsAll(wantedExtension)) {
                        E e = WikittyUtil.newInstance(proxy.getSecurityToken(),
                                proxy.getWikittyService(), clazz, (Wikitty) t);
                        castedResult.add(e);
                    } else {
                        // silently pass current object, this object is not put
                        // in result
                        if (log.isDebugEnabled()) {
                            log.debug(_(
                                "Illegal object result class '%s' can't convert it to '%s'" +
                                "there is no same extension %s != %s",
                                t.getClass().getName(), clazz.getName(),
                                wikittyExtension, wantedExtension));
                        }
                    }
                }
            }
        }
        PagedResult<E> result = new PagedResult<E>(
                firstIndice, numFound, queryString, facets, castedResult);
        return result;
    }

    public int getFirstIndice() {
        return firstIndice;
    }

    public int getNumFound() {
        return numFound;
    }

    public String getQueryString() {
        return queryString;
    }

    /**
     * Return name of all facet used in query.
     * 
     * @return result's facets names
     */
    public Collection<String> getFacetNames() {
        Collection<String> result = facets.keySet();
        return result;
    }

    /**
     * Return all topic for the specified facet.
     * 
     * @param facetName name of the wanted facet
     * @return facet's topics
     */
    public List<FacetTopic> getTopic(String facetName) {
        List<FacetTopic> result = facets.get(facetName);
        return result;
    }

    /**
     * Get map represent facets.
     * 
     * @return all facets
     */
    public Map<String, List<FacetTopic>> getFacets() {
        return facets;
    }

    /**
     * Return the first element in result
     * 
     * Can throw an exception if no element available
     * @return first element
     */
    public T getFirst() {
        T result = get(0);
        return result;
    }

    /**
     * Return element at index.
     * 
     * @param i index
     * @return element at index
     */
    public T get(int i) {
        T result = results.get(i);
        return result;
    }

    /**
     * Return unmodifiable list of all result.
     * 
     * @return all results
     */
    public List<T> getAll() {
        return results;
    }

    /**
     * Return the number of result in this object.
     * 
     * @return result number
     */
    public int size() {
        int result = results.size();
        return result;
    }

}
