package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: AbstractTopiaApplicationContext.java 3008 2014-02-11 15:21:33Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-2/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/AbstractTopiaApplicationContext.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.nuiton.topia.persistence.TopiaApplicationContext;
import org.nuiton.topia.persistence.TopiaApplicationContextCache;
import org.nuiton.topia.persistence.TopiaConfigurationConstants;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaPersistenceContext;
import org.nuiton.topia.persistence.internal.support.TopiaServiceSupportImpl;
import org.nuiton.topia.persistence.event.TopiaEntitiesVetoable;
import org.nuiton.topia.persistence.event.TopiaEntityListener;
import org.nuiton.topia.persistence.event.TopiaEntityVetoable;
import org.nuiton.topia.persistence.event.TopiaSchemaListener;
import org.nuiton.topia.persistence.event.TopiaTransactionListener;
import org.nuiton.topia.persistence.event.TopiaTransactionVetoable;
import org.nuiton.topia.persistence.internal.support.TopiaFiresSupport;
import org.nuiton.topia.persistence.TopiaService;
import org.nuiton.topia.persistence.support.TopiaListenableSupport;
import org.nuiton.topia.persistence.support.TopiaServiceSupport;
import org.nuiton.topia.persistence.util.TopiaUtil;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaIdFactory;

import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.WeakHashMap;

/**
 * The application context is the main class in ToPIA usage. This class is a kind of equivalent of the RootTopiaContext.
 * It contains only high level methods and new contexts creation (transaction begin, ...). This class has to be extended
 * by the user, even if some default one could be automatically generated.
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 3.0
 */
public abstract class AbstractTopiaApplicationContext<K extends TopiaPersistenceContext> implements TopiaApplicationContext<K> {

    private static final Log log = LogFactory.getLog(AbstractTopiaApplicationContext.class);

    protected TopiaIdFactory topiaIdFactory;

    protected ImmutableMap<String, String> configuration;

    protected TopiaFiresSupport topiaFiresSupport = new TopiaFiresSupport();

    protected TopiaServiceSupport topiaServiceSupport;

    protected HibernateProvider hibernateProvider;

    protected TopiaHibernateSessionRegistry sessionRegistry = new TopiaHibernateSessionRegistry();

    protected boolean closed = false;

    protected Set<TopiaPersistenceContext> persistenceContexts = Collections.newSetFromMap(
            new WeakHashMap<TopiaPersistenceContext, Boolean>());

    public AbstractTopiaApplicationContext(Properties properties) {
        // TODO arnaud 11/10/13 vérifier le comportement
        this((Map) ImmutableMap.copyOf(properties));
    }

    public AbstractTopiaApplicationContext(Map<String, String> configuration) {
        Map<String, String> configurationCopy = Maps.newHashMap();
        configurationCopy.putAll(configuration);
        if ( ! configuration.containsKey(TopiaConfigurationConstants.CONFIG_PERSISTENCE_CLASSES)) {
            configurationCopy.put(TopiaConfigurationConstants.CONFIG_PERSISTENCE_CLASSES, getImplementationClassesAsString());
        }
        this.configuration = ImmutableMap.copyOf(configurationCopy);
        this.topiaServiceSupport = new TopiaServiceSupportImpl(this);
        // FIXME AThimel 11/10/13 This is not nice, need a 2-step init because of mutual dependency with hibernateProvider
        ((TopiaServiceSupportImpl)this.topiaServiceSupport).init();
    }

    protected abstract Class<? extends TopiaEntity>[] getImplementationClasses();

    protected String getImplementationClassesAsString() {
        StringBuilder buffer = new StringBuilder();
        for (Class<? extends TopiaEntity> aClass : getImplementationClasses()) {
            buffer.append(',').append(aClass.getName());
        }
        return buffer.substring(1);
    }

    protected void registerPersistenceContext(TopiaPersistenceContext persistenceContext) {
        persistenceContexts.add(persistenceContext);
    }

    // FIXME AThimel 25/11/13 I don't like it to be public, but necessary for services. Review it
    public HibernateProvider getHibernateProvider() {
        if (hibernateProvider == null) {
            hibernateProvider = new HibernateProvider(configuration, topiaServiceSupport, sessionRegistry);
        }
        return hibernateProvider;
    }

    protected TopiaListenableSupport getTopiaListenableSupport() {
        return topiaFiresSupport;
    }

    @Override
    public ImmutableMap<String, String> getConfiguration() {
        return configuration;
    }

    protected TopiaIdFactory getTopiaIdFactory() {
        if (topiaIdFactory == null) {
            String topiaIdFactoryClassName =
                    getConfiguration().get(TopiaConfigurationConstants.CONFIG_PERSISTENCE_TOPIA_ID_FACTORY_CLASS_NAME);
            if (StringUtils.isEmpty(topiaIdFactoryClassName)) {
                topiaIdFactory = new DefaultTopiaIdFactory();
            } else {
                try {
                    Class topiaIdFactoryClass = Class.forName(topiaIdFactoryClassName);
                    Preconditions.checkState(TopiaIdFactory.class.isAssignableFrom(topiaIdFactoryClass),
                            topiaIdFactoryClassName + " is not a valid class name. The class must implements "
                                    + TopiaIdFactory.class.getSimpleName());

                    topiaIdFactory = (TopiaIdFactory) topiaIdFactoryClass.newInstance();
                } catch (ClassNotFoundException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                } catch (InstantiationException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                } catch (IllegalAccessException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                }
            }

        }
        return topiaIdFactory;
    }

    public TopiaHibernateSessionRegistry getSessionRegistry() {
        return sessionRegistry;
    }

    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        topiaFiresSupport.addTopiaEntityListener(listener);
    }

    @Override
    public void addTopiaEntityListener(Class<? extends TopiaEntity> entityClass, TopiaEntityListener listener) {
        topiaFiresSupport.addTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        topiaFiresSupport.removeTopiaEntityListener(listener);
    }

    @Override
    public void removeTopiaEntityListener(Class<? extends TopiaEntity> entityClass, TopiaEntityListener listener) {
        topiaFiresSupport.removeTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaFiresSupport.addTopiaEntityVetoable(vetoable);
    }

    @Override
    public void addTopiaEntityVetoable(Class<? extends TopiaEntity> entityClass, TopiaEntityVetoable vetoable) {
        topiaFiresSupport.addTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntityVetoable(vetoable);
    }

    @Override
    public void removeTopiaEntityVetoable(Class<? extends TopiaEntity> entityClass, TopiaEntityVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void addTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        topiaFiresSupport.addTopiaEntitiesVetoable(vetoable);
    }

    @Override
    public void removeTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntitiesVetoable(vetoable);
    }

    @Override
    public void addTopiaTransactionListener(TopiaTransactionListener listener) {
        topiaFiresSupport.addTopiaTransactionListener(listener);
    }

    @Override
    public void removeTopiaTransactionListener(TopiaTransactionListener listener) {
        topiaFiresSupport.removeTopiaTransactionListener(listener);
    }

    @Override
    public void addTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        topiaFiresSupport.addTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void removeTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        topiaFiresSupport.removeTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        topiaFiresSupport.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        topiaFiresSupport.removePropertyChangeListener(listener);
    }

    @Override
    public void addTopiaSchemaListener(TopiaSchemaListener listener) {
        topiaFiresSupport.addTopiaSchemaListener(listener);
    }

    @Override
    public void removeTopiaSchemaListener(TopiaSchemaListener listener) {
        topiaFiresSupport.removeTopiaSchemaListener(listener);
    }

    @Override
    public <E extends TopiaService> boolean serviceEnabled(Class<E> interfaceService) {
        return topiaServiceSupport.serviceEnabled(interfaceService);
    }

    @Override
    public <E extends TopiaService> E getService(Class<E> interfaceService) throws TopiaNotFoundException {
        return topiaServiceSupport.getService(interfaceService);
    }

    @Override
    public Map<String, TopiaService> getServices() {
        return topiaServiceSupport.getServices();
    }

    @Override
    public List<Class<?>> getPersistenceClasses() {
        return getHibernateProvider().getPersistentClasses();
    }

    @Override
    public boolean isSchemaEmpty() {
        Configuration configuration = getHibernateProvider().getHibernateConfiguration();
        boolean result = TopiaUtil.isSchemaEmpty(configuration);
        return result;
    }

    @Override
    public boolean isTableExists(Class<?> clazz) {
        Configuration configuration = getHibernateProvider().getHibernateConfiguration();
        boolean result = TopiaUtil.isSchemaExist(configuration, clazz.getName());
        return result;
    }

    @Override
    public String getSchemaName() {
        // TODO AThimel 02/08/13 I absolutely don't know if it works
        return getConfiguration().get(TopiaConfigurationConstants.CONFIG_DEFAULT_SCHEMA);
    }

    @Override
    public void createSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }
            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreCreateSchema(this);
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).execute(showSchema, true, false, true);
            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePostCreateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not create schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    @Override
    public void showCreateSchema() {
        try {
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).
                    execute(true, false, false, true);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not show create schema for reason: %s",
                            eee.getMessage()), eee);
        }

    }

    @Override
    public void updateSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }

            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreUpdateSchema(this);
            new SchemaUpdate(getHibernateProvider().getHibernateConfiguration()).execute(showSchema,
                    true);
            // topiaFiresSupport.firePostUpdateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not update schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    @Override
    public void dropSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }

            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreDropSchema(this);
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).execute(showSchema, true, true, false);
//            topiaFiresSupport.firePostDropSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not drop schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    @Override
    public void closeContext() throws TopiaException {

        // Throw exception if context is already closed
        Preconditions.checkState(!closed, "Context was already closed");

        // Iterate over the children PersistenceContexts and close them
        for (TopiaPersistenceContext persistenceContext : persistenceContexts) {
            if (persistenceContext == null) {
                if (log.isWarnEnabled()) {
                    log.warn("null TopiaPersistenceContext found in #persistenceContexts");
                }
            } else {
                // Avoid to have exception from checkClosed method on child
                try {
                    if (!persistenceContext.isClosed()) {
                        persistenceContext.closeContext();
                    }
                } catch (Exception eee) {
                    // Don't let any exception stop the application closing
                    if (log.isWarnEnabled()) {
                        log.warn("Unable to close TopiaPersistenceContext", eee);
                    }
                }
            }
        }

        hibernateProvider.close();
        closed = true;

        // Context is closed, make sure it is not referenced anymore from the context cache
        TopiaApplicationContextCache.removeContext(this);

        if (log.isDebugEnabled()) {
            log.debug("TopiaApplicationContext closed");
        }
    }

    @Override
    public boolean isClosed() {
        return closed;
    }

}
