/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: AbstractTopiaDao.java 2864 2013-11-04 15:36:32Z bleny $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-4/topia-persistence/src/main/java/org/nuiton/topia/persistence/AbstractTopiaDao.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * TopiaDAOAbstract.java
 *
 * Created: 31 déc. 2005 13:10:34
 *
 * @author poussin <poussin@codelutin.com>
 * @version $Revision: 2864 $
 *
 * Last update: $Date: 2013-11-04 16:36:32 +0100 (Mon, 04 Nov 2013) $
 * by : $Author: bleny $
 */

package org.nuiton.topia.persistence;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.metadata.ClassMetadata;
import org.nuiton.topia.TopiaDaoSupplier;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaHibernateSupport;
import org.nuiton.topia.TopiaJpaSupport;
import org.nuiton.topia.TopiaListenableSupport;
import org.nuiton.topia.TopiaSqlSupport;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.framework.TopiaFiresSupport;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.PagerBeanUtil;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

/**
 * Cette classe permet d'avoir un ensemble de méthode implantée de façon
 * standard et plus spécifiquement pour Hibernate.
 * <p/>
 * Certains accès à Hibernate sont tout de même fait ici, car on a pris le choix
 * de se baser entièrement sur hibernate pour la persistence, et il est ainsi
 * possible d'accèder au meta information hibernate sur les classes lorque l'on
 * en a besoin.
 *
 * @param <E> le type de l'entite
 * @author bpoussin <poussin@codelutin.com>
 * @version $Id: AbstractTopiaDao.java 2864 2013-11-04 15:36:32Z bleny $
 */
public abstract class AbstractTopiaDao<E extends TopiaEntity> extends LegacyTopiaDao<E> implements TopiaDao<E> {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(AbstractTopiaDao.class);

    /**
     * Default batch size used to iterate on data.
     *
     * @since 2.6.14
     */
    protected int batchSize = 1000;

    protected TopiaHibernateSupport topiaHibernateSupport;

    protected TopiaJpaSupport topiaJpaSupport;

    protected TopiaSqlSupport topiaSqlSupport;

    protected TopiaListenableSupport topiaListenableSupport;

    protected TopiaIdFactory topiaIdFactory;

    protected TopiaFiresSupport topiaFiresSupport;

    protected TopiaDaoSupplier topiaDaoSupplier;

    public abstract TopiaEntityEnum getTopiaEntityEnum();

    public abstract Class<E> getEntityClass();

    /**
     * When AbstractTopiaContext create the TopiaDAOHibernate, it must call this
     * method just after.
     */
    public void init(
            TopiaHibernateSupport topiaHibernateSupport,
            TopiaJpaSupport topiaJpaSupport,
            TopiaSqlSupport topiaSqlSupport,
            TopiaListenableSupport topiaListenableSupport,
            TopiaIdFactory topiaIdFactory,
            TopiaFiresSupport topiaFiresSupport,
            TopiaDaoSupplier topiaDaoSupplier) {
        if (log.isDebugEnabled()) {
            log.debug("init dao for " + getEntityClass());
        }
        this.topiaHibernateSupport = topiaHibernateSupport;
        this.topiaJpaSupport = topiaJpaSupport;
        this.topiaSqlSupport = topiaSqlSupport;
        this.topiaListenableSupport = topiaListenableSupport;
        this.topiaIdFactory = topiaIdFactory;
        this.topiaFiresSupport = topiaFiresSupport;
        this.topiaDaoSupplier = topiaDaoSupplier;
    }

    public TopiaFiresSupport getTopiaFiresSupport() {
        return topiaFiresSupport;
    }

    @Override
    public int getBatchSize() {
        return batchSize;
    }

    @Override
    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    protected String newFromClause(String alias) {
        String hql = "from " + getTopiaEntityEnum().getImplementationFQN();
        if (StringUtils.isNotBlank(alias)) {
            hql += " " + alias;
        }
        return hql;
    }

    @Override
    public E newInstance() {
        if (log.isTraceEnabled()) {
            log.trace("entityClass = " + getEntityClass());
        }
        Class<E> implementation = (Class<E>)
                getTopiaEntityEnum().getImplementation();
        try {
            E newInstance = implementation.newInstance();
            return newInstance;
        } catch (InstantiationException e) {
            throw new TopiaException(
                    "Impossible de trouver ou d'instancier la classe "
                            + implementation);
        } catch (IllegalAccessException e) {
            throw new TopiaException(
                    "Impossible de trouver ou d'instancier la classe "
                            + implementation);
        }
    }

    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        topiaListenableSupport.addTopiaEntityListener(getEntityClass(), listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaListenableSupport.addTopiaEntityVetoable(getEntityClass(), vetoable);
    }

    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        topiaListenableSupport.removeTopiaEntityListener(getEntityClass(), listener);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaListenableSupport.removeTopiaEntityVetoable(getEntityClass(), vetoable);
    }

    @Override
    public E create(E entity) {
        try {
            // first set topiaId
            if (StringUtils.isBlank(entity.getTopiaId())) {

                // only set id if not already on
                String topiaId = topiaIdFactory.newTopiaId(getEntityClass(), entity);
                entity.setTopiaId(topiaId);
            }

            if (entity instanceof TopiaEntityContextable) {
                TopiaEntityContextable contextable = (TopiaEntityContextable) entity;
                contextable.setTopiaDAOSupplier(this.topiaDaoSupplier);
            }

            // save entity
            topiaHibernateSupport.getHibernateSession().save(entity);
            topiaFiresSupport.warnOnCreateEntity(entity);
            return entity;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public E create(String propertyName, Object propertyValue, Object... otherPropertyNamesAndValues) {
        Map<String, Object> properties =
                TopiaUtil.convertPropertiesArrayToMap(propertyName, propertyValue, otherPropertyNamesAndValues);
        E result = create(properties);
        return result;
    }

    @Override
    public E create(Map<String, Object> properties) {

        E result = newInstance();

        try {
            for (Map.Entry<String, Object> e : properties.entrySet()) {
                String propertyName = e.getKey();
                Object value = e.getValue();
                PropertyUtils.setProperty(result, propertyName, value);
            }
        } catch (IllegalAccessException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (InvocationTargetException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (NoSuchMethodException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        }

        create(result);

        return result;
    }

    @Override
    public E create() {
        E result = newInstance();
        create(result);
        return result;
    }

    @Override
    public Iterable<E> createAll(Iterable<E> entities) {
        for (E entity : entities) {
            create(entity);
        }
        return entities;
    }

    @Override
    public E update(E entity) {
        try {
            topiaHibernateSupport.getHibernateSession().saveOrUpdate(entity);
            topiaFiresSupport.warnOnUpdateEntity(entity);
            return entity;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public Iterable<E> updateAll(Iterable<E> entities) {
        for (E entity : entities) {
            update(entity);
        }
        return entities;
    }

    @Override
    public void delete(E entity) {
        try {
            topiaHibernateSupport.getHibernateSession().delete(entity);
            entity.notifyDeleted();
            topiaFiresSupport.warnOnDeleteEntity(entity);
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public void deleteAll(Iterable<E> entities) {
        for (E entity : entities) {
            delete(entity);
        }
    }

    protected HqlAndParametersBuilder<E> newHqlAndParametersBuilder() {
        HqlAndParametersBuilder<E> result = new HqlAndParametersBuilder<E>(getEntityClass());
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForProperties(String propertyName,
                                                             Object propertyValue,
                                                             Object... otherPropertyNamesAndValues) {
        Map<String, Object> properties =
                TopiaUtil.convertPropertiesArrayToMap(propertyName, propertyValue, otherPropertyNamesAndValues);
        HqlAndParametersBuilder<E> result = getHqlForProperties(properties);
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForNoConstraint() {
        Map<String, Object> properties = Collections.emptyMap();
        HqlAndParametersBuilder<E> result = getHqlForProperties(properties);
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForProperties(Map<String, Object> properties) {
        HqlAndParametersBuilder<E> result = newHqlAndParametersBuilder();
        for (Map.Entry<String, Object> property : properties.entrySet()) {
            result.addEquals(property.getKey(), property.getValue());
        }
        return result;
    }

    protected TopiaQueryBuilderRunQueryStep<E> forHql(String hql) {
        Map<String, Object> properties = Collections.emptyMap();
        TopiaQueryBuilderRunQueryStep<E> result = forHql(hql, properties);
        return result;
    }

    protected TopiaQueryBuilderRunQueryStep<E> forHql(String hql, Map<String, Object> hqlParameters) {
        TopiaQueryBuilderRunQueryStep result = new TopiaQueryBuilderRunQueryStep(this, hql, hqlParameters);
        return result;
    }

    protected TopiaQueryBuilderRunQueryStep<E> forHql(String hql, String parameterName,
                                                      Object parameterValue,
                                                      Object... otherParameterNamesAndValues) {
        Map<String, Object> hqlParameters =
                TopiaUtil.convertPropertiesArrayToMap(parameterName, parameterValue, otherParameterNamesAndValues);
        TopiaQueryBuilderRunQueryStep<E> result = forHql(hql, hqlParameters);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaStep<E> forProperties(Map<String, Object> properties) {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = getHqlForProperties(properties);
        TopiaQueryBuilderAddCriteriaStep result = new TopiaQueryBuilderAddCriteriaStep(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaStep<E> forProperties(String propertyName,
                                                             Object propertyValue,
                                                             Object... otherPropertyNamesAndValues) {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = getHqlForProperties(propertyName, propertyValue, otherPropertyNamesAndValues);
        TopiaQueryBuilderAddCriteriaStep result = new TopiaQueryBuilderAddCriteriaStep(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaStep<E> newQueryBuilder() {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = newHqlAndParametersBuilder();
        TopiaQueryBuilderAddCriteriaStep result = new TopiaQueryBuilderAddCriteriaStep(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forContains(String propertyName, Object propertyValue) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addContains(propertyName, propertyValue);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forEquals(String propertyName, Object propertyValue) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addEquals(propertyName, propertyValue);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forIn(String propertyName, Iterable<Object> propertyValues) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addIn(propertyName, propertyValues);
        return result;
    }

    protected boolean exists(String hql, Map<String, Object> hqlParameters) {
        List<E> entities = topiaJpaSupport.find(hql, 0, 0, hqlParameters);
        boolean result = !entities.isEmpty();
        return result;
    }

    protected long count(String hql, Map<String, Object> hqlParameters) {
        Preconditions.checkArgument(hql.toLowerCase().trim().startsWith("select count("));
        Long result = findUnique(hql, hqlParameters, Long.class);
        return result;
    }

    protected E findUnique(String hql, Map<String, Object> hqlParameters) {
        E result = findUnique(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected E findUniqueOrNull(String hql, Map<String, Object> hqlParameters) {
        E result = findUniqueOrNull(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected Optional<E> tryFindUnique(String hql, Map<String, Object> hqlParameters) {
        Optional<E> result = tryFindUnique(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected E findFirst(String hql, Map<String, Object> hqlParameters) {
        E result = findFirst(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected E findFirstOrNull(String hql, Map<String, Object> hqlParameters) {
        E result = findFirstOrNull(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected Optional<E> tryFindFirst(String hql, Map<String, Object> hqlParameters) {
        Optional<E> result = tryFindFirst(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected E findAny(String hql, Map<String, Object> hqlParameters) {
        E result = findAny(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected E findAnyOrNull(String hql, Map<String, Object> hqlParameters) {
        E result = findAnyOrNull(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected Optional<E> tryFindAny(String hql, Map<String, Object> hqlParameters) {
        Optional<E> result = tryFindAny(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected <R> R findUnique(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R result = findUniqueOrNull(hql, hqlParameters, type);
        if (result == null) {
            // TODO brendan 30/09/13 throw another exception if no result
            throw new TopiaException("query " + hql + " returns no elements");
        }
        return result;
    }

    protected <R> Optional<R> tryFindUnique(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R uniqueOrNull = findUniqueOrNull(hql, hqlParameters, type);
        Optional<R> result = Optional.fromNullable(uniqueOrNull);
        return result;
    }

    protected <R> R findUniqueOrNull(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        List<R> results = findAll(hql, hqlParameters, type, 0, 1);
        // If there is more than 1 result, throw an exception
        if (results.size() > 1) {
            String message = String.format(
                    "The query '%s' returns more than 1 unique result", hql);
            // TODO AThimel 02/08/13 Throw another exception if more than 1 result is found
            throw new TopiaException(message);
        }
        // otherwise return the first one, or null
        R result = Iterables.getOnlyElement(results, null);
        return result;
    }

    protected <R> R findFirst(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R result = findFirstOrNull(hql, hqlParameters, type);
        if (result == null) {
            // TODO brendan 30/09/13 throw another exception if no result
            throw new TopiaException("query " + hql + " returns no elements");
        }
        return result;
    }

    protected <R> Optional<R> tryFindFirst(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R firstOrNull = findFirstOrNull(hql, hqlParameters, type);
        Optional<R> result = Optional.fromNullable(firstOrNull);
        return result;
    }

    protected <R> R findFirstOrNull(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        Preconditions.checkArgument(hql.toLowerCase().contains("order by"));
        R result = findAnyOrNull(hql, hqlParameters, type);
        return result;
    }

    protected <R> R findAny(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R result = findAnyOrNull(hql, hqlParameters, type);
        if (result == null) {
            // TODO brendan 30/09/13 throw another exception if no result
            throw new TopiaException(String.format("Query '%s' returns no elements", hql));
        }
        return result;
    }

    protected <R> Optional<R> tryFindAny(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        R anyOrNull = findAnyOrNull(hql, hqlParameters, type);
        Optional<R> result = Optional.fromNullable(anyOrNull);
        return result;
    }

    protected <R> R findAnyOrNull(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<R> results = findAll(hql, hqlParameters, type, 0, 0);
        R result = Iterables.getOnlyElement(results, null);
        return result;
    }

    protected List<E> findAll(String hql, Map<String, Object> hqlParameters) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<E> result = topiaJpaSupport.findAll(hql, hqlParameters);
        return result;
    }

    protected List<E> findAll(String hql, Map<String, Object> hqlParameters, int startIndex, int endIndex) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<E> result = topiaJpaSupport.find(hql, startIndex, endIndex, hqlParameters);
        return result;
    }

    protected List<E> findAll(String hql, Map<String, Object> hqlParameters, TopiaPagerBean pager) {
        List<E> result = findAll(hql, hqlParameters, getEntityClass(), pager);
        return result;
    }

    protected <R> List<R> findAll(String hql, Map<String, Object> hqlParameters, Class<R> type) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        Preconditions.checkNotNull(type);
        List<R> result = topiaJpaSupport.findAll(hql, hqlParameters);
        return result;
    }

    protected <R> List<R> findAll(String hql, Map<String, Object> hqlParameters, Class<R> type, int startIndex, int endIndex) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        Preconditions.checkNotNull(type);
        List<R> result = topiaJpaSupport.find(hql, startIndex, endIndex, hqlParameters);
        return result;
    }

    protected <R> List<R> findAll(String hql, Map<String, Object> hqlParameters, Class<R> type, TopiaPagerBean pager) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        Preconditions.checkNotNull(type);
        Preconditions.checkNotNull(pager);

        if (StringUtils.isNotBlank(pager.getSortColumn())) {
            hql += " order by " + pager.getSortColumn();
            if (!pager.isSortAscendant()) {
                hql += " desc";
            }
        }

        List<R> result = topiaJpaSupport.find(
                hql,
                (int) pager.getRecordStartIndex(),
                (int) pager.getRecordEndIndex() - 1,
                hqlParameters);

        return result;
    }

    protected Iterable<E> findAllLazy(String hql, Map<String, Object> hqlParameters) {
        Iterable<E> result = findAllLazy(hql, hqlParameters, getEntityClass());
        return result;
    }

    protected <R> Iterable<R> findAllLazy(String hql, Map<String, Object> hqlParameters, Class<R> type) {

        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        Preconditions.checkNotNull(type);

        final Iterator<R> iterator = new FindAllIterator<E, R>(this,
                type,
                batchSize,
                hql,
                hqlParameters);

        Iterable<R> result = new Iterable<R>() {
            @Override
            public Iterator<R> iterator() {
                return iterator;
            }
        };

        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forTopiaIdEquals(String topiaId) {
        Preconditions.checkNotNull(topiaId, "given topiaId is null");
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = forEquals(TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forTopiaIdIn(Iterable<String> topiaIds) {
        Preconditions.checkNotNull(topiaIds, "given topiaIds is null");
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = forIn(TopiaEntity.PROPERTY_TOPIA_ID, (Iterable) topiaIds);
        return result;
    }

    @Override
    public E findByTopiaId(String id) {
        // AThimel 30/10/13 Not using findUnique to avoid querying several elements (cf. findUnique implementation)
        E result = forTopiaIdEquals(id).findAny();
        return result;
    }

    @Override
    public Optional<E> tryFindByTopiaId(String topiaId) {
        Optional<E> result = forTopiaIdEquals(topiaId).tryFindAny();
        return result;
    }

    @Override
    public List<String> findAllIds() {
        List<String> result = newQueryBuilder().findAllIds();
        return result;
    }

    @Override
    public List<E> findAll() {
        List<E> result = newQueryBuilder().findAll();
        return result;
    }

    @Override
    public Iterable<E> findAllLazy() {
        String hql = "from " + getTopiaEntityEnum().getImplementationFQN() + " order by id";
        Map<String, Object> hqlParameters = Collections.emptyMap();
        Iterable<E> result = findAllLazy(hql, hqlParameters);
        return result;
    }

    @Override
    public Iterator<E> iterator() {
        Iterator<E> result = findAllLazy().iterator();
        return result;
    }

    @Override
    public long count() {
        long result = newQueryBuilder().count();
        return result;
    }

    /**
     * package locale method because this is hibernate specific method and
     * we don't want expose it.
     *
     * @return the meta-data of the entity
     * @throws org.nuiton.topia.TopiaException if any pb
     */
    protected ClassMetadata getClassMetadata() {
        ClassMetadata meta = topiaHibernateSupport.getHibernateFactory()
                .getClassMetadata(getEntityClass());
        if (meta == null) {
            meta = topiaHibernateSupport.getHibernateFactory().getClassMetadata(
                    getTopiaEntityEnum().getImplementationFQN());
        }
        return meta;
    }

    public static class FindAllIterator<E extends TopiaEntity, R> implements Iterator<R> {

        protected Iterator<R> data;

        protected final AbstractTopiaDao<E> dao;

        protected final Class<R> type;

        protected final String hql;

        protected final Map<String, Object> params;

        protected TopiaPagerBean pager;

        public FindAllIterator(AbstractTopiaDao<E> dao,
                               Class<R> type,
                               int batchSize,
                               String hql,
                               Map<String, Object> params) {
            this.dao = dao;
            this.type = type;
            this.hql = hql;
            this.params = params;

            String hql2 = hql.toLowerCase();
            int i = hql2.indexOf("order by");
            if (i == -1) {
                throw new IllegalStateException(
                        "must have a *order by* in hql, " +
                        "but did not find it in " + hql);
            }

            // get the count (removing the order-by)
            long count2 = dao.count("SELECT COUNT(*) " +
                    hql.substring(0, i), params);
            pager = new TopiaPagerBean();
            pager.setRecords(count2);
            pager.setPageSize(batchSize);
            PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

            // empty iterator (will be changed at first next call)
            data = Iterators.emptyIterator();
        }

    
        public boolean hasNext() {
            boolean result = data.hasNext() || // no more data
                    pager.getPageIndex() < pager.getPagesNumber();
            return result;
        }

    
        public R next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }

            if (!data.hasNext()) {

                // must load iterator

                // increments page index
                pager.setPageIndex(pager.getPageIndex() + 1);
                PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

                // load new window of data
                data = dao.findAll(hql, params, type, pager).iterator();

            }

            R next = data.next();
            return next;
        }

    
        public void remove() {
            throw new UnsupportedOperationException(
                    "This iterator does not support remove operation.");
        }
    }

    public static class TopiaQueryBuilderAddCriteriaStep<E extends TopiaEntity> implements TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> {

        protected AbstractTopiaDao<E> topiaDAO;

        protected HqlAndParametersBuilder<E> hqlAndParametersBuilder;

        protected boolean hasOrderByClause = false;

        protected TopiaQueryBuilderAddCriteriaStep(AbstractTopiaDao<E> topiaDAO, HqlAndParametersBuilder<E> hqlAndParametersBuilder) {
            this.topiaDAO = topiaDAO;
            this.hqlAndParametersBuilder = hqlAndParametersBuilder;
        }

        @Override
        public TopiaQueryBuilderRunQueryStep<E> setOrderByArguments(Set<String> orderByArguments) {
            hasOrderByClause = ! Iterables.isEmpty(orderByArguments);
            hqlAndParametersBuilder.setOrderByArguments(orderByArguments);
            return getNextStep();
        }

        @Override
        public TopiaQueryBuilderRunQueryStep<E> setOrderByArguments(String... orderByArguments) {
            hqlAndParametersBuilder.setOrderByArguments(orderByArguments);
            return getNextStep();
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addEquals(String property, Object value) {
            hqlAndParametersBuilder.addEquals(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addIn(String property, Iterable<Object> values) {
            hqlAndParametersBuilder.addIn(property, values);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addContains(String property, Object value) {
            hqlAndParametersBuilder.addContains(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNull(String property) {
            hqlAndParametersBuilder.addNull(property);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNotNull(String property) {
            hqlAndParametersBuilder.addNotNull(property);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdEquals(String property, String topiaId) {
            hqlAndParametersBuilder.addTopiaIdEquals(property, topiaId);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdIn(String property, Iterable<String> topiaIds) {
            hqlAndParametersBuilder.addTopiaIdIn(property, topiaIds);
            return this;
        }

        // shortcuts to next step

        @Override
        public boolean exists() {
            return getNextStep().exists();
        }

        @Override
        public E findAnyOrNull() {
            return getNextStep().findAnyOrNull();
        }

        @Override
        public E findUniqueOrNull() {
            return getNextStep().findUniqueOrNull();
        }

        @Override
        public E findAny() {
            return getNextStep().findAny();
        }

        @Override
        public E findUnique() {
            return getNextStep().findUnique();
        }

        @Override
        public E findFirst() {
            return getNextStep().findFirst();
        }

        @Override
        public E findFirstOrNull() {
            return getNextStep().findFirstOrNull();
        }

        @Override
        public Optional<E> tryFindAny() {
            return getNextStep().tryFindAny();
        }

        @Override
        public Optional<E> tryFindFirst() {
            return getNextStep().tryFindFirst();
        }

        @Override
        public Optional<E> tryFindUnique() {
            return getNextStep().tryFindUnique();
        }

        @Override
        public List<E> findAll() {
            return getNextStep().findAll();
        }

        @Override
        public List<E> find(int startIndex, int endIndex) {
            return getNextStep().find(startIndex, endIndex);
        }

        @Override
        public Iterable<E> findAllLazy() {
            return getNextStep().findAllLazy();
        }

        @Override
        public long count() {
            return getNextStep().count();
        }

        @Override
        public List<String> findIds(int startIndex, int endIndex) {
            return getNextStep().findIds(startIndex, endIndex);
        }

        @Override
        public List<String> findAllIds() {
            return getNextStep().findAllIds();
        }

        protected TopiaQueryBuilderRunQueryStep<E> getNextStep() {
            String hql = hqlAndParametersBuilder.getHql();
            Map<String, Object> hqlParameters = hqlAndParametersBuilder.getHqlParameters();
            TopiaQueryBuilderRunQueryStep nextStep = new TopiaQueryBuilderRunQueryStep(topiaDAO, hql, hqlParameters);
            return nextStep;
        }

    }

    public static class TopiaQueryBuilderRunQueryStep<E extends TopiaEntity> implements org.nuiton.topia.persistence.TopiaQueryBuilderRunQueryStep<E> {

        protected final String hql;

        protected final Map<String, Object> hqlParameters;

        protected AbstractTopiaDao<E> topiaDAO;

        protected TopiaQueryBuilderRunQueryStep(AbstractTopiaDao<E> topiaDAO, String hql, Map<String, Object> hqlParameters) {
            this.hql = hql;
            this.hqlParameters = hqlParameters;
            this.topiaDAO = topiaDAO;
        }

        @Override
        public boolean exists() {
            return topiaDAO.exists(hql, hqlParameters);
        }

        @Override
        public long count() {
            String hqlWithSelectClause = "select count(topiaId) " + hql;
            return topiaDAO.count(hqlWithSelectClause, hqlParameters);
        }

        @Override
        public E findUnique() {
            return topiaDAO.findUnique(hql, hqlParameters);
        }

        @Override
        public E findUniqueOrNull() {
            return topiaDAO.findUniqueOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindUnique() {
            return topiaDAO.tryFindUnique(hql, hqlParameters);
        }

        @Override
        public E findFirst() {
            return topiaDAO.findFirst(hql, hqlParameters);
        }

        @Override
        public E findFirstOrNull() {
            return topiaDAO.findFirstOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindFirst() {
            return topiaDAO.tryFindFirst(hql, hqlParameters);
        }

        @Override
        public E findAny() {
            return topiaDAO.findAny(hql, hqlParameters);
        }

        @Override
        public E findAnyOrNull() {
            return topiaDAO.findAnyOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindAny() {
            return topiaDAO.tryFindAny(hql, hqlParameters);
        }

        @Override
        public List<E> findAll() {
            return topiaDAO.findAll(hql, hqlParameters);
        }

        @Override
        public Iterable<E> findAllLazy() {
            return topiaDAO.findAllLazy(hql, hqlParameters);
        }

        @Override
        public List<E> find(int startIndex, int endIndex) {
            return topiaDAO.findAll(hql, hqlParameters, startIndex, endIndex);
        }

        @Override
        public List<String> findAllIds() {
            // XXX brendan 30/09/13 does this truely work ?
            String hqlWithSelectClause = "select topiaId " + hql;
            return topiaDAO.findAll(hqlWithSelectClause, hqlParameters, String.class);
        }

        @Override
        public List<String> findIds(int startIndex, int endIndex) {
            // XXX brendan 30/09/13 does this truely work ?
            String hqlWithSelectClause = "select topiaId " + hql;
            return topiaDAO.findAll(hqlWithSelectClause, hqlParameters, String.class, startIndex, endIndex);
        }

    }

}
