/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: Binder.java 1948 2010-11-17 21:31:00Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.5.1/src/main/java/org/nuiton/util/beans/Binder.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util.beans;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ObjectUtil;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * A {@code binder} permits to copy some properties from an object to another
 * one.
 * <p/>
 * It is based on a {@link BinderModel} which contains the mapping of properties
 * to transfert from the source object to the destination object.
 * <p/>
 * Use the method {@link #copy(Object, Object,String...)} to transfert properties.
 * <p/>
 * Use the method {@link #obtainProperties(Object,String...)} to obtain
 *
 * @author tchemit <chemit@codelutin.com>
 * @param <I> the source bean type
 * @param <O> the destination bean type
 * @since 1.1.5
 */
public class Binder<I, O> implements Serializable {

    /** Logger */
    private static final Log log = LogFactory.getLog(Binder.class);

    private static final long serialVersionUID = 1L;

    /**
     * Types of loading of collections.
     *
     * @since 1.3
     */
    public enum CollectionStrategy {

        /** To just copy the reference of the collection. */
        copy {
            public Object copy(Object readValue) {

                // by default, just return same reference
                return readValue;
            }
        },

        /** To duplicate the collection */
        duplicate {

            @Override
            public Object copy(Object readValue) {
                if (readValue instanceof Set<?>) {
                    return new HashSet((Set<?>) readValue);
                }
                // in any other cases, let says this is a ArrayList
                if (readValue instanceof Collection<?>) {
                    return new ArrayList((Collection<?>) readValue);
                }
                return readValue;
            }
        };

        /**
         * Copy a given collection.
         *
         * @param readValue the collection value to copy
         * @return the copied collection
         */
        public abstract Object copy(Object readValue);
    }

    /** the model of the binder */
    protected BinderModel<I, O> model;

    /**
     * Obtains the type of the source bean.
     *
     * @return the type of the source bean
     */
    public Class<I> getSourceType() {
        return getModel().getSourceType();
    }

    /**
     * Obtains the type of the target bean.
     *
     * @return the type of the target bean
     */
    public Class<O> getTargetType() {
        return getModel().getTargetType();
    }

    /**
     * Obtain from the given object all properties registred in the binder
     * model.
     * <p/>
     * <b>Note:</b> If a property's value is null, it will not be injected in
     * the result.
     *
     * @param source        the bean to read
     * @param propertyNames subset of properties to load
     * @return the map of properties obtained indexed by their property name,
     *         or an empty map is the given {@code from} is {@code null}.
     */
    public Map<String, Object> obtainProperties(I source,
                                                String... propertyNames) {
        if (source == null) {
            // special limit case
            return Collections.emptyMap();
        }

        propertyNames = getProperties(propertyNames);

        Map<String, Object> result = new TreeMap<String, Object>();
        for (String sourceProperty : propertyNames) {

            try {
                Object read;
                Method readMethod = model.getSourceReadMethod(sourceProperty);
                read = readMethod.invoke(source);
                if (log.isDebugEnabled()) {
                    log.debug("property " + sourceProperty + ", type : " +
                              readMethod.getReturnType() + ", value = " + read);
                }
                if (readMethod.getReturnType().isPrimitive() &&
                    ObjectUtil.getNullValue(
                            readMethod.getReturnType()).equals(read)) {
                    // for primitive type case, force nullity
                    read = null;
                }
                if (read == null) {
                    continue;
                }
                if (model.containsBinderProperty(sourceProperty)) {
                    if (model.containsCollectionProperty(sourceProperty)) {
                        read = bindCollection(sourceProperty, read);
                    } else {
                        read = bindProperty(sourceProperty, read);
                    }
                } else if (model.containsCollectionProperty(sourceProperty)) {

                    // specific collection strategy is set, must use it
                    read = getCollectionValue(sourceProperty, read);
                }

                result.put(sourceProperty, read);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            } catch (InstantiationException e) {
                throw new RuntimeException(e);
            }
        }
        return result;
    }

    /**
     * Copy properties from a source bean to a destination one according to
     * the model of the binder. If {@code propertyNames} is defined, only
     * those properties will be copied.
     * <p/>
     * <b>Note:</b> If {@code from} object is null, then {@code null} values
     * will be set to mapped properties into {@code dst}
     *
     * @param source        the bean to read
     * @param target        the bean to write
     * @param propertyNames optional subset of properties to copy (if none is
     *                      specifed, will use all the properties defined in
     *                      binder)
     * @throws NullPointerException if target parameter is {@code null}
     */
    public void copy(I source, O target, String... propertyNames) {
        copy(source, target, false, propertyNames);
    }

    /**
     * Copy properties from a source bean to a destination one according to
     * the model of the binder excluding {@code propertyNames}.
     * <p/>
     * <b>Note:</b> If {@code from} object is null, then {@code null} values
     * will be set to mapped properties into {@code dst}.
     *
     * @param source        the bean to read
     * @param target        the bean to write
     * @param propertyNames optional subset of properties to copy (if none is
     *                      specifed, will use all the properties defined in
     *                      binder)
     * @throws NullPointerException if target parameter is {@code null}
     */
    public void copyExcluding(I source, O target, String... propertyNames) {
        copy(source, target, true, propertyNames);
    }

    /**
     * Copy properties from a source bean to a destination one according to
     * the model of the binder.
     * <p/>
     * <b>Note:</b> If {@code from} object is null, then {@code null} values
     * will be set to mapped properties into {@code dst}.
     *
     * @param source            the bean to read
     * @param target            the bean to write
     * @param excludeProperties true to exclude following {@code propertyNames}
     * @param propertyNames     optional subset of properties to copy (if none is
     *                          specifed, will use all the properties defined in
     *                          binder)
     * @throws NullPointerException if target parameter is {@code null}
     */
    protected void copy(I source, O target, boolean excludeProperties,
                        String... propertyNames)
            throws NullPointerException {
        if (target == null) {
            throw new NullPointerException("parameter 'target' can no be null");
        }

        propertyNames = excludeProperties ?
                        getAllPropertiesExclude(propertyNames) :
                        getProperties(propertyNames);


        for (String sourceProperty : propertyNames) {

            String targetProperty = model.getTargetProperty(sourceProperty);

            try {
                Object read = null;
                Method readMethod = model.getSourceReadMethod(sourceProperty);
                if (source != null) {
                    // obtain value from source
                    read = readMethod.invoke(source);
                }
                // obtain acceptable null value (for primitive types, use
                // default values).
                if (read == null) {
                    read = ObjectUtil.getNullValue(readMethod.getReturnType());
                }
                if (log.isDebugEnabled()) {
                    log.debug("property " + sourceProperty + ", type : " +
                              readMethod.getReturnType() + ", value = " + read);
                }

                if (model.containsBinderProperty(sourceProperty)) {
                    if (model.containsCollectionProperty(sourceProperty)) {
                        read = bindCollection(sourceProperty, read);
                    } else {
                        read = bindProperty(sourceProperty, read);
                    }
                } else if (model.containsCollectionProperty(sourceProperty)) {

                    // specific collection strategy is set, must use it
                    read = getCollectionValue(sourceProperty, read);
                }
                model.getTargetWriteMethod(targetProperty).invoke(target, read);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            } catch (InstantiationException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Get the model of the binder.
     *
     * @return the model of the binder
     */
    protected BinderModel<I, O> getModel() {
        return model;
    }

    /**
     * Set the model of the binder.
     *
     * @param model the model of the binder
     */
    protected void setModel(BinderModel<?, ?> model) {
        this.model = (BinderModel<I, O>) model;
    }

    /**
     * Obtain the properties, if none is given in {@code propertyNames}
     * parameter, will use all property names defined in binder's model,
     * otherwise, check that all given property names are safe (registred in
     * binder's model).
     *
     * @param propertyNames optional subset of properties to get
     * @return the array of property names
     */
    protected String[] getProperties(String... propertyNames) {

        if (propertyNames.length == 0) {
            // use all properties in the binder
            propertyNames = model.getSourceDescriptors();
        } else {

            // use a subset of properties, must check them
            for (String propertyName : propertyNames) {
                if (!model.containsSourceProperty(propertyName)) {
                    throw new IllegalArgumentException(
                            "property '" + propertyName +
                            "' is not known by binder");
                }
            }
        }
        return propertyNames;
    }

    /**
     * Obtains all properties from binder's model except those {@code
     * propertyNameExcludes}. Unknown properties will be ignored.
     *
     * @param propertyNameExcludes name of properties to exclude
     * @return the array of property names without those in argument
     */
    protected String[] getAllPropertiesExclude(String... propertyNameExcludes) {
        List<String> excludes = Arrays.asList(propertyNameExcludes);
        List<String> results = new ArrayList<String>();
        for (String propertyName : model.getSourceDescriptors()) {
            if (!excludes.contains(propertyName)) {
                results.add(propertyName);
            }
        }
        return results.toArray(new String[results.size()]);
    }

    protected Object getCollectionValue(String sourceProperty, Object readValue) {
        CollectionStrategy strategy =
                model.getCollectionStrategy(sourceProperty);
        Object result = strategy.copy(readValue);
        return result;
    }

    protected Object bindProperty(String sourceProperty, Object read) throws IllegalAccessException, InstantiationException {
        Binder binder = model.getBinder(sourceProperty);
        Object result = bind(binder, read);
        return result;
    }


    protected Object bindCollection(String sourceProperty, Object read) throws IllegalAccessException, InstantiationException {

        if (read == null) {
            return null;
        }

        Binder binder = model.getBinder(sourceProperty);

        Collection result = null;

        if (read instanceof Set<?>) {
            result = new HashSet();
        }
        // in any other cases, let says this is a ArrayList
        if (read instanceof Collection<?>) {
            result = new ArrayList();
        }

        for (Object o : (Collection<?>) read) {
            Object r = bind(binder, o);
            result.add(r);
        }
        return result;
    }

    protected Object bind(Binder binder, Object read) throws IllegalAccessException, InstantiationException {
        Object result = read.getClass().newInstance();
        binder.copy(read, result);
        return result;
    }
}
