package org.nuiton.util.config;

/*
 * #%L
 * Nuiton Utils :: Nuiton Config
 * $Id: ApplicationConfigHelper.java 2521 2013-03-03 15:20:18Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6.10/nuiton-config/src/main/java/org/nuiton/util/config/ApplicationConfigHelper.java $
 * %%
 * Copyright (C) 2011 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashSet;
import java.util.ServiceLoader;
import java.util.Set;

/**
 * Helper about {@link ApplicationConfig}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.4.8
 */
public class ApplicationConfigHelper {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(ApplicationConfigHelper.class);

    protected ApplicationConfigHelper() {
        // helper with no instance
    }

    /**
     * Obtain all providers on class-path.
     *
     * @param classLoader optional classLoader used to seek for providers
     * @param includes    optional includes providers to use (if none then accept all providers)
     * @param excludes    optional excludes providers (if none the no reject)
     * @param verbose     verbose flag
     * @return sets of providers
     */
    public static Set<ApplicationConfigProvider> getProviders(ClassLoader classLoader,
                                                              Set<String> includes,
                                                              Set<String> excludes,
                                                              boolean verbose) {
        ServiceLoader<ApplicationConfigProvider> loader;
        if (classLoader == null) {
            loader = ServiceLoader.load(ApplicationConfigProvider.class);

        } else {
            loader = ServiceLoader.load(ApplicationConfigProvider.class,
                                        classLoader);
        }

        Set<ApplicationConfigProvider> result =
                new HashSet<ApplicationConfigProvider>();

        for (ApplicationConfigProvider configProvider : loader) {
            String name = configProvider.getName();
            if (includes != null && !includes.contains(name)) {

                // reject by include
                if (verbose) {
                    log.info("configuration named '" + name +
                             "' is rejected by includes.");
                }
                continue;
            }
            if (excludes != null && excludes.contains(name)) {

                // reject by exclude
                if (verbose) {
                    log.info("configuration named '" + name +
                             "' is rejected by excludes.");
                }
                continue;
            }
            if (verbose) {
                log.info("configuration named '" + name +
                         "' will be generated.");
            }
            result.add(configProvider);
        }
        return result;
    }

    public static ApplicationConfigProvider getProvider(ClassLoader classLoader,
                                                        String name) {
        Set<ApplicationConfigProvider> providers = getProviders(
                classLoader, null, null, false);
        ApplicationConfigProvider result = null;
        for (ApplicationConfigProvider provider : providers) {
            if (name.equals(provider.getName())) {
                result = provider;
                break;
            }
        }
        return result;
    }

    /**
     * Load default options from all given config providers.
     *
     * @param config    config where to add default options.
     * @param providers providers to use
     * @since 2.6.7
     */
    public static void loadAllDefaultOption(ApplicationConfig config,
                                            Set<ApplicationConfigProvider> providers) {

        for (ApplicationConfigProvider provider : providers) {
            if (log.isInfoEnabled()) {
                log.info("Load default options from configuration: " +
                         provider.getName());
            }
            if (log.isInfoEnabled()) {
                for (ConfigOptionDef optionDef : provider.getOptions()) {
                    log.info(" " + optionDef.getKey() +
                             " (" + optionDef.getDefaultValue() + ')');
                }
            }
            config.loadDefaultOptions(provider.getOptions());
        }
    }

    /**
     * Gets all transient options from the given providers.
     *
     * @param providers providers to inspect
     * @return the set of all options that are transient
     * @see ConfigOptionDef#isTransient()
     * @since 2.6.7
     */
    public static Set<ConfigOptionDef> getTransientOptions(Set<ApplicationConfigProvider> providers) {
        Set<ConfigOptionDef> result = new HashSet<ConfigOptionDef>();
        for (ApplicationConfigProvider provider : providers) {
            for (ConfigOptionDef def : provider.getOptions()) {
                if (def.isTransient()) {
                    result.add(def);
                }
            }
        }
        return result;
    }


    /**
     * Gets all final options from the given providers.
     *
     * @param providers providers to inspect
     * @return the set of all options that are final
     * @see ConfigOptionDef#isFinal()
     * @since 2.6.7
     */
    public static Set<ConfigOptionDef> getFinalOptions(Set<ApplicationConfigProvider> providers) {
        Set<ConfigOptionDef> result = new HashSet<ConfigOptionDef>();
        for (ApplicationConfigProvider provider : providers) {
            for (ConfigOptionDef def : provider.getOptions()) {
                if (def.isFinal()) {
                    result.add(def);
                }
            }
        }
        return result;
    }

    /**
     * Get all option keys that should not be saved in the user config file
     * from the given options providers.
     * <p/>
     * Such options are {@code transient} or {@code final}.
     *
     * @param providers providers to inspect
     * @return the set of options key not to store in the config file
     * @see ConfigOptionDef#isFinal()
     * @see ConfigOptionDef#isTransient()
     * @since 2.6.7
     */
    public static Set<String> getTransientOrFinalOptionKey(Set<ApplicationConfigProvider> providers) {
        Set<String> result = new HashSet<String>();
        for (ConfigOptionDef def : getTransientOptions(providers)) {
            result.add(def.getKey());
        }
        for (ConfigOptionDef def : getFinalOptions(providers)) {
            result.add(def.getKey());
        }
        return result;
    }
}
