/*
 * #%L
 * JAXX :: Maven plugin
 * 
 * $Id: AbstractGenerateHelpMojo.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.1/maven-jaxx-plugin/src/main/java/org/nuiton/jaxx/plugin/AbstractGenerateHelpMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.jaxx.plugin;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.nuiton.i18n.I18nUtil;

import java.io.*;
import java.util.*;

/**
 * Abstract Mojo to generate help stuff.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0.0
 */
public abstract class AbstractGenerateHelpMojo extends AbstractJaxxMojo {

    /**
     * The locales to generate for help, separeted by comma.
     * <p/>
     * The first locale given is the default locale.
     *
     * @parameter expression="${jaxx.locales}"
     * @required
     * @since 2.0.0
     */
    protected String locales;
    /**
     * Where to generate helpIds files.
     *
     * @parameter expression="${jaxx.outputHelpIds}" default-value="target/generated-sources/jaxx"
     * @required
     * @since 1.3
     */
    private File outputHelpIds;
    /**
     * The store of helpIds generated by the goal {@link GenerateMojo} and then
     * used by the goal {@link GenerateHelpMojo}.
     *
     * @parameter expression="${jaxx.helpIdsFilename}" default-value="helpIds.properties"
     * @required
     * @since 1.3
     */
    private String helpIdsFilename;
    /**
     * The store of cumulate helpIds generated by the goal {@link GenerateMojo}
     * and then used by the goal {@link GenerateHelpMojo}.
     *
     * @parameter expression="${jaxx.mergeHelpIdsFilename}" default-value="helpIds-all.properties"
     * @required
     * @since 1.3
     */
    private String mergeHelpIdsFilename;
    /**
     * Flag to generate the search index.
     *
     * @parameter expression="${jaxx.generateSearch}" default-value="true"
     * @required
     * @since 1.3
     */
    protected boolean generateSearch;
    /**
     * Default locale (the first locale in {@link #localesToTreate}.
     */
    private Locale defaultLocale;
    /**
     * Locales to treate
     */
    private Locale[] localesToTreate;

    /**
     * Do the action for the given locale.
     *
     * @param locale          the locale to treate
     * @param isDefaultLocale {@code true} if given locale is de the default
     *                        locale
     * @param source          where are stored help files for the given locale
     * @param localePath      the locale path to use (is {@code default} if
     *                        given locale is default).
     * @throws Exception if any pb
     */
    protected abstract void doActionForLocale(
            Locale locale,
            boolean isDefaultLocale,
            File source,
            String localePath) throws Exception;

    /**
     * Call back after doing all stuff for all locales declared
     */
    protected abstract void postDoAction();

    @Override
    protected void init() throws Exception {

        if (StringUtils.isEmpty(locales)) {
            throw new MojoFailureException(
                    "You must set the 'locales' property properly (was " +
                    locales + ").");
        }

        // check there is a outHelp
        if (getTargetDirectory() == null) {
            throw new MojoFailureException(
                    "You must set the 'outputHelpXXX' property.");
        }

        List<Locale> tmp = new ArrayList<Locale>();
        for (String loc : locales.split(",")) {
            Locale l = I18nUtil.newLocale(loc);
            tmp.add(l);
        }

        if (tmp.isEmpty()) {
            throw new MojoFailureException(
                    "No locale to react, you must set the 'locales' property.");
        }

        localesToTreate = tmp.toArray(new Locale[tmp.size()]);
        defaultLocale = localesToTreate[0];

        createDirectoryIfNecessary(getTargetDirectory());
    }

    @Override
    protected boolean checkSkip() {
        if (!generateHelp) {
            getLog().info("generateHelp flag is off, will skip goal.");
            return false;
        }
        return true;
    }

    @Override
    protected void doAction() throws Exception {

        for (Locale locale : localesToTreate) {

            boolean isDefaultLocale = defaultLocale.equals(locale);

            String language = locale.getLanguage();

            String localePath = isDefaultLocale ? "default" : language;

            File source = new File(getTargetDirectory(), localePath);

            createDirectoryIfNecessary(source);

            doActionForLocale(locale, isDefaultLocale, source, localePath);

        }

        postDoAction();
    }

    public File getOutputHelpIds() {
        return outputHelpIds;
    }

    public void setOutputHelpIds(File outputHelpIds) {
        this.outputHelpIds = outputHelpIds;
    }

    public File getHelpIdsStoreFile() {
        return outputHelpIds == null ? null :
               new File(outputHelpIds, helpIdsFilename);
    }

    public File getMergeHelpIdsStoreFile() {
        return outputHelpIds == null ? null :
               new File(outputHelpIds, mergeHelpIdsFilename);
    }

    public String getHelpIdsFilename() {
        return helpIdsFilename;
    }

    public void setHelpIdsFilename(String helpIdsFilename) {
        this.helpIdsFilename = helpIdsFilename;
    }

    public String getMergeHelpIdsFilename() {
        return mergeHelpIdsFilename;
    }

    public void setMergeHelpIdsFilename(String mergeHelpIdsFilename) {
        this.mergeHelpIdsFilename = mergeHelpIdsFilename;
    }

    protected void cleanHelpIdsStore() throws IOException {
        File idsStore = getHelpIdsStoreFile();
        if (idsStore.exists()) {
            if (getLog().isDebugEnabled()) {
                getLog().debug("delete id store " + idsStore);
            }
            boolean b = idsStore.delete();
            if (!b) {
                throw new IOException("could not delete file " + idsStore);
            }
        }
    }

    protected Set<String> loadHelpIds(File file) throws IOException {

        BufferedReader reader = null;
        Set<String> result = new HashSet<String>();
        reader = new BufferedReader(new InputStreamReader(
                new FileInputStream(file), getEncoding()));
        try {
            String id;
            while ((id = reader.readLine()) != null) {
                result.add(id.trim());
            }
            if (isVerbose()) {
                getLog().info("load " + result.size() +
                              " help ids from file " + file);
            }
            return result;
        } finally {
            reader.close();
        }
    }

    protected void storeHelpIds(File file, Set<String> ids) throws IOException {

        createDirectoryIfNecessary(file.getParentFile());

        StringBuilder buffer = new StringBuilder();

        for (String helpId : ids) {
            buffer.append(removeQuote(helpId)).append('\n');
        }
        writeFile(file, buffer.toString(), getEncoding());
        if (isVerbose()) {
            getLog().info("stored " + ids.size() + " help ids in " + file);
        }
    }

    protected String removeQuote(String txt) {
        if (txt.startsWith("\"")) {
            txt = txt.substring(1);
        }
        if (txt.endsWith("\"")) {
            txt = txt.substring(0, txt.length() - 1);
        }
        return txt;
    }

}
