/* *##%
 * Copyright (C) 2007
 *     JaxxPlugin, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/
package org.nuiton.jaxx;

import jaxx.beaninfos.BeanInfoUtil;
import jaxx.compiler.CompilerOptions;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXCompilerLaunchor;
import jaxx.runtime.JAXXContext;
import jaxx.tags.TagManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.codehaus.plexus.util.DirectoryScanner;
import org.nuiton.util.FileUpdaterHelper;
import org.nuiton.util.MirroredFileUpdater;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import jaxx.compiler.CompiledObjectDecorator;
import jaxx.compiler.HelpRootCompiledObjectDecorator;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.apache.maven.project.MavenProject;

/**
 * Classe permettant de transformer des sources jaxx vers du source java.
 *
 * @author chemit
 * @goal generate
 * @phase process-sources
 * @requiresDependencyResolution compile
 * @requiresProject
 */
public class JaxxGeneratorMojo extends AbstractJaxxMojo {

    /**
     * Le compilateur à utiliser (par défaut celui de Swing)
     *
     * @parameter expression="${jaxx.compilerFQN}" default-value="jaxx.compiler.SwingCompiler"
     */
    protected String compilerFQN;
    /**
     * Le compilateur à utiliser (par défaut celui de Swing)
     *
     * @parameter expression="${jaxx.validatorFQN}" default-value="jaxx.runtime.validator.swing.SwingValidator"
     */
    protected String validatorFQN;
    /**
     * chemin du repertoire de generation des resources.
     *
     * @parameter expression="${jaxx.outResource}" default-value="${basedir}/target/generated-sources/resources"
     */
    protected File outResource;
    /**
     * chemin du repertoire de compilation des resources.
     *
     * @parameter expression="${jaxx.outClass}" default-value="${basedir}/target/classes"
     */
    protected File outClass;
    /**
     * Repertoire sources des fichiers jaxx a generer.
     *
     * @parameter expression="${jaxx.src}" default-value="${maven.src.dir}/main/java"
     */
    protected File src;
    /**
     * pour optimizer le code compile ou genere ?
     *
     * @parameter expression="${jaxx.optimize}" default-value="false"
     */
    protected boolean optimize;
    /**
     * les options de la compilation
     *
     * @parameter expression="${jaxx.javaOpts}"
     */
    protected String javaOpts = null;
    /**
     * pour filter les fichiers a traiter
     *
     * @parameter expression="${jaxx.includes}"
     */
    protected String[] includes;
    /**
     * pour filter les fichiers a ne pas traiter
     *
     * @parameter expression="${jaxx.excludes}"
     */
    protected String[] excludes;
    /**
     * flag to include in compiler classpath the java sources directories (src and outJava).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addSourcesToClassPath}" default-value="false"
     */
    protected boolean addSourcesToClassPath;
    /**
     * flag to include in compiler classpath the java resources directories (src and outJava).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addResourcesToClassPath}" default-value="false"
     * @since 1.6.0
     */
    protected boolean addResourcesToClassPath;
    /**
     * flag to include in compiler classpath the compile class-path (can only be used in a test phase).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addCompileClassPath}" default-value="false"
     * @since 1.6.0
     */
    protected boolean addCompileClassPath;
    /**
     * flag to include in compiler classpath the project compile classpath.
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addProjectClassPath}" default-value="false"
     */
    protected boolean addProjectClassPath;
    /**
     * to force generation of java source for any jaxx files with no timestamp checking.
     * <p/>
     * By default, never force generation.
     *
     * @parameter expression="${jaxx.force}" default-value="false"
     */
    protected boolean force;
    /**
     * flag to add logger to each generated jaxx file.
     * <p/>
     * By default, always add it.
     *
     * @parameter expression="${jaxx.addLogger}" default-value="true"
     */
    protected boolean addLogger;
    /**
     * flag to keep compilers after the generate operation (usefull for tests.
     * <p/>
     * By default, always reset.
     *
     * @parameter expression="${jaxx.resetAfterCompile}" default-value="true"
     */
    protected boolean resetAfterCompile;
    /**
     * the name of implementation of {@link jaxx.runtime.JAXXContext}
     * to be used on {@link jaxx.runtime.JAXXObject}.
     * <p/>
     * Must not be abstract.
     *
     * @parameter expression="${jaxx.jaxxContextImplementorClass}" default-value="jaxx.runtime.DefaultJAXXContext"
     * @required
     */
    protected String jaxxContextImplementorClass;
    /**
     * extra path to be added in {@link java.beans.Introspector#setBeanInfoSearchPath(String[])}.
     * <p/>
     * add beanInfoSearchPath to be registred by {@link BeanInfoUtil#addJaxxBeanInfoPath(String[])}
     * <p/>
     * and then will be use by {@link jaxx.tags.swing.SwingInitializer#initialize()}.
     * <p/>
     * <p/>
     * This permit to use real beanInfo of imported graphic libraries.
     *
     * @parameter expression="${jaxx.beanInfoSearchPath}"
     */
    protected String[] beanInfoSearchPath;
    /**
     * list of fqn of class toimport for all generated jaxx files
     *
     * @parameter expression="${jaxx.extraImports}"
     *
     * @deprecated Prefer use of extraImportList as a string, so
     *             could be use in properties section.
     */
    protected String[] extraImports;
    /**
     * list of fqn of class toimport for all generated jaxx files
     *
     * @parameter expression="${jaxx.extraImportList}"
     */
    protected String extraImportList;
    /**
     * the FQN of the ui to use for error notification.
     * <p/>
     * If not given, will use the one defined in validator
     *
     * @parameter expression="${jaxx.defaultErrorUIFQN}"
     * @see jaxx.runtime.validator.swing.SwingValidator#DEFAULT_UI_CLASS
     */
    protected String defaultErrorUIFQN;
    /**
     * the FQN of the ui to use for error notification.
     * <p/>
     * If not given, will use the one defined in validator
     *
     * @parameter expression="${jaxx.defaultDecorator}" default-value="jaxx.compiler.DefaultCompiledObjectDecorator"
     *
     * @see jaxx.compiler.CompiledObjectDecorator
     */
    protected String defaultDecoratorFQN;
    /**
     * a flag to use UIManager to retreave icons.
     *
     * @parameter expression="${jaxx.useUIManagerForIcon}" default-value="false"
     */
    protected boolean useUIManagerForIcon;
    /**
     * flag to activate profile mode.
     * <p/>
     * By default, not active.
     *
     * @parameter expression="${jaxx.profile}" default-value="false"
     */
    protected boolean profile;
    /**
     * flag to activate help generation process.
     * <p/>
     * By default, not active.
     *
     * @parameter expression="${jaxx.generateHelp}" default-value="false"
     *
     * @since 1.3
     */
    protected boolean generateHelp;
    /**
     * the FQN of help broker
     * <p/>
     * By default, none.
     *
     * @parameter expression="${jaxx.helpBrokerFQN}"
     *
     * @since 1.3
     */
    protected String helpBrokerFQN;
    /**
     * A flag to mark themojo to be used in a test phase. This will permits to add generated sources in test compile roots.
     *
     * @parameter expression="${jaxx.testPhase}" default-value="false"
     * @since 1.6.0
     */
    protected boolean testPhase;
    protected String[] files;
    private static final String[] INCLUDES = {"**\\/*.jaxx"};
    protected CompilerOptions options;
    protected MirroredFileUpdater updater;
    private Class<?> defaultErrorUIClass;
    private Class<?> validatorClass;
    private Class<? extends CompiledObjectDecorator> defaultDecoratorClass;
    private Class<? extends JAXXCompiler> compilerClass;

    @SuppressWarnings("unchecked")
    @Override
    public void init() throws Exception {

        if (project != null && ("pom".equals(project.getPackaging()) || "site".equals(project.getPackaging()))) {
            // nothing to be done for this type of packaging
            skip = true;
            getLog().info("skip generate goal for packaging " + project.getPackaging());
            return;
        }

        if (generateHelp) {
            // check there is some bundle
            if (helpIdStore == null) {
                throw new MojoFailureException("you must set the helpIdStore property.");
            }
        }
        skip = false;

        checkJaxxContextImplementorClass();

        if (beanInfoSearchPath != null && beanInfoSearchPath.length > 0) {
            // register extra path
            BeanInfoUtil.addJaxxBeanInfoPath(beanInfoSearchPath);
        }
        if (!outResource.exists()) {
            outResource.mkdirs();
        }

        if (!outJava.exists()) {
            outJava.mkdirs();
        }

        fixCompileSourceRoots();

        if (addSourcesToClassPath || addProjectClassPath) {
            cl = initClassLoader(project, getLog());
            Thread.currentThread().setContextClassLoader(cl);
        } else {
            cl = getClass().getClassLoader();
            //cl = Thread.currentThread().getContextClassLoader();
        }


        compilerClass = (Class<? extends JAXXCompiler>) Class.forName(compilerFQN, false, cl);
        defaultDecoratorClass = (Class<? extends CompiledObjectDecorator>) Class.forName(defaultDecoratorFQN, false, cl);

        // check the validator class is correct
        validatorClass = Class.forName(validatorFQN, false, cl);

        if (defaultErrorUIFQN != null && !defaultErrorUIFQN.trim().isEmpty()) {
            defaultErrorUIClass = Class.forName(defaultErrorUIFQN, false, cl);
        }

        DirectoryScanner ds;
        ds = new DirectoryScanner();
        ds.setBasedir(src);
        boolean noIncludes = includes == null || includes.length == 0;
        ds.setIncludes(noIncludes ? INCLUDES : includes);

        if (excludes != null && excludes.length > 0) {
            ds.setExcludes(excludes);
        }

        ds.scan();
        String[] filesFind = ds.getIncludedFiles();
        if (verbose) {
            getLog().info("jaxx - discover " + filesFind.length + " jaxx file(s). ");
        }

        updater = FileUpdaterHelper.newJaxxFileUpdater(src, outJava);

        if (force) {
            // we will regenerate all files
            this.files = filesFind;
        } else {
            // filter files
            List<String> listFiles = new ArrayList<String>();

            for (String file : filesFind) {
                if (updater.isFileUpToDate(new File(src, file))) {
                    if (verbose) {
                        getLog().info("jaxx - skip file [" + file + "].");
                    }
                } else {
                    if (verbose) {
                        getLog().info("jaxx - detect modify file [" + file + "].");
                    }
                    listFiles.add(file);
                }
            }
            this.files = listFiles.toArray(new String[listFiles.size()]);
        }

        if (extraImportList != null && !extraImportList.isEmpty()) {
            String[] imports = extraImportList.split(",");
            int i = 0;
            for (String importS : imports) {
                imports[i++] = importS.trim();
            }
            if (verbose) {
                getLog().info("extra imports " + java.util.Arrays.toString(imports));
            }
            extraImports = imports;

        }
        options = toCompilerOptions();

        if (verbose) {
            printInit();
        }

    }

    @Override
    public void doAction() throws MojoExecutionException {
        getLog().info("jaxx - detects " + this.files.length + " modify jaxx file(s). ");

        try {

            // force compiler init from here, not in a static block
            TagManager.reset(verbose);

            JAXXCompilerLaunchor launchor = JAXXCompilerLaunchor.newLaunchor(src, files, options);
            boolean success = launchor.compile();
            getLog().info("jaxx - generate " + launchor.getCompilerCount() + " file(s). ");

            if (!success) {
                throw new MojoExecutionException("Aborting due to errors reported by jaxxc");
            }

            if (generateHelp) {
                // generate help
                generateHelp();
            }

        } catch (MojoExecutionException e) {
            getLog().error(e);
            throw e;
        } catch (Exception e) {
            //getLog().error(e);
            Throwable e2 = e;
            while (e2.getCause() != null) {
                e2 = e.getCause();
            }
            getLog().error(e2);

            throw new MojoExecutionException(e2.getMessage(), e2);
        }
    }

    public CompilerOptions toCompilerOptions() {
        CompilerOptions result = new CompilerOptions();
        result.setClassPath(src.getPath());
        if (javaOpts != null && !"".equals(javaOpts)) {
            result.setJavacOpts(javaOpts);
        }
        result.setCompilerClass(compilerClass);
        result.setValidatorFQN(validatorFQN);
        result.setValidatorClass(validatorClass);
        result.setKeepJavaFiles(true);
        result.setOptimize(optimize);
        result.setJavacTargetDirectory(outClass);
        result.setTargetDirectory(outJava);
        result.setVerbose(verbose);
        result.setI18nable(i18nable);
        result.setAddLogger(addLogger);
        result.setProfile(profile);
        result.setResetAfterCompile(resetAfterCompile);
        result.setJaxxContextImplementorClass(jaxxContextImplementorClass);
        result.setExtraImports(extraImports);
        result.setDefaultErrorUI(defaultErrorUIClass);
        result.setUseUIManagerForIcon(useUIManagerForIcon);
        result.setDefaultDecoratorClass(defaultDecoratorClass);
        result.setGenerateHelp(generateHelp);
        result.setHelpBrokerFQN(helpBrokerFQN);
        result.setHelpsetTitleI18nSuffix(helpsetTitleI18nSuffix);
        result.setHelpsetIndexI18nSuffix(helpsetIndexI18nSuffix);
        result.setHelpsetTocI18nSuffix(helpsetTocI18nSuffix);
        result.setHelpSetName(helpSetName);
        result.setHelpsetI18nPrefix(helpsetI18nPrefix);

        if (cl != null) {
            result.setClassLoader(cl);
        }
        return result;
    }

    protected void fixCompileSourceRoots() {
//        //fixme should remove this silly test when we will make real maven plugin tests :)
//        if (project != null) {
//            if (!project.getCompileSourceRoots().contains(outJava.getPath())) {
//                project.addCompileSourceRoot(outJava.getPath());
//            }
//        }
        if (project == null) {
            // no project defined, can not fix anything
            // this case could appears if we wanted to do some tests of the plugin
            return;
        }

        if (testPhase) {
            if (!project.getTestCompileSourceRoots().contains(
                    outJava.getPath())) {
                getLog().info("Add test compile source root : " + outJava);
                project.addTestCompileSourceRoot(outJava.getPath());
            }
        } else {
            if (!project.getCompileSourceRoots().contains(outJava.getPath())) {
                getLog().info("Add compile source root : " + outJava);
                project.addCompileSourceRoot(outJava.getPath());
            }
        }
    }

    protected void printInit() {
        getLog().info(options.toString());
        getLog().info("includes : " + Arrays.toString(includes));
        for (String file : files) {
            getLog().info("will generate " + file);
        }

        ClassLoader threadLoader = Thread.currentThread().getContextClassLoader();
        getLog().info(threadLoader.toString());
        if (threadLoader.getClass().getSimpleName().equals("RealmClassLoader")) {
            try {
                java.lang.reflect.Method m = threadLoader.getClass().getDeclaredMethod("getURLs");
                m.setAccessible(true);
                URL[] urls = (URL[]) m.invoke(threadLoader);

                for (URL url : urls) {
                    getLog().info("url in class loader " + url);
                }
            } catch (Exception e) {
                getLog().warn("??? : " + e.getMessage(), e);
            }
        }

        //fixme should remove this silly test when we will make real maven plugin tests :)
        if (getPluginContext() != null) {
            for (Object e : getPluginContext().keySet()) {
                getLog().info("pluginContext " + e + " : " + getPluginContext().get(e));
            }
        }
    }

    protected void checkJaxxContextImplementorClass() {
        if (jaxxContextImplementorClass == null) {
            throw new IllegalArgumentException("jaxxContextImplementor can not be null");
        }
        try {
            Class<?> jaxxContextImplementor = Class.forName(jaxxContextImplementorClass);
            if (!JAXXContext.class.isAssignableFrom(jaxxContextImplementor)) {
                throw new IllegalArgumentException("jaxxContextImplementor '" + jaxxContextImplementor + "' does not implements " + JAXXContext.class);
            }
            if (Modifier.isAbstract(jaxxContextImplementor.getModifiers())) {
                throw new IllegalArgumentException("jaxxContextImplementor '" + jaxxContextImplementor + "' can not be abstract.");

            }
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("could not find jaxxContextImplementor class : " + jaxxContextImplementorClass);
        }
    }

    protected void generateHelp() throws IOException {
        Set<String> helpIds = HelpRootCompiledObjectDecorator.getHelpIds();
        if (helpIds.isEmpty()) {
            if (verbose) {
                // no ids detected in this compilation round
                getLog().info("no helpIds detected.");
            }
            return;
        }

        if (!helpIdStore.getParentFile().exists()) {
            helpIdStore.getParentFile().mkdirs();
        }

        Properties p = new Properties();

        for (String helpId : helpIds) {
            String path = helpId.replaceAll("\\.", File.separator);
            path = removeQuote(path) + ".html";
            p.put(removeQuote(helpId), path);
        }

        FileOutputStream w = new FileOutputStream(helpIdStore);

        try {
            p.store(w, null);
        } finally {
            w.close();
        }

        getLog().info("helpIdStore generated in " + helpIdStore);

        helpIds.clear();
    }

    @SuppressWarnings({"unchecked"})
    protected URLClassLoader initClassLoader(MavenProject project, org.apache.maven.plugin.logging.Log log) throws MalformedURLException {
        URLClassLoader loader = null;
        if (project != null) {

            URLClassLoader result;
            try {

                List<URL> lUrls = new ArrayList<URL>();
                List<String> sources = project.getCompileSourceRoots();

                if (addSourcesToClassPath) {
                    for (String source : sources) {
                        lUrls.add(new File(source).toURI().toURL());
                    }
                    if (testPhase) {
                        for (Object source : project.getTestCompileSourceRoots()) {
                            lUrls.add(new File(source.toString()).toURI().toURL());
                        }
                    }
                }
                if (addResourcesToClassPath) {
                    for (Object source : project.getResources()) {
                        Resource r = (Resource) source;
                        lUrls.add(new File(r.getDirectory()).toURI().toURL());
                    }
                }
                if (testPhase && addCompileClassPath) {
                    if (project != null) {
                        lUrls.add(new File(project.getBuild().getOutputDirectory()).toURI().toURL());
                    }
                }
                if (addProjectClassPath) {
                    getLog().info("use project compile scope class-path");
                    // add also all dependencies of the project in compile scope
                    Set<?> artifacts = project.getArtifacts();
                    for (Object o : artifacts) {
                        Artifact a = (Artifact) o;
                        lUrls.add(a.getFile().toURI().toURL());
                    }
                }

                result = new URLClassLoader(lUrls.toArray(new URL[lUrls.size()]), getClass().getClassLoader());

            } catch (IOException e) {
                throw new RuntimeException("Can't create ClassLoader for reason " + e.getMessage(), e);
            }
            loader = result;
        } else {
            List<URL> lUrls = new ArrayList<URL>();
            if (addSourcesToClassPath) {
                lUrls.add(src.toURI().toURL());
            }
            loader = new URLClassLoader(lUrls.toArray(new URL[lUrls.size()]), getClass().getClassLoader());
        }
        if (verbose) {
            for (URL entry : loader.getURLs()) {
                log.info("classpath : " + entry);
            }
        }
        return loader;
    }

    protected String removeQuote(String txt) {
        if (txt.startsWith("\"")) {
            txt = txt.substring(1);
        }
        if (txt.endsWith("\"")) {
            txt = txt.substring(0, txt.length() - 1);
        }
        return txt;
    }
}
