package org.nuiton.i18n.plugin.parser;

import org.apache.maven.plugin.logging.Log;
import org.nuiton.io.SortedProperties;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

/**
 * A thread to launch parser on files.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.2
 */
public class ParserThread extends Thread implements I18nParserConfiguration {

    /**
     * the incoming configuration (from mojo which contains shared result and
     * logger)
     */
    protected final I18nParserConfiguration configuration;

    /** queue of tasks to consume */
    protected final BlockingQueue<ParserTask> tasks;

    /** list of files consumed */
    protected final List<File> treatedFiles;

    /** list of files touched (says having at least one i18n key) */
    protected final List<File> touchedFiles;

    /** number of files registred to consume */
    protected int nbFiles;

    /**
     * object used to lock method {@link #terminatesAndWaits()} until the thread
     * is not fully terminated.
     */
    private final Object lock = new Object();

    /**
     * object used to lock in {@link #run()} until all tasks are consumed after
     * the shutdown of the executor
     */
    private final Object lock2 = new Object();

    public ParserThread(I18nParserConfiguration configuration) {
        super(ParserThread.class.getSimpleName());
        this.configuration = configuration;
        tasks = new LinkedBlockingQueue<ParserTask>();
        touchedFiles = new ArrayList<File>();
        treatedFiles = new ArrayList<File>();
    }

    @Override
    public boolean isVerbose() {
        return getConfiguration().isVerbose();
    }

    @Override
    public boolean isSilent() {
        return getConfiguration().isSilent();
    }

    @Override
    public boolean isShowTouchedFiles() {
        return getConfiguration().isShowTouchedFiles();
    }

    @Override
    public Log getLog() {
        return getConfiguration().getLog();
    }

    @Override
    public SortedProperties getResult() {
        return getConfiguration().getResult();
    }

    public List<File> getTreatedFiles() {
        return treatedFiles;
    }

    public List<File> getTouchedFiles() {
        return touchedFiles;
    }

    /**
     * Add a file to be consumed.
     *
     * @param parser the parser of the file
     * @param files  files to parse
     */
    public void addFile(FileParser parser, File... files) {

        for (File f : files) {

            nbFiles++;
            if (isVerbose()) {
                getLog().info("[" + nbFiles + "] " + f);
            }
            tasks.offer(new ParserTask(parser, f));
        }
    }

    /**
     * Ask the thread to stop.
     * <p/>
     * It will finish all incoming files (but will not accept more files to
     * parse)
     * <p/>
     * <b>Note:</b> The method does not return until all files are not
     * consumed.
     */
    public synchronized void terminatesAndWaits() {

        if (isVerbose()) {
            getLog().info("add terminate runner");
        }
        tasks.offer(new ParserTask(null, null));

        // we waits a little time to make sure, the thread receive the
        // sentinel, we can not use a lock here
        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            if (getLog().isErrorEnabled()) {
                getLog().error(e);
            }
        }

        if (getNbFilesToTreate() > 0) {

            // there is still something to treate,
            // MUST block until the thread is fully terminated

            if (isVerbose()) {
                getLog().info("block until ends...");
            }

            synchronized (lock) {
                try {
                    lock.wait();
                } catch (InterruptedException e) {
                    if (getLog().isErrorEnabled()) {
                        getLog().error(e);
                    }
                }
            }
        }

        if (getNbFilesToTreate() > 0) {
            // this case should never happens : once we are here, all files
            // MUST have been consumed
            throw new IllegalStateException(
                    "should have " + nbFiles + " files treated, but found " +
                    treatedFiles.size());
        }
        if (getLog().isDebugEnabled()) {
            getLog().debug("thread is terminated.");
        }
    }

    @Override
    public void run() {

        // create pool executor
        ExecutorService executor = createExecutor();

        boolean canQuit = false;

        while (!canQuit) {

            // waiting for a new file to treate
            ParserTask runner;
            try {
                runner = tasks.take();
            } catch (InterruptedException e) {
                if (getLog().isErrorEnabled()) {
                    getLog().error(e);
                }
                canQuit = true;
                continue;
            }

            if (runner.getFile() == null) {
                // ask to quit the tread
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Termination is asked..., will terminate to " +
                                   "resting consume files");
                }
                canQuit = true;
                continue;
            }

            if (getLog().isDebugEnabled()) {
                getLog().debug("consume " + runner);
            }
            executor.execute(runner);

            if (getLog().isDebugEnabled()) {
                getLog().debug("after consume " + runner);
            }
        }

        // ask executor to terminate
        executor.shutdown();

        if (nbFiles == 0) {
            // no file consumed
            if (isVerbose()) {
                getLog().info("No file consumed.");
            }
        } else {
            if (isVerbose()) {
                getLog().info("Will waits until all files (" + nbFiles +
                              ") are consumed (still " + getNbFilesToTreate() +
                              " file(s) to consume)");
            }
        }

        // waits end of executor

        canQuit = false;

        while (!canQuit) {

            int nbFilesToTreate = getNbFilesToTreate();

            // can quit if and only if executore is terminated are there is no
            // more files to consume
            // Note : normally, the second test is not necessary...
            canQuit = executor.isTerminated() || nbFilesToTreate == 0;

            if (canQuit) {
                continue;
            }

            if (getLog().isDebugEnabled()) {
                getLog().debug("Waiting end... (still " + nbFilesToTreate +
                               " file(s) to consume)");
            }

            // waits a task to be done
            synchronized (lock2) {
                try {
                    lock2.wait();
                } catch (InterruptedException e) {
                    getLog().error(e);
                    break;
                }
            }

            if (getLog().isDebugEnabled()) {
                getLog().debug("A task was consumed, still " + getNbFilesToTreate() +
                               " file(s) to treate.");
            }
        }

        if (isVerbose()) {
            getLog().info("Executor is terminated, will release thread lock.");
        }

        // service was shutdown and terminated, release thread main lock.

        synchronized (lock) {
            lock.notifyAll();
        }

    }

    /** clean internal state after usage of the thread. */
    public void clear() {
        treatedFiles.clear();
        touchedFiles.clear();
        nbFiles = 0;
    }

    /**
     * Creates the {@link ExecutorService} which will consumed i8n tasks.
     *
     * @return the instanciated executor
     */
    protected ExecutorService createExecutor() {

        // creates thread factory
        ThreadFactory factory =
                new ThreadFactory() {
                    final ThreadFactory defaultFactory =
                            Executors.defaultThreadFactory();

                    public Thread newThread(Runnable r) {
                        Thread thread =
                                defaultFactory.newThread(r);
                        thread.setName(ParserThread.this + "-" +
                                       thread.getName());
                        return thread;
                    }
                };

        // create pool executor
        ExecutorService executor = new ThreadPoolExecutor(
                8, 10,
                1L, TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                factory) {

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                try {
                    super.afterExecute(r, t);
                    ParserTask i18n = (ParserTask) r;
                    i18n.registerResult(treatedFiles, touchedFiles, getResult());
                } finally {

                    // task is done, notify thread

                    synchronized (lock2) {
                        lock2.notifyAll();
                    }
                }
            }
        };
        return executor;
    }

    protected I18nParserConfiguration getConfiguration() {
        return configuration;
    }

    protected int getNbFilesToTreate() {
        return nbFiles - treatedFiles.size();
    }

    /**
     * This is a task to parse a {@link #file}.
     * <p/>
     * The task will be executed in the executor service created in the thread.
     */
    class ParserTask implements Runnable {

        /** the file parser */
        protected final FileParser parser;

        /** the file to parse */
        protected final File file;

        /** starting time */
        protected long startingTime;

        /** ending time */
        protected long endingTime;

        ParserTask(FileParser parser, File file) {
            this.parser = parser;
            this.file = file;
        }

        @Override
        public void run() {
            startingTime = System.nanoTime();
            if (getLog().isDebugEnabled()) {
                getLog().debug("starting action for " + file);
            }
            try {

                parser.parseFile(file);

            } catch (IOException e) {
                if (getLog().isErrorEnabled()) {
                    getLog().error("could not parse file " + file, e);
                }
            } finally {

                if (getLog().isDebugEnabled()) {
                    getLog().debug("ending action for " + file);
                }
                endingTime = System.nanoTime();
            }
        }

        @Override
        public String toString() {
            return super.toString() + " - " + file;
        }

        protected File getFile() {
            return file;
        }

        protected long getDelay() {
            return endingTime - startingTime;
        }

        protected void destroy() {
            parser.destroy();
        }

        @Override
        protected void finalize() throws Throwable {
            super.finalize();
            destroy();
        }

        /**
         * Register the result of the parsing of the {@link #file} after {@link
         * #run()} method was invoked.
         * <p/>
         * This method should be invoked by the executor as an ending hook.
         *
         * @param treatedFiles list of files already treated
         * @param touchedFiles list of files already touched
         * @param result       shared result.
         */
        protected synchronized void registerResult(List<File> treatedFiles,
                                                   List<File> touchedFiles,
                                                   SortedProperties result) {

            try {
                treatedFiles.add(file);
                if (getLog().isDebugEnabled()) {
                    String delay = PluginHelper.convertTime(getDelay());
                    getLog().debug("[" + treatedFiles.size() + "] " + file +
                                   " in " + delay);
                }

                if (parser.isTouched()) {

                    // mark file as touched
                    touchedFiles.add(file);

                    if (isShowTouchedFiles()) {
                        getLog().info("touch " + file);
                    }
                    if (isVerbose()) {
                        String delay = PluginHelper.convertTime(getDelay());
                        getLog().info("[" + treatedFiles.size() + "] touchs " +
                                      file + " in " + delay);
                    }

                    // merge file result with
                    // merge result
                    result.putAll(parser.getResult());
                }
            } finally {

                // destroy runner
                destroy();
            }
        }
    }
}
