/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.nuiton.guix.tags;

//~--- non-JDK imports --------------------------------------------------------

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

//~--- JDK imports ------------------------------------------------------------

import java.util.HashMap;
import java.util.Map;

/**
 * Manages tags of the .guix files.
 */
public class TagManager {

    /**
     * Namespace for Guix's non-class tags, such as &lt;Button;&gt;.  The namespace normally does not
     * need to be specified but can be used to resolve ambiguities.
     */
    public static final String GUIX_NAMESPACE = "org.nuiton.guix.*";

    /** log */
    protected static final Log log = LogFactory.getLog(TagManager.class);

    /** List of the Guix classes such as &lt;Button;&gt; */
    protected static Map<String, Class> guixClassHandlers = new HashMap<String, Class>();

    /** Maps simple tag names to their default namespaces (package names). */
    private static Map<String, String> defaultNamespaces = new HashMap<String, String>();

    private TagManager() {    /* not instantiable */
    }

    /**
     * Sets the default namespace for a tag.  When the tag is encountered with no namespace specified,
     * the specified namespace will be assumed.  Mapping the same tag to two or more default namespaces
     * removes the mapping and marks the entry as being ambiguous (by putting a <code>null</code>
     * value into the map);  this causes an error to be thrown if the tag is used without a namespace being
     * specified.
     * <p/>
     * Java package names on tags are automatically converted into namespaces (e.g. &lt;javax.swing.JButton/&gt;
     * and &lt;JButton xmlns="javax.swing.*"/&gt; are equivalent), so tags with package names are considered
     * to have namespaces specified.
     *
     * @param tag       tag name
     * @param namespace namespace
     */
    public static void registerDefaultNamespace(String tag, String namespace) {
        if (defaultNamespaces.containsKey(tag) && (defaultNamespaces.get(tag) != null)
                &&!defaultNamespaces.get(tag).equals(namespace)) {
            defaultNamespaces.put(tag, null);    // tag name is now ambiguous
        } else {
            defaultNamespaces.put(tag, namespace);
        }
    }

    /**
     * Register a TagHandler for a particular tag
     *
     * @param className the tag or className to catch
     * @param clazz the tagHandler class who handles clazz
     */
    public static void registerGuixClassHandler(String className, Class clazz) {
        guixClassHandlers.put(className.toUpperCase(), clazz);
    }

    /**
     * Get the tagHandler registered for a particular tag
     *
     * @param className the class name or tag you'd like to get the TagHandler
     * @return a TagHandler instance which handles the className class or tag
     */
    public static TagHandler getGuixClassHandler(String className) {
        if(guixClassHandlers.get(className.toUpperCase()) != null) {
            try {
                return (TagHandler)guixClassHandlers.get(className.toUpperCase()).newInstance();
                            }
            catch (InstantiationException eee) {
            }
            catch (IllegalAccessException eee) {
            } 
        }
        else {
            for(Class clazz : guixClassHandlers.values()) {
                try {
                    TagHandler th = (TagHandler) clazz.newInstance();
                    if(th.getClassToGenerate().getName().equals(className)) {
                        return th;
                    }
                }
                catch (InstantiationException eee) {
                }
                catch (IllegalAccessException eee) {
                }
                catch (NullPointerException eee) {
                }
            }
        }
        return null;
    }

    /**
     * Resolves a simple class name (like <code>Object</code> or <code>String</code>) to its fully-qualified name.  Inner
     * classes should be represented as they would appear in Java source code (e.g. JPopupMenu.Separator).  Fully-qualified names,
     * such as <code>java.lang.Object</code> are legal and will be returned unmodified (and in fact it is generally impossible to
     * even know whether a given reference is fully qualified until it has been resolved).  Returns <code>null</code> if no matching
     * class could be found.
     *
     * @param name     name to resolve
     * @return the resolved fqn class name
     */
    public static String resolveClassName(String name) {
        if (name.endsWith("[]")) {
            return resolveClassName(name.substring(0, name.length() - 2)) + "[]";
        }

        if (name.indexOf("<") != -1) {
            name = name.substring(0, name.indexOf("<"));    // strip off generic types
        }
        
        name = name.intern();

        if (name.equals("boolean") || name.equals("byte") || name.equals("short") || name.equals("int")
                || name.equals("long") || name.equals("float") || name.equals("double") || name.equals("char")) {
            return name;
        }
        else if(name.equals("Boolean") || name.equals("Byte") || name.equals("Short") || name.equals("Integer") || name.equals("String")
                || name.equals("Long") || name.equals("Float") || name.equals("Double") || name.equals("Character")) {
            return "java.lang." + name;
        }

        if (guixClassHandlers.containsKey(name)) {
            return name;
        }

        // if the name of the tag is the fully qualified class name
        if (name.lastIndexOf('.') >= 0) {
            return name;
        }

        return null;
    }
}

