/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.tags;

import java.beans.BeanInfo;
import java.beans.EventSetDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * Default TagHandler, containing common methods for all the tags
 *
 * @author kmorin
 */
public abstract class DefaultTagHandler implements TagHandler {

    /** Maps the common name for an attribute with its real name for the class to generate */
    protected Map<String,String> attrMap = new HashMap<String,String>();

    /** Maps property names to their respective ProxyEventInfos. */
    private Map<String, ProxyEventInfo> eventInfos;

    /** The JAXXBeanInfo for the beanClass. */
    protected BeanInfo beanInfo;
    
    /** Maps property names to their respective JAXXPropertyDescriptors. */
    private Map<String, PropertyDescriptor> properties;

    /** Maps event names to their respective JAXXEventSetDescriptors. */
    private Map<String, EventSetDescriptor> events;

    /**
     * Encapsulates information about a "proxy event handler", which is an event handler that
     * fires PropertyChangeEvents when it is triggered.  ProxyEventInfos simplify the data binding
     * system by allowing all dependencies to fire the same kind of event even if they would
     * normally throw something else, like <code>DocumentEvent</code>.
     */
    private class ProxyEventInfo {
        /** The name of the method or field being proxied, e.g. "getText". */
        String memberName;

        /** The "actual" event listener for the property in question, e.g. DocumentListener. */
        Class listenerClass;

        /**
         * In cases where a different object (such as a model) is more directly responsible for
         * managing the property, this is the name of the property where that object can be
         * found, e.g. "document" (which is turned into a call to "getDocument()").  This property
         * is also treated as a dependency of the data binding expression, and any updates to it
         * (assuming it is bound) will cause the listener to be removed from the old value and
         * attached to the new value, and the data binding to be processed.
         */
        String modelName;

        /** The name of the method used to add the "native" event listener, e.g. "addDocumentListener". */
        String addMethod;

        /** The name of the method used to remove the "native" event listener, e.g. "removeDocumentListener". */
        String removeMethod;
    }

    /**
     * Performs introspection on the beanClass and stores the results.
     *
     * @throws java.beans.IntrospectionException
     *          TODO
     */
    protected void init() throws IntrospectionException {
        if (beanInfo == null) {
            // perform introspection & cache the results
            beanInfo = Introspector.getBeanInfo(getClassToGenerate());

            PropertyDescriptor[] propertiesArray = beanInfo.getPropertyDescriptors();
            properties = new HashMap<String, PropertyDescriptor>();
            for (int i = propertiesArray.length - 1; i >= 0; i--) {
                properties.put(propertiesArray[i].getName(), propertiesArray[i]);
            }

            EventSetDescriptor[] eventsArray = beanInfo.getEventSetDescriptors();
            events = new HashMap<String, EventSetDescriptor>();
            for (int i = eventsArray.length - 1; i >= 0; i--) {
                Method[] methods = eventsArray[i].getListenerMethods();
                for (Method method : methods) {
                    events.put(method.getName(), eventsArray[i]);
                }
            }

            configureProxyEventInfo();
        }
    }

    /**
     * Configures the event handling for members which do not fire <code>PropertyChangeEvent</code> when
     * modified.  The default implementation does nothing.  Subclasses should override this method to call
     * <code>addProxyEventInfo</code> for each member which requires special handling.
     */
    protected void configureProxyEventInfo() {
        
    }


    /**
     * Configures a proxy event handler which fires <code>PropertyChangeEvents</code> when a non-bound
     * member is updated.  This is necessary for all fields (which cannot be bound) and for methods that are
     * not bound property <code>get</code> methods.  The proxy event handler will attach the specified kind
     * of listener to the class and fire a <code>PropertyChangeEvent</code> whenever the listener receives
     * any kind of event.
     * <p/>
     * Even though this method can theoretically be applied to fields (in addition to methods), it would be an
     * unusual situation in which that would actually work -- as fields cannot fire events when modified, it would
     * be difficult to have a listener that was always notified when a field value changed.
     *
     * @param memberName    the name of the field or method being proxied
     * @param listenerClass the type of listener which receives events when the field or method is updated
     */
    public void addProxyEventInfo(String memberName, Class listenerClass) {
        addProxyEventInfo(memberName, listenerClass, null);
    }


    /**
     * Configures a proxy event handler which fires <code>PropertyChangeEvents</code> when a non-bound
     * member is updated.  This is necessary for all fields (which cannot be bound) and for methods that are
     * not bound property <code>get</code> methods.  This variant attaches a listener to a property of the
     * object (such as <code>model</code>) and not the object itself, which is useful when there is a model
     * that is the "real" container of the information.  The proxy event handler will attach the specified kind
     * of listener to the property's value (retrieved using the property's <code>get</code> method) and fire
     * a <code>PropertyChangeEvent</code> whenever the listener receives any kind of event.
     * <p/>
     * If the property is itself bound (typically the case with models), any updates to the property's value will
     * cause the listener to be removed from the old property value and reattached to the new property value,
     * as well as cause a <code>PropertyChangeEvent</code> to be fired.
     * <p/>
     * Even though this method can theoretically be applied to fields (in addition to methods), it would be an
     * unusual situation in which that would actually work -- as fields cannot fire events when modified, it would
     * be difficult to have a listener that was always notified when a field value changed.
     *
     * @param memberName    the name of the field or method being proxied
     * @param listenerClass the type of listener which receives events when the field or method is updated
     * @param modelName     the JavaBeans-style name of the model property
     */
    public void addProxyEventInfo(String memberName, Class listenerClass, String modelName) {
        String listenerName = listenerClass.getName();
        listenerName = listenerName.substring(listenerName.lastIndexOf(".") + 1);
        addProxyEventInfo(memberName, listenerClass, modelName, "add" + listenerName, "remove" + listenerName);
    }

    /**
     * Configures a proxy event handler which fires <code>PropertyChangeEvents</code> when a non-bound
     * member is updated.  This is necessary for all fields (which cannot be bound) and for methods that are
     * not bound property <code>get</code> methods.  This variant attaches a listener to a property of the
     * object (such as <code>model</code>) and not the object itself, which is useful when there is a model
     * that is the "real" container of the information.  The proxy event handler will attach the specified kind
     * of listener to the property's value (retrieved using the property's <code>get</code> method) and fire
     * a <code>PropertyChangeEvent</code> whenever the listener receives any kind of event.
     * <p/>
     * If the property is itself bound (typically the case with models), any updates to the property's value will
     * cause the listener to be removed from the old property value and reattached to the new property value,
     * as well as cause a <code>PropertyChangeEvent</code> to be fired.
     * <p/>
     * This variant of <code>addProxyEventInfo</code> allows the names of the methods that add and remove
     * the event listener to be specified, in cases where the names are not simply <code>add&lt;listenerClassName&gt;</code>
     * and <code>remove&lt;listenerClassName&gt;</code>.
     * <p/>
     * Even though this method can theoretically be applied to fields (in addition to methods), it would be an
     * unusual situation in which that would actually work -- as fields cannot fire events when modified, it would
     * be difficult to have a listener that was always notified when a field value changed.
     *
     * @param memberName    the name of the field or method being proxied
     * @param listenerClass the type of listener which receives events when the field or method is updated
     * @param modelName     the JavaBeans-style name of the model property
     * @param addMethod     add method name
     * @param removeMethod  remove method name
     */
    public void addProxyEventInfo(String memberName, Class listenerClass,
                                  String modelName, String addMethod, String removeMethod) {
        ProxyEventInfo info = new ProxyEventInfo();
        info.memberName = memberName;
        info.listenerClass = listenerClass;
        info.modelName = modelName;
        info.addMethod = addMethod;
        info.removeMethod = removeMethod;
        if (eventInfos == null) {
            eventInfos = new HashMap<String, ProxyEventInfo>();
        }
        eventInfos.put(memberName, info);
    }

    @Override
    public boolean hasEventInfosAboutMethod(String methodName) {
        return eventInfos != null && eventInfos.get(methodName) != null;
    }

    @Override
    public Class getEventInfosListenerClass(String methodName) {
        return eventInfos.get(methodName) != null ? eventInfos.get(methodName).listenerClass : null;
    }

    @Override
    public String getEventInfosAddListenerMethodName(String methodName) {
        return eventInfos.get(methodName) != null ? eventInfos.get(methodName).addMethod : null;
    }

    @Override
    public String getEventInfosRemoveListenerMethodName(String methodName) {
        return eventInfos.get(methodName) != null ? eventInfos.get(methodName).removeMethod : null;
    }

    @Override
    public String getEventInfosModelName(String methodName) {
        return eventInfos.get(methodName) != null ? eventInfos.get(methodName).modelName : null;
    }

    @Override
    public String getAttrToGenerate(String attr) {
        return attrMap.get(attr);
    }

    @Override
    public String getDefaultConstructor() {
        return null;
    }
  
}
