/**
 * *##% guix-compiler-gwt
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.tags.gwt;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.guix.model.AttributeDescriptor;
import org.nuiton.guix.model.GuixModelObject;

/**
 * Represents a table
 *
 * @author kmorin
 */
public class TableHandler extends UIObjectHandler {

    private static Log log = LogFactory.getLog(TableHandler.class);

    protected enum VerticalAlignment {

        ALIGN_TOP, ALIGN_MIDDLE, ALIGN_BOTTOM
    }
    protected enum HorizontalAlignment {

        ALIGN_LEFT, ALIGN_CENTER, ALIGN_RIGHT, ALIGN_DEFAULT
    }
    public static final String DEFAULT_INSETS = "3, 3, 3, 3";
    public static final HorizontalAlignment DEFAULT_HORIZONTAL_ALIGNMENT = HorizontalAlignment.ALIGN_DEFAULT;
    public static final VerticalAlignment DEFAULT_VERTICAL_ALIGNMENT = VerticalAlignment.ALIGN_TOP;
    public static final String DEFAULT_FILL = "both";
    public static final Integer DEFAULT_IPADX = 1;
    public static final Integer DEFAULT_IPADY = 1;
    public static final Double DEFAULT_WEIGHTX = 0.0;
    public static final Double DEFAULT_WEIGHTY = 0.0;
    protected Double weightx = null;
    protected Double weighty = null;
    protected Integer ipadx = null;
    protected Integer ipady = null;
    protected HorizontalAlignment horizontalAlignment = null;
    protected VerticalAlignment verticalAlignment = null;
    protected String fill = null;
    protected String insets = null;

    /**
     * Constructor
     */
    public TableHandler() {
        super();
    }

    /**
     * Constructor
     *
     * @param table the GuixModelObject which represents the table
     */
    public TableHandler(GuixModelObject table) {
        for (AttributeDescriptor attr : table.getAttributeDescriptors()) {
            if (attr.getName().equalsIgnoreCase("weightx")) {
                weightx = Double.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("weighty")) {
                weighty = Double.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("ipadx")) {
                ipadx = Integer.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("ipady")) {
                ipady = Integer.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("horizontalAlignment")) {
                horizontalAlignment = getHalign(attr.getValue());
                if (horizontalAlignment == null) {
                    if (log.isErrorEnabled()) {
                        log.error(table.getId() + "'s horizontalAlignment attribute has a wrong value : " + attr.getValue());
                    }
                    horizontalAlignment = HorizontalAlignment.ALIGN_DEFAULT;
                }
            }
            else if (attr.getName().equalsIgnoreCase("verticalAlignment")) {
                verticalAlignment = getValign(attr.getValue());
                if (verticalAlignment == null) {
                    if (log.isErrorEnabled()) {
                        log.error(table.getId() + "'s verticalAlignment attribute has a wrong value : " + attr.getValue());
                    }
                    verticalAlignment = VerticalAlignment.ALIGN_TOP;
                }
            }
            else if (attr.getName().equalsIgnoreCase("fill")) {
                fill = attr.getValue();
            }
            else if (attr.getName().equalsIgnoreCase("insets")) {
                insets = attr.getValue();
            }
        }
    }


    public HorizontalAlignment getHorizontalAlignment() {
        return horizontalAlignment;
    }

    public void setHorizontalAlignment(HorizontalAlignment horizontalAlignment) {
        this.horizontalAlignment = horizontalAlignment;
    }

    public VerticalAlignment getVerticalAlignment() {
        return verticalAlignment;
    }

    public void setVerticalAlignment(VerticalAlignment verticalAlignment) {
        this.verticalAlignment = verticalAlignment;
    }

    public String getFill() {
        return fill;
    }

    public void setFill (String fill) {
        this.fill = fill;
    }

    public String getInsets() {
        return insets;
    }

    public void setInsets(String insets) {
        this.insets = insets;
    }

    public Integer getIpadx() {
        return ipadx;
    }

    public void setIpadx(Integer ipadx) {
        this.ipadx = ipadx;
    }

    public Integer getIpady() {
        return ipady;
    }

    public void setIpady(Integer ipady) {
        this.ipady = ipady;
    }

    public Double getWeightx() {
        return weightx;
    }

    public void setWeightx(Double weightx) {
        this.weightx = weightx;
    }

    public Double getWeighty() {
        return weighty;
    }

    public void setWeighty(Double weighty) {
        this.weighty = weighty;
    }

    public HorizontalAlignment getHalign(String halign) {
        if (halign.equalsIgnoreCase("left")) {
            return HorizontalAlignment.ALIGN_LEFT;
        }
        if (halign.equalsIgnoreCase("right")) {
            return HorizontalAlignment.ALIGN_RIGHT;
        }
        if (halign.equalsIgnoreCase("center")) {
            return HorizontalAlignment.ALIGN_CENTER;
        }
        return null;
    }

    public VerticalAlignment getValign(String valign) {
        if (valign.equalsIgnoreCase("top")) {
            return VerticalAlignment.ALIGN_TOP;
        }
        if (valign.equalsIgnoreCase("bottom")) {
            return VerticalAlignment.ALIGN_BOTTOM;
        }
        if (valign.equalsIgnoreCase("middle")) {
            return VerticalAlignment.ALIGN_MIDDLE;
        }
        return null;
    }
    
    /**
     * Creates a cell with the right column and generates the code to add it to the table
     *
     * @param parentId the table id
     * @param cl the CellHandler containing the infos
     * @param layout the matrix representing the free spaces in the table
     * @param iC the last column where a cell was added in the row
     * @param r the row of the cell
     * @return the code to add the cell in the table
     */
    public static String createCell(String parentId, CellHandler cl, List<List<int[]>> layout, int iC, int r) {
        StringBuffer result = new StringBuffer();
        //is the column defined by the user ?
        boolean columnSetByUser = true;
        boolean rowSetByUser = true;
        //FIXME
        if (cl.getColumn() == -1) {
            cl.setColumn(iC);
            columnSetByUser = false;
        }
        else if (cl.getColumn() < iC) {
            if (log.isErrorEnabled()) {
                log.error("Error cell column number");
            }
        }
        if(cl.getRow() == -1) {
            cl.setRow(r);
            rowSetByUser = false;
        }
        else if(rowSetByUser && cl.getRow() < r) {
            if(log.isErrorEnabled()) {
                log.error("Error cell row number");
            }
        }
        if(columnSetByUser) {
            iC = cl.getColumn();
        }
        //if the row has not been added yet to layout
        if (r >= layout.size()) {
            for (int i = layout.size(); i <= r; i++) {
                layout.add(new ArrayList<int[]>());
            }
        }
        //is there enough place to put the cell at column iC ?
        boolean enoughPlace = false;
        while (!enoughPlace) {
            enoughPlace = true;
            //while the column iC is occupied, go to next column
            while (layout.get(r).size() > iC && layout.get(r).get(iC) != null) {
                if (columnSetByUser && cl.getColumn() == iC) {
                    if (log.isErrorEnabled()) {
                        log.error("This column is already occupied.");
                    }
                }
                iC = iC + layout.get(r).get(iC)[1];
            }
            //if there is a free space
            if (layout.get(r).size() > iC) {
                int j = 0;
                enoughPlace = layout.get(r).get(iC)[1] < cl.getColumns();
                if (!enoughPlace) {
                    iC = iC + layout.get(r).get(iC)[1];
                }
            }
        }
        cl.setColumn(iC);
        //for each row that must contain the cell
        layout.get(r).add(new int[]{cl.getRows(), cl.getColumns()});
        
        result.append(parentId).append(".setWidget(")
                .append(cl.getRow()).append(",")
                .append(cl.getColumn()).append(",")
                .append(cl.getId()).append(");\n");
        if(cl.getRows() > 1)
            result.append(parentId).append(".getFlexCellFormatter().setRowSpan(")
                .append(cl.getRow()).append(",")
                .append(cl.getColumn()).append(",")
                .append(cl.getRows()).append(");\n");
        if(cl.getColumns() > 1)
            result.append(parentId).append(".getFlexCellFormatter().setColSpan(")
                .append(cl.getRow()).append(",")
                .append(cl.getColumn()).append(",")
                .append(cl.getColumns()).append(");\n");
        result.append(parentId).append(".getFlexCellFormatter().setHorizontalAlignment(")
                .append(cl.getRow()).append(",")
                .append(cl.getColumn()).append(",")
                .append("com.google.gwt.user.client.ui.HasHorizontalAlignment.")
                .append(cl.getHorizontalAlignment()).append(");\n");
        result.append(parentId).append(".getFlexCellFormatter().setVerticalAlignment(")
                .append(cl.getRow()).append(",")
                .append(cl.getColumn()).append(",")
                .append("com.google.gwt.user.client.ui.HasVerticalAlignment.")
                .append(cl.getVerticalAlignment()).append(");\n");
        result.append(parentId)
                .append(".getFlexCellFormatter().addStyleName(")
                .append(cl.getRow()). append(",")
                .append(cl.getColumn()).append(", \"Cell\");\n");
        return result.toString();
    }

    @Override
    public Class getClassToGenerate() {
        return com.google.gwt.user.client.ui.FlexTable.class;
    }

}
