package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * %%
 * Copyright (C) 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.stereotype.StereotypeDefinition;
import org.nuiton.eugene.models.stereotype.StereotypeDefinitionProvider;
import org.nuiton.eugene.models.stereotype.Stereotypes;

/**
 * Defines all stereotypes managed by Java templates.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.models.stereotype.StereotypeDefinitionProvider" role-hint="eugene-java-templates"
 * @since 2.5.6
 */
public class JavaTemplatesStereoTypes extends StereotypeDefinitionProvider {

    /**
     * Stereotype for JavaBean objects to place on a classifier.
     *
     * @see #hasBeanStereotype(ObjectModelClassifier, ObjectModelPackage)
     * @see #hasBeanStereotype(ObjectModelPackage)
     * @since 2.5.6
     */
    @StereotypeDefinition(
            target = {ObjectModelClassifier.class, ObjectModelPackage.class},
            documentation = "To specify that a class is a JavaBean")
    public static final String STEREOTYPE_BEAN = "bean";

    /**
     * Tag value to authorize user to generate some bean with methods, some generators won't generate them : Lots code).
     *
     * By default, user should never add methods in bean classes, simply write them in your java code!.
     *
     * You can globaly use it on the complete model, on packages, or to a specific classifier.
     *
     * @see #isAcceptBeanWithMethods(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 3.0
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To authorize to generate bean with method (in some generator such method won't be generated).")
    public static final String STEREOTYPE_ACCEPT_BEAN_WITH_METHODS = "acceptBeanWithMethods";

    /**
     * Tag value to authorize user to override abstract classes.
     *
     * By default, user should never override abstract classes but works on implementation ones.
     *
     * You can globaly use it on the complete model, on packages, or to a specific classifier.
     *
     * @see #isOverrideAbstractClasses(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 3.0
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To authorize to override abstract classes.")
    public static final String STEREOTYPE_OVERRIDE_ABSTRACT_CLASSES = "overrideAbstractClasses";

    /**
     * Tag value to generate property change support on generated beans.
     *
     * You can globaly use it on the complete model, on packages, or to a specific classifier.
     *
     * @see #isSkipGeneratePropertyChangeSupport(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 2.12
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To generate property change support code.")
    public static final String STEREOTYPE_SKIP_GENERATE_PROPERTY_CHANGE_SUPPORT = "generatePropertyChangeSupport";

    /**
     * Tag value to generate lazy instanciation of any collection to avoid NPEs.
     *
     * You can globaly use it on the complete model or a package, or to a specific classifier.
     *
     * @see #isSkipGenerateNotEmptyCollections(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 2.12
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To lazy instanciate collections to avoid NPE codes.")
    public static final String STEREOTYPE_SKIP_GENERATE_NOT_EMPTY_COLLECTIONS = "generateNotEmptyCollections";

    /**
     * To generate an interface of each simple bean.
     *
     * You can globaly use it on the complete model or to a specific classifier.
     *
     * @see #isSimpleBeanSkipGenerateInterface(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 2.6.2
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To generate an interface for each bean " +
                            "for a class or any class of a model.\n" +
                            "(only effective with simple bean generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_INTERFACE = "simpleBeanGenerateInterface";

    /**
     * To generate a factory of generated simple beans.
     *
     * You must use it on the complete model.
     *
     * @see #isSimpleBeanSkipGenerateFactory(ObjectModel)
     * @since 2.6.2
     */
    @StereotypeDefinition(
            target = {ObjectModel.class},
            documentation = "To generate a factory of simple bean.\n" +
                            "(only effective with simple bean generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_FACTORY = "simpleBeanGenerateFactory";

    /**
     * To generate a factory of generated simple beans.
     *
     * You must use it on the complete model.
     *
     * @see #isSimpleBeanSkipGenerateDefaults(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 2.7.2
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To generate defaults class with simple operations on the type.\n" +
                            "(only effective with simple bean generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_DEFAULTS = "simpleBeanGenerateDefaults";

    /**
     * To generate a factory of generated simple beans.
     *
     * You must use it on the complete model.
     *
     * @see #isSimpleBeanWithNoInterfaceSkipGenerateDefaults(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 3.0
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To skip generation of defaults class with simple operations on the type.\n" +
                            "(only effective with simple bean generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_DEFAULTS = "simpleBeanWithNoInterfaceGenerateDefaults";


    /**
     * To generate or not guava predicates on each property of the bean.
     *
     * You can globaly use it on the complete model or to a specific classifier.
     *
     * @see #isSimpleBeanWithNoInterfaceSkipGeneratePredicates(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 3.0
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To generate or not guava predicates on propertyes on beans.\n" +
                            "(only effective with simple bean with no interface generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_PREDICATES = "simpleBeanWithNoInterfaceGeneratePredicates";

    /**
     * To generate or not guava functions on each property of the bean.
     *
     * You can globaly use it on the complete model or to a specific classifier.
     *
     * @see #isSimpleBeanWithNoInterfaceSkipGenerateFunctions(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
     * @since 3.0
     */
    @StereotypeDefinition(
            target = {ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class},
            documentation = "To generate or not guava predicates on propertyes on beans.\n" +
                            "(only effective with simple bean with no interface generator)")
    public static final String STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_FUNCTIONS = "simpleBeanWithNoInterfaceGenerateFunctions";

    /**
     * Check if the given classifier has the {@link #STEREOTYPE_BEAN} stereotype.
     *
     * @param classifier classifier to test
     * @return {@code true} if stereotype was found, {@code false otherwise}
     * @see #STEREOTYPE_BEAN
     */
    public static boolean hasBeanStereotype(ObjectModelClassifier classifier, ObjectModelPackage aPackage) {
        return Stereotypes.findDirectStereotype(STEREOTYPE_BEAN, classifier, aPackage);
    }

    /**
     * Check if the given aPackage has the {@link #STEREOTYPE_BEAN} stereotype.
     *
     * @param aPackage classifier to test
     * @return {@code true} if stereotype was found, {@code false otherwise}
     * @see #STEREOTYPE_BEAN
     */
    public static boolean hasBeanStereotype(ObjectModelPackage aPackage) {
        return Stereotypes.findDirectStereotype(STEREOTYPE_BEAN, aPackage);
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_ACCEPT_BEAN_WITH_METHODS} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_ACCEPT_BEAN_WITH_METHODS
     * @since 3.0
     */
    public boolean isAcceptBeanWithMethods(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_ACCEPT_BEAN_WITH_METHODS, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_OVERRIDE_ABSTRACT_CLASSES} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_OVERRIDE_ABSTRACT_CLASSES
     * @since 3.0
     */
    public boolean isOverrideAbstractClasses(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_OVERRIDE_ABSTRACT_CLASSES, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SKIP_GENERATE_PROPERTY_CHANGE_SUPPORT} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SKIP_GENERATE_PROPERTY_CHANGE_SUPPORT
     * @since 2.12
     */
    public boolean isSkipGeneratePropertyChangeSupport(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SKIP_GENERATE_PROPERTY_CHANGE_SUPPORT, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SKIP_GENERATE_NOT_EMPTY_COLLECTIONS} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SKIP_GENERATE_NOT_EMPTY_COLLECTIONS
     * @since 2.12
     */
    public boolean isSkipGenerateNotEmptyCollections(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SKIP_GENERATE_NOT_EMPTY_COLLECTIONS, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_INTERFACE} tag value on the given model or classifier.
     *
     * It will first look on the model, and then in the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_INTERFACE
     * @since 2.3
     */
    public boolean isSimpleBeanSkipGenerateInterface(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_INTERFACE, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_FACTORY} tag value on the given model or classifier.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_FACTORY
     * @since 2.7.2
     */
    public boolean isSimpleBeanSkipGenerateFactory(ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_FACTORY, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_DEFAULTS} tag value on the given model or classifier.
     *
     * It will first look on the model, and then in the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_DEFAULTS
     * @since 2.7.2
     */
    public boolean isSimpleBeanSkipGenerateDefaults(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_SKIP_GENERATE_DEFAULTS, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_DEFAULTS} tag value on the given model or classifier.
     *
     * It will first look on the model, and then in the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_DEFAULTS
     * @since 3.0
     */
    public boolean isSimpleBeanWithNoInterfaceSkipGenerateDefaults(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_DEFAULTS, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_PREDICATES} tag value on the given model or classifier.
     *
     * It will first look on the model, and then in the given classifier.
     *
     * <strong>If not filled, then use default {@code s} value.</strong>
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_PREDICATES
     * @since 3.0
     */
    public boolean isSimpleBeanWithNoInterfaceSkipGeneratePredicates(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_PREDICATES, classifier, aPackage, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_FUNCTIONS} tag value on the given model or classifier.
     *
     * It will first look on the model, and then in the given classifier.
     *
     * <strong>If not filled, then use default {@code s} value.</strong>
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_FUNCTIONS
     * @since 3.0
     */
    public boolean isSimpleBeanWithNoInterfaceSkipGenerateFunctions(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        boolean value = Stereotypes.findDirectStereotype(STEREOTYPE_SIMPLE_BEAN_WITH_NO_INTERFACE_SKIP_GENERATE_FUNCTIONS, classifier, aPackage, model);
        return value;
    }


}
