package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * $Id: SimpleJavaBeanTransformer.java 1302 2013-09-29 14:38:04Z tchemit $
 * $HeadURL: https://svn.nuiton.org/eugene/tags/eugene-2.8/eugene-java-templates/src/main/java/org/nuiton/eugene/java/SimpleJavaBeanTransformer.java $
 * %%
 * Copyright (C) 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */




import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorException;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * SimpleJavaBeanTransformer generates simple bean with pcs support
 * (and nothing else) according to the JavaBeans 1.1 norm with no Impl generation mecanism.
 * <p/>
 * So if there is so operation described on model, you should use the
 * {@link JavaBeanTransformer} instead.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.SimpleJavaBeanTransformer"
 * @since 2.6
 */
public class SimpleJavaBeanTransformer extends AbstractJavaBeanTransformer {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SimpleJavaBeanTransformer.class);

    @Override
    public void transformFromModel(ObjectModel model) {

        String className = model.getName() + "BeanFactory";

        if (canGenerateFactory(model, className)) {

            generateBeanFactory(model, className);
        }
    }

    @Override
    public void transformFromClass(ObjectModelClass input) {

        if (!JavaTemplatesGeneratorUtil.hasBeanStereotype(input)) {

            //  not a bean
            return;
        }

        String interfaceName = getBeanInterfaceName(input);
        String className = getBeanClassName(input);
        String defaultClassName = getBeanDefaultsClassName(input);

        boolean generateDefaults = canGenerateBeanDefaults(input);

        boolean generateInterface = canGenerateInterface(input, interfaceName);

        boolean generateClass = canGenerateBean(input, className);

        if (generateClass || generateInterface) {

            ObjectModelClass outputClass = generateBeanClass(input,
                                                             className,
                                                             interfaceName);
            if (generateInterface) {

                generateBeanInterface(input, interfaceName, outputClass);

                if (generateClass) {

                    // add override annotation on all public operations

                    for (ObjectModelOperation operation : getPublicOperations(outputClass)) {
                        addAnnotation(outputClass,
                                      operation,
                                      Override.class);
                    }

                }
            }

            if (!generateClass) {

                // remove bean class from builder (just used to build the interface)
                builder.getModel().getClasses().remove(outputClass);
            }
        }

        if (generateDefaults) {

            generateBeanDefaults(input, defaultClassName);
        }
    }

    protected boolean canGenerateFactory(ObjectModel model, String className) {

        boolean generateFactory =
                JavaTemplatesGeneratorUtil.isSimpleBeanGenerateFactory(model);

        String defaultPackage =
                getConfiguration().getProperty(PROP_DEFAULT_PACKAGE);

        String fqn = defaultPackage + "." + className;
        boolean canGenerate = generateFactory &&
                              !isInClassPath(fqn);

        return canGenerate;
    }

    protected boolean canGenerateInterface(ObjectModelClass input,
                                           String className) {

        boolean generateInterface =
                JavaTemplatesGeneratorUtil.isSimpleBeanGenerateInterface(model, input);

//        boolean generateInterface = generateTagValue != null &&
//                                    Boolean.valueOf(generateTagValue);
//
        String fqn = input.getPackageName() + "." + className;

        boolean canGenerate = generateInterface && !isInClassPath(fqn);

        if (canGenerate) {

            // check there is no operation on input
            if (!input.getOperations().isEmpty()) {

                throw new GeneratorException(
                        "Can't generate a simple bean interface as class " +
                        fqn +
                        " contains so operations." +
                        "\nUse instead the JavaBeanTransformer.");
            }
        }
        return canGenerate;
    }

    protected boolean canGenerateBean(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean canGenerate = !isInClassPath(fqn);

        if (canGenerate) {

            // check there is no operation on input
            if (!input.getOperations().isEmpty()) {

                throw new GeneratorException(
                        "Can't generate a simple bean as class " +
                        fqn +
                        " contains so operations." +
                        "\nUse instead the JavaBeanTransformer.");
            }
        }
        return canGenerate;
    }

    protected boolean canGenerateBeanDefaults(ObjectModelClass input) {

        boolean withInput = input != null;

        boolean canGenerate =
                JavaTemplatesGeneratorUtil.isSimpleBeanGenerateDefaults(
                        model, input);

        if (canGenerate) {

            if (withInput) {

                // class in not abstract
                // class is a bean

                canGenerate = !input.isAbstract() &&
                              JavaTemplatesGeneratorUtil.hasBeanStereotype(input);

            }
        }

        return canGenerate;
    }

    protected boolean canGenerateBeanAbstractDefaults(ObjectModelClass input,
                                                      String className) {

        String classPackage = input.getPackageName();

        String fqn = classPackage + "." + className;

        boolean canGenerate = !isInClassPath(fqn);

        return canGenerate;
    }

    protected void generateBeanFactory(ObjectModel model, String className) {
        String defaultPackage =
                getConfiguration().getProperty(PROP_DEFAULT_PACKAGE);

        ObjectModelClass output = createClass(className, defaultPackage);

        for (ObjectModelClass aClass : model.getClasses()) {
            if (!aClass.isAbstract() &&
                JavaTemplatesGeneratorUtil.hasBeanStereotype(aClass)) {
                String packageName = aClass.getPackageName();
                String typeName = getBeanInterfaceName(aClass);
                String typeBeanName = getBeanClassName(aClass);
                addImport(output, packageName + "." + typeName);
                addImport(output, packageName + "." + typeBeanName);
                ObjectModelOperation operation = addOperation(
                        output,
                        "typeOf" + typeName,
                        "<BeanType extends " + typeName + "> Class<BeanType>",
                        ObjectModelJavaModifier.STATIC,
                        ObjectModelJavaModifier.PUBLIC
                );
                setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeBeanName+".class;\n"
+"    "
                );
                operation = addOperation(
                        output,
                        "new" + typeName,
                        typeName,
                        ObjectModelJavaModifier.STATIC,
                        ObjectModelJavaModifier.PUBLIC
                );
                setOperationBody(operation, ""
    +"\n"
+"        return new "+typeBeanName+"();\n"
+"    "
                );
            }
        }
    }

    protected ObjectModelInterface generateBeanInterface(ObjectModelClass input,
                                                         String className,
                                                         ObjectModelClass outputClass) {

        ObjectModelInterface output =
                createInterface(className, input.getPackageName());

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean superClassIsBean = false;

        String superClass = null;

        // test if a super class has bean stereotype

        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesGeneratorUtil.hasBeanStereotype(superclass)) {
                    superClassIsBean = true;
                    superClass = superclass.getPackageName() + "." +
                                 getBeanInterfaceName(superclass);
                    break;
                }
                superClass = superclass.getQualifiedName();
            }
        }

        if (superClass == null) {

            // try to find a super class by tag-value
            superClass =
                    JavaTemplatesGeneratorUtil.getSimpleBeanInterfaceSuperClassTagValue(
                            model, input);
        }

        boolean serializableFound = addInterfaces(input, output, superClass);

        if (serializableFound || superClassIsBean) {
            addInterface(output, Serializable.class);
        }

        generateI18nBlockAndConstants(input, output);

        for (ObjectModelOperation operation : getPublicOperations(outputClass)) {
            cloneOperation(operation, output, true);
        }

        return output;
    }

    protected ObjectModelClass generateBeanClass(ObjectModelClass input,
                                                 String className,
                                                 String interfaceName) {

        boolean generateInterface = interfaceName != null;

        String superClass = null;

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesGeneratorUtil.hasBeanStereotype(superclass)) {
                    superClassIsBean = true;
                    superClass = getBeanClassName(superclass);
                    break;
                } else {
                    superClass = superclass.getQualifiedName();
                }
            }
        }

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClass =
                    JavaTemplatesGeneratorUtil.getSimpleBeanSuperClassTagValue(
                            model, input);
            if (superClass != null) {

                // will act as if super class is a bean
                superClassIsBean = true;
            }
        }

        ObjectModelClass output;

        if (input.isAbstract()) {
            output = createAbstractClass(className, input.getPackageName());
        } else {
            output = createClass(className, input.getPackageName());
        }

        if (superClass != null) {
            setSuperClass(output, superClass);
        }
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean serializableFound;

        if (generateInterface) {

            addInterface(output, interfaceName);

            serializableFound = true;
        } else {

            serializableFound = addInterfaces(input, output, null);

            generateI18nBlockAndConstants(input, output);
        }

        addSerializable(input, output, serializableFound || superClassIsBean);

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        boolean usePCS = !JavaTemplatesGeneratorUtil.isNoPCS(model, input);
//        boolean usePCS = StringUtils.isEmpty(noPCSTagValue) ||
//                         !"true".equals(noPCSTagValue.trim());

        boolean generateBooleanGetMethods =
                !JavaGeneratorUtil.isDoNotGenerateBooleanGetMethods(model,
                                                                    input);
//        boolean generateBooleanGetMethods =
//                StringUtils.isEmpty(noGenerateBooleanGetMethods) ||
//                !"true".equals(noGenerateBooleanGetMethods.trim());

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output,
                           attr,
                           usePCS,
                           generateBooleanGetMethods);
        }

        if (!superClassIsBean) {
            addDefaultMethodForNoneBeanSuperClass(output, usePCS, properties);
        }
        return output;
    }

    protected void generateBeanDefaults(ObjectModelClass aClass,
                                        String defaultClassName) {

        boolean generateDefault = canGenerateBeanDefaults(aClass);

        if (!generateDefault) {
            return;
        }

        String packageName = aClass.getPackageName();
        String typeName = getBeanInterfaceName(aClass);
        String typeBeanName = getBeanClassName(aClass);

        String abstractoutclassName = "Abstract" + defaultClassName;

        if (!isInClassPath(packageName, defaultClassName)) {

            // generate defaults class

            ObjectModelClass output = createClass(defaultClassName, packageName);
            setSuperClass(output, packageName + "." + abstractoutclassName);
            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }
        }


        if (!isInClassPath(packageName, abstractoutclassName)) {

            // generate abstract defaults class

            // try to find a super class by tag-value
            String superClassName =
                    JavaTemplatesGeneratorUtil.getSimpleBeanDefaultsSuperClassTagValue(
                            model, aClass);

            ObjectModelClass output = createAbstractClass(abstractoutclassName, packageName);
            if (StringUtils.isNotBlank(superClassName)) {
                setSuperClass(output, superClassName);
            }

            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }

            ObjectModelOperation operation = addOperation(
                    output,
                    "typeOf" + typeName,
                    "<BeanType extends " + typeName + "> Class<BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeBeanName+".class;\n"
+"    "
            );
            operation = addOperation(
                    output,
                    "new" + typeName,
                    typeName,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return new "+typeBeanName+"();\n"
+"    "
            );

            addImport(output, Binder.class);
            addImport(output, BinderFactory.class);
            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType,BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        BeanType result = new"+typeName+"(source, binder);\n"
+"        return result;\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            addParameter(operation, "Binder<BeanType, BeanType>", "binder");
            setOperationBody(operation, ""
    +"\n"
+"        BeanType result = (BeanType) new"+typeName+"();\n"
+"        binder.copy(source, result);\n"
+"        return result;\n"
+"    "
            );
        }
    }

    protected Collection<ObjectModelOperation> getPublicOperations(ObjectModelClass clazz) {

        Collection<ObjectModelOperation> result = new ArrayList<ObjectModelOperation>();
        for (ObjectModelOperation operation : clazz.getOperations()) {

            ObjectModelJavaModifier visibility =
                    ObjectModelJavaModifier.fromVisibility(
                            operation.getVisibility());
            if (ObjectModelJavaModifier.PUBLIC == visibility) {
                result.add(operation);
            }
        }
        return result;
    }

    protected String getBeanInterfaceName(ObjectModelClass input) {
        String interfaceNamePrefix = JavaTemplatesGeneratorUtil.getSimpleBeanInterfaceNamePrefixTagValue(model, input);
        String interfaceNameSuffix = JavaTemplatesGeneratorUtil.getSimpleBeanInterfaceNameSuffixTagValue(model, input);

        return generateName(
                interfaceNamePrefix,
                input.getName(),
                interfaceNameSuffix
        );
    }

    protected String getBeanClassName(ObjectModelClass input) {
        String classNamePrefix = JavaTemplatesGeneratorUtil.getSimpleBeanClassNamePrefixTagValue(model, input);
        String classNameSuffix = JavaTemplatesGeneratorUtil.getSimpleBeanClassNameSuffixTagValue(model, input);

        return generateName(
                classNamePrefix,
                input.getName(),
                classNameSuffix
        );
    }

    protected String getBeanDefaultsClassName(ObjectModelClass input) {
        String classNamePrefix = JavaTemplatesGeneratorUtil.getSimpleBeanDefaultsClassNamePrefixTagValue(model, input);
        String classNameSuffix = JavaTemplatesGeneratorUtil.getSimpleBeanDefaultsClassNameSuffixTagValue(model, input);
        return generateName(
                classNamePrefix,
                input.getName(),
                classNameSuffix
        );
    }

    protected String generateName(String prefix,
                                  String name,
                                  String suffix) {
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            sb.append(prefix);
        }
        sb.append(name);
        if (StringUtils.isNotEmpty(suffix)) {
            sb.append(suffix);
        }
        return sb.toString();
    }
}
