/*
 * Copyright  2003-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package com.izforge.izpack.util.file.types.selectors;

import java.io.File;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;

import com.izforge.izpack.api.data.InstallData;
import com.izforge.izpack.util.file.types.Parameter;

/**
 * Selector that selects a certain kind of file: directory or regular.
 */
public class TypeSelector extends BaseExtendSelector
{

    private FileType type;

    /**
     * Key to used for parameterized custom selector
     */
    public static final String TYPE_KEY = "type";

    /**
     * Creates a new <code>TypeSelector</code> instance.
     */
    public TypeSelector()
    {
    }

    /**
     * @return a string describing this object
     */
    public String toString()
    {
        StringBuffer buf = new StringBuffer("{typeselector type: ");
        buf.append(type.getAttribute());
        buf.append("}");
        return buf.toString();
    }

    /**
     * Set the type of file to require.
     *
     * @param fileType the type of file - file or dir
     */
    public void setType(FileType fileType)
    {
        this.type = fileType;
    }

    /**
     * When using this as a custom selector, this method will be called.
     * It translates each parameter into the appropriate setXXX() call.
     *
     * @param parameters the complete set of parameters for this selector
     */
    public void setParameters(Parameter[] parameters)
    {
        super.setParameters(parameters);
        if (parameters != null)
        {
            for (Parameter parameter : parameters)
            {
                String paramname = parameter.getName();
                if (TYPE_KEY.equalsIgnoreCase(paramname))
                {
                    FileType type = FileType.getFromAttribute(parameter.getValue());
                    if (type != null)
                    {
                        setType(type);
                    }
                    else
                    {
                        setError("Invalid " + TYPE_KEY + " setting " + parameter.getValue());
                    }
                }
                else
                {
                    setError("Invalid parameter " + paramname);
                }
            }
        }
    }

    /**
     * Checks to make sure all settings are kosher. In this case, it
     * means that the pattern attribute has been set.
     */
    public void verifySettings()
    {
        if (type == null)
        {
            setError("The type attribute is required");
        }
    }

    /**
     * The heart of the matter. This is where the selector gets to decide
     * on the inclusion of a file in a particular fileset.
     *
     * @param basedir  the base directory the scan is being done from
     * @param filename is the name of the file to check
     * @param file     is a java.io.File object the selector can use
     * @return whether the file should be selected or not
     */
    public boolean isSelected(InstallData idata, File basedir, String filename, File file)
            throws Exception
    {
        // throw Exception on error
        validate();

        if (file.isDirectory())
        {
            return type.equals(FileType.DIR);
        }
        else
        {
            return type.equals(FileType.FILE);
        }
    }

    public enum FileType
    {
        FILE("file"), DIR("dir");

        private static Map<String, FileType> lookup;

        private String attribute;

        FileType(String attribute)
        {
            this.attribute = attribute;
        }

        static
        {
            lookup = new HashMap<String, FileType>();
            for (FileType mapperType : EnumSet.allOf(FileType.class))
            {
                lookup.put(mapperType.getAttribute(), mapperType);
            }
        }

        public String getAttribute()
        {
            return attribute;
        }

        public static FileType getFromAttribute(String attribute)
        {
            if (attribute != null && lookup.containsKey(attribute))
            {
                return lookup.get(attribute);
            }
            return null;
        }
    }
}
