/*
 * #%L
 * Pollen :: Vote Counting
 * $Id: Choice.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/business/Choice.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.business;

import java.util.ArrayList;
import java.util.List;

import org.chorem.pollen.votecounting.utils.ListBehavior;
import org.chorem.pollen.votecounting.utils.PercentageBehavior;

/**
 * Cette classe décrit le choix d'un sondage. Sa valeur sera modifiée au moment
 * de l'exécution de la méthode de dépouillement.
 *
 * @author fdesbois
 * @version $Id: Choice.java 3423 2012-06-05 16:50:22Z tchemit $
 */
public class Choice implements ListBehavior<Group>, PercentageBehavior {
    /**
     * identifiant du choix
     */
    private String idChoice;
    /**
     * nom du choix
     */
    private String name;
    /**
     * valeur du choix déterminé par le dépouillement
     */
    private double value;
    /**
     * Map des groupes liés au choix
     */
    private List<Group> groups;

    /**
     * Constructeur
     *
     * @param id : identifiant du choix
     */
    public Choice(String id) {
        this.idChoice = id;
        this.groups = new ArrayList<Group>();
        this.value = 0.;
    }

    /**
     * Constructeur
     *
     * @param id : identifiant du choix
     * @param name : nom du choix
     */
    public Choice(String id, String name) {
        this(id);
        this.name = name;
    }
    
    /**
     * setValue : Mise à jour de la valeur du choix
     *
     * @param value : nouvelle valeur
     */
    public void setValue(double value) {
        this.value = value;
    }

    /**
     * getValue : Renvoie la valeur du choix
     *
     * @return valeur courante du choix
     */
    public double getValue() {
        return this.value;
    }

    /**
     * getId : Renvoie l'identifiant du choix
     *
     * @return identifiant du choix
     */
    public String getIdChoice() {
        return this.idChoice;
    }
    
    /**
     * getName : Renvoie le nom du choix
     *
     * @return nom du choix
     */
    public String getName() {
        return this.name;
    }

    /**
     * getGroups : Renvoie la map des groupes
     *
     * @return map des groupes pour ce choix
     */
    public List<Group> getGroups() {
        return this.groups;
    }

    /**
     * addGroup : Ajout d'un groupe lié au choix
     *
     * @param weight : poids du groupe dans le sondage
     * @param idGroup : identifiant du groupe
     */
    public void addGroup(String idGroup, double weight) {
        this.groups.add(new Group(idGroup, weight));
    }

    /**
     * getGroup : Renvoie un groupe existant dans la map
     *
     * @param idGroup : identifiant du groupe dans la map
     * @return le groupe ayant l'identifiant idGroup
     */
    public Group getGroup(String idGroup) {
        for (Group g : this.groups) {
            if (g.getIdGroup().equals(idGroup)) {
                return g;
            }
        }
        return null;
    }

    @Override
    public double value() {
        return getValue();
    }

    @Override
    public List<Group> list() {
        return getGroups();
    }

    @Override
    public String toString() {
        return "Choice id=" + idChoice + " _ value=" + value;
    }
}
