/*
 * #%L
 * Pollen :: Vote Counting
 * 
 * $Id: ServiceExportImpl.java 3122 2012-01-30 20:43:30Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/services/ServiceExportImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.services;

import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.CommentDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.PollExportDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * Implémentation du service d'export.
 *
 * @author boukhary
 * @author rannou
 * @version $Id: ServiceExportImpl.java 3122 2012-01-30 20:43:30Z tchemit $
 */
@SuppressWarnings("unchecked")
public class ServiceExportImpl implements ServiceExport {

    private PollDTO poll;

    private List<VoteCountingResultDTO> voteCountingResults;

    private static Element racine;

    private static Document document;

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceExportImpl.class);

    /**
     * Constructor
     */
    public ServiceExportImpl() {

    }

    @Override
    public String executeExport(PollExportDTO pollExport) {
        poll = pollExport.getPoll();
        voteCountingResults = pollExport.getVoteCountingResults();

        String filePath = "export/" + poll.getPollId() + ".xml";
        savePoll(filePath);
        return filePath;
    }

    @Override
    public File executeFileExport(PollExportDTO pollExport) {
        String filePath = executeExport(pollExport);
        return new File(filePath);
    }

    @Override
    public PollExportDTO executeImport(String filePath) {

        // Création d'une instance de SAXBuilder
        SAXBuilder sxb = new SAXBuilder();
        try {

            // Création d'un nouveau document JDOM avec le fichier XML en argument
            document = sxb.build(new File(filePath));
        } catch (Exception e) {
            log.error("Erreur lors de l'analyse du document : " + filePath, e);
        }

        // Initialisation d'un nouvel élément racine
        racine = document.getRootElement();
        if (racine == null) {
            if (log.isErrorEnabled()) {
                log.error("racine null");
            }
        }

        return loadPoll();
    }

    /**
     * Construction du document XML à partir du sondage.
     *
     * @param le chemin du fichier à créer
     * @return le fichier créé
     */
    private void savePoll(String filePath) {

        // Création de la racine et du document
        racine = new Element("pollExport");
        document = new Document(racine);

        // Création du sondage
        Element pollElm = new Element("poll");
        pollElm.setAttribute("pollId", poll.getPollId());
        pollElm.setAttribute("pollType", poll.getPollType().name());
        pollElm.setAttribute("choiceType", poll.getChoiceType().name());
        pollElm.setAttribute("voteCounting", poll.getVoteCounting().name());
        pollElm.setAttribute("closed", poll.isClosed() ? "true" : "false");
        pollElm.setAttribute("choiceAddAllowed",
                poll.isChoiceAddAllowed() ? "true" : "false");
        pollElm.setAttribute("anonymousVoteAllowed", poll
                .isAnonymousVoteAllowed() ? "true" : "false");
        pollElm
                .setAttribute("anonymous", poll.isAnonymous() ? "true"
                        : "false");
        pollElm.setAttribute("publicResults", poll.isPublicResults() ? "true"
                : "false");
        pollElm.setAttribute("continuousResults",
                poll.isContinuousResults() ? "true" : "false");

        Element titleElm = new Element("title");
        titleElm.setText(poll.getTitle());
        pollElm.addContent(titleElm);

        Element descriptionElm = new Element("description");
        descriptionElm.setText(poll.getDescription());
        pollElm.addContent(descriptionElm);

        Element creatorIdElm = new Element("creatorId");
        creatorIdElm.setText(poll.getCreatorId());
        pollElm.addContent(creatorIdElm);

        Element creatorEmailElm = new Element("creatorEmail");
        creatorEmailElm.setText(poll.getCreatorEmail());
        pollElm.addContent(creatorEmailElm);

        Element maxChoiceNbElm = new Element("maxChoiceNb");
        maxChoiceNbElm.setText(Integer.toString(poll.getMaxChoiceNb()));
        pollElm.addContent(maxChoiceNbElm);

        // Création des commentaires
        Element commentsElm = new Element("comments");

        Iterator iteratorComment = poll.getComments().listIterator();
        while (iteratorComment.hasNext()) {
            Element commentElm = save((CommentDTO) iteratorComment.next());
            commentsElm.addContent(commentElm);
        }

        pollElm.addContent(commentsElm);

        // Création des groupes de votants
        Element groupsElm = new Element("groups");

        Iterator iteratorGroup = poll.getVotingGroups().listIterator();
        while (iteratorGroup.hasNext()) {
            Element groupElm = save((VotingGroupDTO) iteratorGroup.next());
            groupsElm.addContent(groupElm);
        }

        pollElm.addContent(groupsElm);

        // Création des choix
        Element pollChoicesElm = new Element("pollChoices");

        Iterator itPollChoices = poll.getChoices().listIterator();
        while (itPollChoices.hasNext()) {
            Element pollChoiceElm = save((PollChoiceDTO) itPollChoices.next());
            pollChoicesElm.addContent(pollChoiceElm);
        }
        pollElm.addContent(pollChoicesElm);

        // Création des résultats
        Element voteCountingResultsElm = new Element("voteCountingResults");

        Iterator iteratorvoteCountingResults = voteCountingResults
                .listIterator();
        while (iteratorvoteCountingResults.hasNext()) {
            Element voteCountingResult = save((VoteCountingResultDTO) iteratorvoteCountingResults
                    .next());
            voteCountingResultsElm.addContent(voteCountingResult);
        }

        // Ajout du sondage et des résultats à la racine
        racine.addContent(pollElm);
        racine.addContent(voteCountingResultsElm);

        if (log.isDebugEnabled()) {
            log.debug(displayDom());
        }

        // Sauvegarde du document dans un fichier
        saveDom(filePath);
    }

    private Element save(CommentDTO comment) {
        Element commentElm = new Element("comment");

        commentElm.setAttribute("votingId", comment.getVotingID());
        commentElm.setText(comment.getText());

        return commentElm;
    }

    private Element save(VotingGroupDTO group) {
        Element groupElm = new Element("group");

        groupElm.setAttribute("idGroup", group.getIdGroup());
        groupElm.setAttribute("name", group.getName());
        groupElm.setAttribute("weight", Double.toString(group.getWeight()));

        Element votingPersonsElm = new Element("votingPersons");

        Iterator iteratorVotingPersons = group.getVotingPersons()
                .listIterator();
        while (iteratorVotingPersons.hasNext()) {
            Element votingPersonElm = save((VotingPersonDTO) iteratorVotingPersons
                    .next());
            votingPersonsElm.addContent(votingPersonElm);
        }
        groupElm.addContent(votingPersonsElm);

        return groupElm;
    }

    private Element save(VotingPersonDTO votingPerson) {
        Element votingPersonElm = new Element("votingPerson");

        votingPersonElm.setAttribute("votingId", votingPerson.getVotingId());
        votingPersonElm.setAttribute("weight", Double.toString(votingPerson
                .getWeight()));

        Element emailElm = new Element("email");
        emailElm.setText(votingPerson.getEmail());
        votingPersonElm.addContent(emailElm);

        Element commentElm = new Element("comment");
        commentElm.setText(votingPerson.getComment());
        votingPersonElm.addContent(commentElm);

        Element choicesElm = new Element("choices");

        Iterator iteratorChoices = votingPerson.getChoices().listIterator();
        while (iteratorChoices.hasNext()) {
            VoteToChoiceDTO voteToChoice = (VoteToChoiceDTO) iteratorChoices
                    .next();
            Element choiceElm = new Element("choice");
            choiceElm.setAttribute("idChoice", voteToChoice.getIdChoice());
            choiceElm.setAttribute("value", Double.toString(voteToChoice
                    .getValue()));
            choicesElm.addContent(choiceElm);
        }

        votingPersonElm.addContent(choicesElm);

        return votingPersonElm;
    }

    private Element save(PollChoiceDTO pollChoice) {
        Element pollChoiceElm = new Element("pollChoice");

        pollChoiceElm.setAttribute("idChoice", pollChoice.getIdChoice());

        Element nameElm = new Element("name");
        nameElm.setText(pollChoice.getName());
        pollChoiceElm.addContent(nameElm);

        Element descriptionElm = new Element("description");
        descriptionElm.setText(pollChoice.getDescription());
        pollChoiceElm.addContent(descriptionElm);

        return pollChoiceElm;
    }

    private Element save(VoteCountingResultDTO voteCountingResults) {
        Element voteCountingResultElm = new Element("voteCountingResult");

        voteCountingResultElm.setAttribute("idPoll", voteCountingResults
                .getIdPoll());

        Element nbVotesElm = new Element("nbVotes");
        nbVotesElm.setText(Integer.toString(voteCountingResults.getNbVotes()));
        voteCountingResultElm.addContent(nbVotesElm);

        Element choiceResultElm = new Element("choiceResult");
        choiceResultElm.setText(voteCountingResults.getChoiceResult());
        voteCountingResultElm.addContent(choiceResultElm);

        Element isByGroupElm = new Element("isByGroup");
        isByGroupElm.setText(Boolean.toString(voteCountingResults.isByGroup()));
        voteCountingResultElm.addContent(isByGroupElm);

        Element choicesElm = new Element("choices");

        Iterator iteratorChoices = voteCountingResults.getChoices()
                .listIterator();
        while (iteratorChoices.hasNext()) {
            Element choiceElm = save((ChoiceDTO) iteratorChoices.next());
            choicesElm.addContent(choiceElm);
        }

        voteCountingResultElm.addContent(choicesElm);

        return voteCountingResultElm;
    }

    private Element save(ChoiceDTO choice) {
        Element choiceElm = new Element("choice");

        choiceElm.setAttribute("idChoice", choice.getIdChoice());
        choiceElm.setAttribute("value", Double.toString(choice.getValue()));

        Element percentageElm = new Element("percentage");
        percentageElm.setText(Double.toString(choice.getPercentage()));
        choiceElm.addContent(percentageElm);

        Element nbVotesElm = new Element("nbVotes");
        nbVotesElm.setText(Integer.toString(choice.getNbVotes()));
        choiceElm.addContent(nbVotesElm);

        return choiceElm;
    }

    private String displayDom() {

        // Mise en forme classique avec getPrettyFormat()
        XMLOutputter sortie = new XMLOutputter(Format.getPrettyFormat());
        return sortie.outputString(document);
    }

    private void saveDom(String filePath) {
        try {
            File file = new File(filePath);
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            }

            // Mise en forme classique avec getPrettyFormat()
            XMLOutputter sortie = new XMLOutputter(Format.getPrettyFormat());
            sortie.output(document, new FileOutputStream(file));
        } catch (java.io.IOException e) {
            log.error("Erreur lors de l'enregistrement du document : "
                    + filePath, e);
        }
    }

    /**
     * Construction du sondage à partir d'un document XML.
     *
     * @return le sondage
     */
    private PollExportDTO loadPoll() {

        // Création du sondage
        Element pollElm = racine.getChild("poll");

        PollDTO poll = new PollDTO(pollElm.getAttributeValue("pollId"));
        poll.setTitle(pollElm.getChild("title").getText());
        poll.setDescription(pollElm.getChild("description").getText());
        poll.setCreatorId(pollElm.getChild("creatorId").getText());
        poll.setCreatorEmail(pollElm.getChild("creatorEmail").getText());
        poll.setMaxChoiceNb(Integer.parseInt(pollElm.getChild("maxChoiceNb")
                .getText()));

        poll.setPollType(PollType
                .valueOf(pollElm.getAttributeValue("pollType")));
        poll.setChoiceType(ChoiceType.valueOf(pollElm
                .getAttributeValue("choiceType")));
        poll.setVoteCounting(VoteCountingType.valueOf(pollElm
                .getAttributeValue("voteCounting")));
        poll.setClosed(Boolean.valueOf(pollElm.getAttributeValue("closed")));
        poll.setChoiceAddAllowed(Boolean.valueOf(pollElm
                .getAttributeValue("choiceAddAllowed")));
        poll.setAnonymousVoteAllowed(Boolean.valueOf(pollElm
                .getAttributeValue("anonymousVoteAllowed")));
        poll.setAnonymous(Boolean.valueOf(pollElm
                .getAttributeValue("anonymous")));
        poll.setPublicResults(Boolean.valueOf(pollElm
                .getAttributeValue("publicResults")));
        poll.setContinuousResults(Boolean.valueOf(pollElm
                .getAttributeValue("continuousResults")));

        // Ajout des commentaires
        Element commentsElm = pollElm.getChild("comments");
        List<Element> listComments = commentsElm.getChildren("comment");
        List vectorComments = loadComments(listComments);

        poll.setComments(vectorComments);

        // Ajout des groupes
        Element groupsElm = pollElm.getChild("groups");
        List<Element> listGroups = groupsElm.getChildren("group");
        List vectorGroups = loadGroups(listGroups);

        poll.setVotingGroups(vectorGroups);

        // Ajout des choix
        Element pollChoicesElm = pollElm.getChild("pollChoices");
        List<Element> listPollChoices = pollChoicesElm
                .getChildren("pollChoice");
        List vectorPollChoices = loadPollChoices(listPollChoices);

        poll.setChoices(vectorPollChoices);

        // Ajout des résultats
        Element voteCountingResultsElm = racine.getChild("voteCountingResults");
        List<Element> listVoteCountingResults = voteCountingResultsElm
                .getChildren("voteCountingResult");
        List vectorVoteCountingResults = loadVoteCountingResults(listVoteCountingResults);

        // Création du sondage PollExportDTO
        PollExportDTO pollExport = new PollExportDTO();
        pollExport.setPollId(poll.getPollId());
        pollExport.setPoll(poll);
        pollExport.setVoteCountingResults(vectorVoteCountingResults);

        return pollExport;
    }

    private List loadComments(List<Element> listComments) {
        List vectorComments = new ArrayList<CommentDTO>();

        Iterator i = listComments.listIterator();
        while (i.hasNext()) {
            Element commentElm = (Element) i.next();
            CommentDTO comment = new CommentDTO(commentElm
                    .getAttributeValue("votingId"), commentElm.getText());
            vectorComments.add(comment);
        }

        return vectorComments;
    }

    private List loadGroups(List<Element> listGroups) {
        List vectorGroups = new ArrayList<VotingGroupDTO>();

        Iterator i = listGroups.listIterator();
        while (i.hasNext()) {
            Element groupElm = (Element) i.next();
            VotingGroupDTO group = new VotingGroupDTO(groupElm
                    .getAttributeValue("idGroup"), Double.parseDouble((groupElm
                    .getAttributeValue("weight"))));

            group.setName(groupElm.getAttributeValue("name"));

            Element votingPersonsElm = groupElm.getChild("votingPersons");
            List<Element> listVotingPersons = votingPersonsElm
                    .getChildren("votingPerson");
            List vectorVotingPersons = loadVotingPersons(listVotingPersons);

            group.setVotingPersons(vectorVotingPersons);

            vectorGroups.add(group);
        }

        return vectorGroups;
    }

    private List loadVotingPersons(List<Element> listVotingPersons) {
        List vectorVotingPersons = new ArrayList<VotingPersonDTO>();

        Iterator i = listVotingPersons.listIterator();
        while (i.hasNext()) {
            Element votingPersonElm = (Element) i.next();
            VotingPersonDTO votingPerson = new VotingPersonDTO(votingPersonElm
                    .getAttributeValue("votingId"), Double
                    .parseDouble((votingPersonElm.getAttributeValue("weight"))));

            votingPerson.setEmail(votingPersonElm.getChild("email").getValue());
            votingPerson.setComment(votingPersonElm.getChild("comment")
                    .getValue());

            Element choicesElm = votingPersonElm.getChild("choices");

            List<Element> listVoteToChoices = choicesElm.getChildren("choice");
            List vectorVoteToChoices = loadVoteToChoices(listVoteToChoices);

            votingPerson.setChoices(vectorVoteToChoices);
            vectorVotingPersons.add(votingPerson);
        }

        return vectorVotingPersons;
    }

    private List loadVoteToChoices(List<Element> listVoteToChoices) {
        List vectorVoteToChoices = new ArrayList<VotingPersonDTO>();

        Iterator i = listVoteToChoices.listIterator();
        while (i.hasNext()) {
            Element voteToChoiceElm = (Element) i.next();

            VoteToChoiceDTO voteToChoice = new VoteToChoiceDTO(voteToChoiceElm
                    .getAttributeValue("idChoice"), Double
                    .parseDouble(voteToChoiceElm.getAttributeValue("value")));
            vectorVoteToChoices.add(voteToChoice);
        }

        return vectorVoteToChoices;
    }

    private List loadPollChoices(List<Element> listPollChoices) {
        List vectorPollChoices = new ArrayList<PollChoiceDTO>();

        Iterator i = listPollChoices.listIterator();
        while (i.hasNext()) {
            Element pollChoiceElm = (Element) i.next();
            PollChoiceDTO pollChoice = new PollChoiceDTO(pollChoiceElm
                    .getAttributeValue("idChoice"));

            String pollChoiceName = pollChoiceElm.getChild("name").getValue();
            pollChoice.setName(pollChoiceName);
            pollChoice.setHidden(pollChoiceName != null
                    && pollChoiceName.startsWith(NumberMethod.HIDDEN_PREFIX));

            pollChoice.setDescription(pollChoiceElm.getChild("description")
                    .getValue());

            vectorPollChoices.add(pollChoice);
        }

        return vectorPollChoices;
    }

    private List loadVoteCountingResults(List<Element> listVoteCountingResults) {
        List vectorVoteCountingResults = new ArrayList<VoteCountingResultDTO>();

        Iterator i = listVoteCountingResults.listIterator();
        while (i.hasNext()) {
            Element voteCountingResultElm = (Element) i.next();
            VoteCountingResultDTO voteCountingResult = new VoteCountingResultDTO();
            voteCountingResult.setIdPoll(voteCountingResultElm
                    .getAttributeValue("idPoll"));
            voteCountingResult.setNbVotes(Integer
                    .parseInt(voteCountingResultElm.getChild("nbVotes")
                            .getValue()));
            voteCountingResult.setChoiceResult(voteCountingResultElm.getChild(
                    "choiceResult").getValue());
            voteCountingResult.setByGroup(Boolean
                    .parseBoolean((voteCountingResultElm.getChild("isByGroup")
                            .getValue())));

            Element choicesElm = voteCountingResultElm.getChild("choices");
            List<Element> listChoices = choicesElm.getChildren("choice");
            List vectorChoices = loadChoices(listChoices);
            voteCountingResult.setChoices(vectorChoices);
            vectorVoteCountingResults.add(voteCountingResult);
        }

        return vectorVoteCountingResults;
    }

    private List loadChoices(List<Element> listChoices) {
        List vectorChoices = new ArrayList<ChoiceDTO>();

        Iterator i = listChoices.listIterator();
        while (i.hasNext()) {
            Element choiceElm = (Element) i.next();
            ChoiceDTO choice = new ChoiceDTO(choiceElm.getAttribute("idChoice")
                    .getName(), Double.parseDouble(choiceElm.getAttribute(
                    "value").getValue()), Double.parseDouble(choiceElm
                    .getChild("percentage").getValue()), Integer
                    .parseInt((choiceElm.getChild("nbVotes").getValue())));
            vectorChoices.add(choice);
        }

        return vectorChoices;
    }

}