/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.utils;

import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.sun.syndication.feed.synd.SyndContent;
import com.sun.syndication.feed.synd.SyndContentImpl;
import com.sun.syndication.feed.synd.SyndEntry;
import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.feed.synd.SyndFeedImpl;
import com.sun.syndication.io.SyndFeedInput;
import com.sun.syndication.io.SyndFeedOutput;
import com.sun.syndication.io.XmlReader;

/**
 * Classe permettant de gérer les flux de syndication (RSS, Atom).
 *
 * @author rannou
 * @version $Id: FeedUtil.java 2731 2009-08-18 12:22:46Z nrannou $
 */
public class FeedUtil {

    /** log. */
    private static final Log log = LogFactory.getLog(FeedUtil.class);

    /**
     * Création d'un flux.
     *
     * @param filename fichier correspondant au flux
     * @param type type de flux (rss_2.0, atom_1.0)
     * @param title titre du canal
     * @param link URL associée au canal
     * @param description description du canal
     */
    public static void createFeed(String filename, String type, String title,
            String link, String description) {
        createFeed(new File(filename), type, title, link, description);
    }

    /**
     * Création d'un flux.
     *
     * @param file fichier correspondant au flux
     * @param type type de flux (rss_2.0, atom_1.0)
     * @param title titre du canal
     * @param link URL associée au canal
     * @param description description du canal
     */
    public static void createFeed(File file, String type, String title,
            String link, String description) {
        try {
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            }

            SyndFeed feed = new SyndFeedImpl();
            feed.setFeedType(type);
            feed.setTitle(title);
            feed.setLink(link);
            feed.setDescription(description);

            Writer writer = new FileWriter(file);
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, writer);
            writer.close();

            if (log.isDebugEnabled()) {
                log.debug("Feed created: " + file.getAbsolutePath());
            }
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Feed creation error", e);
            }
            e.printStackTrace();
        }
    }

    /**
     * Suppression d'un flux.
     *
     * @param filename fichier correspondant au flux
     */
    public static void deleteFeed(String filename) {
        deleteFeed(new File(filename));
    }

    /**
     * Suppression d'un flux.
     *
     * @param file fichier correspondant au flux
     */
    public static void deleteFeed(File file) {
        file.delete();

        if (log.isDebugEnabled()) {
            log.debug("Feed deleted: " + file.getAbsolutePath());
        }
    }

    /**
     * Ajout d'une entrée dans un flux.
     *
     * @param filename fichier correspondant au flux
     * @param title titre de l'entrée
     * @param link URL associée à l'entrée
     * @param content contenu de l'entrée
     */
    public static void feedFeed(String filename, String title, String link,
            String content) {
        feedFeed(new File(filename), title, link, content);
    }

    /**
     * Ajout d'une entrée dans un flux.
     *
     * @param file fichier correspondant au flux
     * @param title titre de l'entrée
     * @param link URL associée à l'entrée
     * @param content contenu de l'entrée
     */
    public static void feedFeed(File file, String title, String link,
            String content) {
        try {
            List entries = new ArrayList();
            SyndFeedInput input = new SyndFeedInput();
            SyndFeed feed = input.build(new XmlReader(file));
            entries.addAll(feed.getEntries());

            SyndEntry entry;
            SyndContent description;
            entry = new SyndEntryImpl();
            entry.setTitle(title);
            entry.setLink(link);
            entry.setPublishedDate(new Date());
            description = new SyndContentImpl();
            description.setType("text/plain");
            description.setValue(content);
            entry.setDescription(description);
            entries.add(entry);

            feed.setEntries(entries);

            Writer writer = new FileWriter(file);
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, writer);
            writer.close();

            if (log.isDebugEnabled()) {
                log.debug("Feed updated: " + file.getAbsolutePath());
            }
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Feed feeding error", e);
            }
            e.printStackTrace();
        }
    }
}
