/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import javax.activation.MimetypesFileTypeMap;

import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.Response;
import org.slf4j.Logger;

/**
 * Composant qui affiche une image.
 *
 * Ce composant permet d'afficher une image dynamique (exemple : image uploadée
 * sur le serveur par un formulaire).
 *
 * <pre>
 * Utilisation :
 * &lt;t:image src=&quot;/img/image.png&quot; /&gt;
 * Pour afficher une image statique il est préférable d'utiliser :
 * &lt;img src=&quot;${asset:context:img}/image.png&quot; /&gt;
 * </pre>
 *
 * @author rannou
 * @version $Id: Image.java 2856 2010-02-01 17:46:41Z echatellier $
 */
public class Image extends ImageContextLink {

    /** Chemin de l'image */
    @Parameter(required = true)
    private String src;

    /** Texte alternatif */
    @Parameter
    private String alt;

    /** Largeur de l'image */
    @Parameter
    private int width;

    /** Hauteur de l'image */
    @Parameter
    private int height;

    @Inject
    private Logger logger;

    @Inject
    private ComponentResources resources;

    /**
     * @return the src
     */
    public String getSrc() {
        return src;
    }

    /**
     * @return the alt
     */
    public String getAlt() {
        return alt;
    }

    /**
     * @return the width
     */
    public int getWidth() {
        return width;
    }

    /**
     * @return the height
     */
    public int getHeight() {
        return height;
    }
    
    /**
     * Create a link to call "thumbLink" event.
     *
     * @return event link
     */
    public Link getThumbLink() {
        return resources.createEventLink("imageLink", src, width,
                height, true);
    }

    /**
     * Create a link to call "imageLink" event.
     *
     * @return event link
     */
    public Link getImageLink() {
        return resources.createEventLink("imageLink", src, width,
                height, false);
    }

    /** Affichage de l'image */
    public StreamResponse onImageLink(final String src, final int width,
            final int height, boolean thumb) {
        
        // security : src containing .. are filtered, to not access to full system
        if (StringUtils.isEmpty(src) || src.contains("..")) {
            return null;
        }

        File file = getFile(src, thumb);
        FileInputStream stream = null;
        try {
            stream = new FileInputStream(file);
        } catch (FileNotFoundException e) {
            logger.error("Fichier inexistant : " + file.getAbsolutePath(), e);
        }

        MimetypesFileTypeMap mimes = new MimetypesFileTypeMap();
        String type = mimes.getContentType(file);

        return new ImageStreamResponse(stream, type);
    }

    /** Création du flux de l'image */
    class ImageStreamResponse implements StreamResponse {
        protected String contentType;
        protected InputStream stream;

        public ImageStreamResponse(InputStream stream, String contentType) {
            this.stream = stream;
            this.contentType = contentType;
        }

        public String getContentType() {
            return contentType;
        }

        public InputStream getStream() throws IOException {
            return stream;
        }

        public void prepareResponse(Response response) {

        }
    }
}