/*
 * #%L
 * Pollen :: Services
 * $Id: PollCommentService.java 3719 2012-10-01 11:32:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollCommentService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.CommentDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollCommentNotFound;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;

import java.util.List;

/**
 * Manage comments on a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollCommentService extends PollenServiceSupport {

    public Comment getExistingPollComment(String commentId) throws PollCommentNotFound {
        Comment result = getEntityById(Comment.class, commentId);
        if (result == null) {
            throw new PollCommentNotFound();
        }
        return result;
    }

    public Comment createComment(Poll poll, Comment comment) {

        Preconditions.checkNotNull(poll);

        PollDAO pollDAO = getDAO(Poll.class);
        CommentDAO dao = getDAO(Comment.class);

        try {

            // get last version of the poll
            Poll pollToUpdate = pollDAO.findByTopiaId(poll.getTopiaId());

            // creates the poll comment
            Comment commentCreated = dao.create(
                    Comment.PROPERTY_POST_DATE, comment.getPostDate(),
                    Comment.PROPERTY_AUTHOR, comment.getAuthor(),
                    Comment.PROPERTY_TEXT, comment.getText()
            );

            // add it to poll
            pollToUpdate.addComment(commentCreated);

            // attach poll account if exists
            PollAccount pollAccount = comment.getPollAccount();
            if (pollAccount.getTopiaId() != null) {

                PollAccount pollAccountLoaded = getEntityById(PollAccount.class, pollAccount.getTopiaId());

                // attach the userAccount
                if (pollAccountLoaded.getUserAccount() == null && pollAccount.getUserAccount() != null) {
                    String userId = pollAccount.getUserAccount().getTopiaId();
                    UserAccount userAccountLoaded = getEntityById(UserAccount.class, userId);
                    pollAccountLoaded.setUserAccount(userAccountLoaded);
                }

                commentCreated.setPollAccount(pollAccountLoaded);
            }

            // feed notification

            PollFeedService pollFeedService = newService(PollFeedService.class);
            pollFeedService.onCommentAdded(poll, commentCreated);

            commitTransaction("Could not create comment (" + comment.getAuthor() + ") " + comment.getText());

            return commentCreated;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void deleteComment(Poll poll,
                              String commentId,
                              String reason) throws PollCommentNotFound {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(commentId);

        Comment comment = getExistingPollComment(commentId);

        CommentDAO dao = getDAO(Comment.class);

        try {

            dao.delete(comment);

            // feed notification

            PollFeedService pollFeedService = newService(PollFeedService.class);
            pollFeedService.onCommentDeleted(poll, comment, reason);

            commitTransaction("Could not delete comment " + comment.getText());

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public Comment getNewComment(PollAccount account, String author, String text) {

        CommentDAO dao = getDAO(Comment.class);
        Comment result = newInstance(dao);

        if (account != null) {
            result.setPollAccount(account);
        }
        result.setAuthor(author);
        result.setText(text);
        result.setPostDate(serviceContext.getCurrentTime());
        return result;
    }

    public List<Comment> getComments(TopiaPagerBean pager, String pollId) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(pollId);

        try {
            CommentDAO dao = getDAO(Comment.class);
            List<Comment> result = dao.findComments(pager, pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain comments", e);
        }
    }

    public List<Comment> getAllComments(String pollId) {

        Preconditions.checkNotNull(pollId);

        try {
            CommentDAO dao = getDAO(Comment.class);
            List<Comment> result = dao.findAllComments(pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain comments", e);
        }
    }

}
