/*
 * #%L
 * Pollen :: Services
 * $Id: FavoriteService.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-services/src/main/java/org/chorem/pollen/services/impl/FavoriteService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.PersonList;
import org.chorem.pollen.business.persistence.PersonListDAO;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.FavoriteListAlreadyExistException;
import org.chorem.pollen.services.exceptions.FavoriteListNotFoundException;
import org.chorem.pollen.services.exceptions.FavoriteListNotOwnedByUserException;
import org.chorem.pollen.services.exceptions.ParticipantAlreadyFoundInListException;
import org.chorem.pollen.services.exceptions.ParticipantNotFoundException;
import org.chorem.pollen.services.exceptions.ParticipantNotFoundInListException;
import org.chorem.pollen.services.exceptions.UserNotFoundException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;

import java.util.List;

public class FavoriteService extends PollenServiceSupport {

    public List<PersonList> getFavoriteLists(UserAccount user,
                                             TopiaFilterPagerUtil.FilterPagerBean pager) {

        Preconditions.checkNotNull(user);

        try {
            PersonListDAO dao = getDAO(PersonList.class);
            List<PersonList> result = dao.getFavoriteLists(user, pager);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<PersonList> getFavoriteLists(UserAccount user) {

        Preconditions.checkNotNull(user);

        try {
            PersonListDAO dao = getDAO(PersonList.class);
            List<PersonList> result = dao.getFavoriteLists(user);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public PersonList getFavoriteList(UserAccount user,
                                      String favoriteListId) throws FavoriteListNotFoundException, FavoriteListNotOwnedByUserException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(favoriteListId);

        try {
            PersonListDAO dao = getDAO(PersonList.class);

            PersonList favoriteList = dao.findByTopiaId(favoriteListId);

            if (favoriteList == null) {
                throw new FavoriteListNotFoundException();
            }

            // check this favorite belongs to the given user
            UserAccount favoriteListOwner = favoriteList.getOwner();
            if (!favoriteListOwner.equals(user)) {
                throw new FavoriteListNotOwnedByUserException();
            }

            return favoriteList;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<PollAccount> getFavoriteListUsers(PersonList favoriteList,
                                                  TopiaFilterPagerUtil.FilterPagerBean pager) {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pager);

        try {

            PollAccountDAO dao = getDAO(PollAccount.class);
            List<PollAccount> result = dao.getFavoriteListUsers(favoriteList, pager);
            return result;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public PersonList createFavoriteList(UserAccount user,
                                         String name)
            throws FavoriteListAlreadyExistException, UserNotFoundException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(name);

        // can't accept favorite list without name
        Preconditions.checkArgument(StringUtils.isNotEmpty(name));

        UserAccountDAO userDAO = getDAO(UserAccount.class);
        PersonListDAO dao = getDAO(PersonList.class);

        PersonList result;
        try {

            UserAccount userToUse =
                    userDAO.findByTopiaId(user.getTopiaId());

            if (userToUse == null) {
                throw new UserNotFoundException();
            }

            // check list does not already exists

            boolean exist = dao.isPersonListExist(user, name);

            if (exist) {

                throw new FavoriteListAlreadyExistException();
            }

            result = dao.create(
                    PersonList.PROPERTY_OWNER, userToUse
            );
            result.setName(name);
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
        return result;
    }

    public PersonList deleteFavoriteList(UserAccount user,
                                         PersonList favoriteList) throws FavoriteListNotFoundException, FavoriteListNotOwnedByUserException, UserNotFoundException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(favoriteList);

        UserAccountDAO userDAO = getDAO(UserAccount.class);
        PersonListDAO dao = getDAO(PersonList.class);

        try {

            UserAccount userToUse =
                    userDAO.findByTopiaId(user.getTopiaId());

            if (userToUse == null) {
                throw new UserNotFoundException();
            }

            PersonList favoriteListToDelete =
                    dao.findByTopiaId(favoriteList.getTopiaId());

            if (favoriteListToDelete == null) {
                throw new FavoriteListNotFoundException();
            }

            // check this favorite belongs to the given user
            UserAccount favoriteListOwner = favoriteListToDelete.getOwner();
            if (!favoriteListOwner.equals(user)) {
                throw new FavoriteListNotOwnedByUserException();
            }

            dao.delete(favoriteListToDelete);

            commitTransaction("Could not delete favorite list");
            return favoriteListToDelete;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void addPollAccountToFavoriteList(PersonList favoriteList,
                                             PollAccount pollAccount) throws ParticipantAlreadyFoundInListException, FavoriteListNotFoundException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        // can't accept poll account without name
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getVotingId()));

        // can't accept poll account without email
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getEmail()));

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            // check there is other poll account in this list with same email
            boolean pollAccountExists = dao.isPollAccountExists(personListToUpdate, pollAccount);
            if (pollAccountExists) {
                throw new ParticipantAlreadyFoundInListException();
            }

            PollAccount pollAccountCreated =
                    dao.create(PollAccount.PROPERTY_PERSON_LIST,
                               personListToUpdate);

            pollAccountCreated.setVotingId(pollAccount.getVotingId());
            pollAccountCreated.setEmail(pollAccount.getEmail());
            personListToUpdate.addPollAccount(pollAccountCreated);

            commitTransaction("Could not add poll account to favorite list");
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void editPollAccountToFavoriteList(PersonList favoriteList,
                                              PollAccount pollAccount) throws ParticipantAlreadyFoundInListException, FavoriteListNotFoundException, ParticipantNotFoundException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        // can't accept poll account without name
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getVotingId()));

        // can't accept poll account without email
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getEmail()));

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            PollAccount pollAccountToUpdate =
                    dao.findByTopiaId(pollAccount.getTopiaId());

            if (pollAccountToUpdate == null) {
                throw new ParticipantNotFoundException();
            }

            // check there is another poll account in this list with same id
            boolean pollAccountExists = dao.isPollAccountExists(personListToUpdate, pollAccount);
            if (pollAccountExists) {
                throw new ParticipantAlreadyFoundInListException();
            }

            pollAccountToUpdate.setVotingId(pollAccount.getVotingId());
            pollAccountToUpdate.setEmail(pollAccount.getEmail());
            commitTransaction("Could not edit poll account favorite list");

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void removePollAccountToFavoriteList(PersonList favoriteList,
                                                PollAccount pollAccount) throws ParticipantNotFoundException, FavoriteListNotFoundException, ParticipantNotFoundInListException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            PollAccount pollAccountToRemove =
                    dao.findByTopiaId(pollAccount.getTopiaId());

            if (pollAccountToRemove == null) {

                throw new ParticipantNotFoundException();
            }

            PollAccount pollAccountInList =
                    personListToUpdate.getPollAccountByTopiaId(pollAccount.getTopiaId());

            if (pollAccountInList == null) {

                throw new ParticipantNotFoundInListException();
            }

            personListToUpdate.removePollAccount(pollAccountToRemove);

            commitTransaction("Could not remove pollaccount to favorite list");
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public PersonList newFavoriteList() {
        PersonListDAO dao = getDAO(PersonList.class);
        PersonList personList = newInstance(dao);
        return personList;
    }

    public PollAccount newPollAccountForFavoriteList() {
        PollAccountDAO dao = getDAO(PollAccount.class);
        PollAccount pollAccount = newInstance(dao);
        return pollAccount;
    }
}
