/*
 * #%L
 * Pollen :: Services
 * $Id: EmailService.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-services/src/main/java/org/chorem/pollen/services/impl/EmailService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.SimpleEmail;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.PollenServiceSupport;
import org.nuiton.util.StringUtil;

import java.util.List;
import java.util.Locale;

import static org.nuiton.i18n.I18n.l_;

public class EmailService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EmailService.class);

    public void onUserCreated(UserAccount user) {

        if (serviceContext.isCanSendEmail() &&
            StringUtil.isEmail(user.getEmail())) {

            Locale locale = getLocale();
            String subject = l_(locale, "pollen.email.userRegister.subject",
                                user.getLogin());
            String content = l_(locale, "pollen.email.userRegister.content",
                                user.getDisplayName(),
                                user.getLogin(), user.getPassword(),
                                serviceContext.getApplicationURL());

            PollenEmail pollenEmail = createPollenEmail(user.getEmail(),
                                                        subject,
                                                        content);

            sendEmail(pollenEmail);
        }
    }

    public void onLostPassword(UserAccount user, String newPassword) {

        if (serviceContext.isCanSendEmail()) {

            Locale locale = getLocale();

            String subject = l_(locale, "pollen.email.lostPassword.subject",
                                user.getLogin());
            String content = l_(locale, "pollen.email.lostPassword.content",
                                user.getDisplayName(),
                                user.getLogin(), newPassword,
                                serviceContext.getApplicationURL());

            PollenEmail pollenEmail = createPollenEmail(user.getEmail(),
                                                        subject,
                                                        content);

            sendEmail(pollenEmail);
        }
    }

    public void onVoteAdded(Poll poll) {

        if (serviceContext.isCanSendEmail()) {

            Locale locale = getLocale();

            String pollTitle = poll.getTitle();

            PollUrlService urlService = newService(PollUrlService.class);

            PollUrl summaryUrl = urlService.getPollSummaryUrl(poll);

            // Mail au créateur

            if (StringUtils.isNotEmpty(poll.getCreator().getEmail())) {

                int nbVotes = poll.sizeVote();

                String subject = l_(locale, "pollen.email.voteEmail.subject",
                                    pollTitle);
                String content = l_(locale, "pollen.email.voteEmail.content",
                                    pollTitle, nbVotes, summaryUrl);

                PollenEmail pollenEmail = createPollenEmail(
                        poll.getCreator().getEmail(), subject, content);

                sendEmail(pollenEmail);
            }
        }
    }

    public void onPollCreated(Poll poll) {

        if (serviceContext.isCanSendEmail()) {

            Locale locale = getLocale();

            String pollTitle = poll.getTitle();

            PollUrlService urlService = newService(PollUrlService.class);
            PollUrl summaryUrl = urlService.getPollSummaryUrl(poll);
            PollUrl voteUrl = urlService.getPollVoteUrl(poll);

            // Mail au créateur
            if (StringUtils.isNotEmpty(poll.getCreator().getEmail())) {

                String subject, content;
                if (PollType.FREE == poll.getPollType()) {
                    subject = l_(locale, "pollen.email.createPoll.subject",
                                 pollTitle);
                    content = l_(locale, "pollen.email.createPoll.content",
                                 pollTitle, poll.getPollId(), summaryUrl);

                } else {
                    subject = l_(locale, "pollen.email.createRestrictedPoll.subject",
                                 pollTitle);
                    content = l_(locale, "pollen.email.createRestrictedPoll.content",
                                 pollTitle, summaryUrl);
                }

                PollenEmail pollenEmail = createPollenEmail(
                        poll.getCreator().getEmail(), subject, content);

                sendEmail(pollenEmail);
            }

            // Mails à tous les votants
            List<PollAccount> votingPollAccounts = poll.getPollAccounts(false);

            if (CollectionUtils.isNotEmpty(votingPollAccounts)) {

                List<PollenEmail> emails = Lists.newArrayList();
                for (PollAccount account : votingPollAccounts) {

                    PollenEmail pollenEmail =
                            createVotingEmail(locale,
                                              pollTitle,
                                              account,
                                              voteUrl);

                    emails.add(pollenEmail);
                }

                sendEmail(emails);
            }
        }
    }

    public void onRestrictedPersonAdded(Poll poll, PollAccount account) {

        if (serviceContext.isCanSendEmail()) {

            Locale locale = getLocale();

            String pollTitle = poll.getTitle();

            PollUrlService urlService = newService(PollUrlService.class);
            PollUrl voteURL = urlService.getPollVoteUrl(poll);

            PollenEmail pollenEmail =
                    createVotingEmail(locale, pollTitle, account, voteURL);

            sendEmail(pollenEmail);
        }
    }

    public void onPollReminder(Poll poll) {

        if (serviceContext.isCanSendEmail()) {

            Locale locale = getLocale();

            String pollTitle = poll.getTitle();

            PollUrl voteURL =
                    newService(PollUrlService.class).getPollVoteUrl(poll);

            // Mails aux votants qui n'ont toujours pas voté
            List<PollAccount> votingPollAccounts = poll.getPollAccounts(true);

            if (CollectionUtils.isNotEmpty(votingPollAccounts)) {
                String subject = l_(locale, "pollen.email.reminderEmail.subject",
                                    pollTitle);

                List<PollenEmail> emails = Lists.newArrayList();
                for (PollAccount account : votingPollAccounts) {
                    PollUrl accountVoteURL =
                            PollUrl.newPollUrl(voteURL, account.getAccountId());

                    String content =
                            l_(locale, "pollen.email.reminderEmail.content",
                               pollTitle, account.getVotingId(), accountVoteURL);

                    PollenEmail pollenEmail = createPollenEmail(
                            account.getEmail(), subject, content);

                    emails.add(pollenEmail);
                }
                sendEmail(emails);
            }
        }
    }

    private PollenEmail createVotingEmail(Locale locale,
                                          String pollTitle,
                                          PollAccount account,
                                          PollUrl voteURL) {

        String subject = l_(locale, "pollen.email.votingEmail.subject",
                            pollTitle);

        PollUrl accountVoteURL =
                PollUrl.newPollUrl(voteURL, account.getAccountId());

        String content =
                l_(locale, "pollen.email.votingEmail.content",
                   pollTitle, account.getVotingId(), accountVoteURL);

        PollenEmail result = createPollenEmail(
                account.getEmail(), subject, content);

        return result;
    }

    protected void sendEmail(Iterable<PollenEmail> pollEmails) {

        //TODO tchemit If two much mails then use SendEmail thread...
        for (PollenEmail pollenEmail : pollEmails) {
            sendEmail(pollenEmail);
        }

//                // send mail preparation
//                try {
//                    sendMailService.prepareMails(poll.getId(), mailList);
//                    sendMailService.wakeUp();
//                } catch (IOException ex) {
//                    if (log.isErrorEnabled()) {
//                        log.error("Can't prepare send mail on disk, mail won't be send !!!", ex);
//                    }
//                }

    }

    protected void sendEmail(PollenEmail pollenEmail) {

        Preconditions.checkNotNull(pollenEmail);

        String to = pollenEmail.getTo();
        Preconditions.checkNotNull(to);

        String subject = pollenEmail.getSubject();
        Preconditions.checkNotNull(subject);

        String content = pollenEmail.getContent();
        Preconditions.checkNotNull(content);

        try {
            // Create the SimpleEmail to send
            SimpleEmail email = new SimpleEmail();

            PollenConfiguration configuration = getConfiguration();

            email.setHostName(configuration.getEmailHost());
            email.setSmtpPort(configuration.getEmailPort());
            email.setFrom(configuration.getEmailFrom());
            email.setCharset(configuration.getCharset());

            email.addTo(to);
            email.setSubject(subject);
            email.setMsg(content);
            email.send();

            if (log.isInfoEnabled()) {
                log.info("Mail sent to : " + to);
            }
            if (log.isDebugEnabled()) {
                log.debug("Email infos : " +
                          "\ndate: " + email.getSentDate() +
                          "\nhostname: " + email.getHostName() +
                          "\nport: " + email.getSmtpPort() +
                          "\nfrom: " + email.getFromAddress().toString());
            }
        } catch (EmailException eee) {
            throw new PollenTechnicalException(
                    "could not send email to " + to, eee);
        }
    }

    public static PollenEmail createPollenEmail(String to,
                                                String subject,
                                                String content) {
        return new PollenEmail(to, subject, content);
    }

    public static class PollenEmail {

        protected String subject;

        protected String content;

        protected String to;


        private PollenEmail(String to, String subject, String content) {
            this.subject = subject;
            this.content = content;
            this.to = to;
        }

        public String getSubject() {
            return subject;
        }

        public String getContent() {
            return content;
        }

        public String getTo() {
            return to;
        }
    }
}
