/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PollCommentService.java 3332 2012-04-30 15:52:29Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollCommentService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.CommentDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollCommentNotFound;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;

import java.util.List;

/**
 * Manage comments on a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollCommentService extends PollenServiceSupport {

    public Comment createComment(Poll poll, Comment comment) {

        Preconditions.checkNotNull(poll);

        PollDAO pollDAO = getDAO(Poll.class);
        CommentDAO dao = getDAO(Comment.class);

        try {

            // get last version of the poll
            Poll pollToUpdate = pollDAO.findByTopiaId(poll.getTopiaId());

            // creates the poll comment
            Comment commentCreated = dao.create(
                    Comment.PROPERTY_POLL, pollToUpdate,
                    Comment.PROPERTY_POST_DATE, comment.getPostDate(),
                    Comment.PROPERTY_AUTHOR, comment.getAuthor(),
                    Comment.PROPERTY_TEXT, comment.getText()
            );

            // add it to poll
            pollToUpdate.addComment(commentCreated);

            // attach poll account if exists
            PollAccount pollAccount = comment.getPollAccount();
            if (pollAccount.getTopiaId() != null) {

                PollAccount pollAccountLoaded = getEntityById(PollAccount.class, pollAccount.getTopiaId());

                // attach the userAccount
                if (pollAccountLoaded.getUserAccount() == null && pollAccount.getUserAccount() != null) {
                    String userId = pollAccount.getUserAccount().getTopiaId();
                    UserAccount userAccountLoaded = getEntityById(UserAccount.class, userId);
                    pollAccountLoaded.setUserAccount(userAccountLoaded);
                }

                commentCreated.setPollAccount(pollAccountLoaded);
            }

            commitTransaction("Could not create comment (" + comment.getAuthor() + ") " + comment.getText());

            // feed notification

            PollService pollService = newService(PollService.class);

            PollUrl pollVoteUrl = pollService.getPollVoteUrl(poll, false);
            PollFeedService pollFeedService = newService(PollFeedService.class);
            pollFeedService.onAddComment(commentCreated, pollVoteUrl);

            return commentCreated;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void deleteComment(String commentId) throws PollCommentNotFound {

        Preconditions.checkNotNull(commentId);

        CommentDAO dao = getDAO(Comment.class);

        try {

            Comment comment = dao.findByTopiaId(commentId);

            if (comment == null) {
                throw new PollCommentNotFound();
            }

            dao.delete(comment);

            commitTransaction("Could not delete comment " + comment.getText());

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public Comment getNewComment(PollAccount account, String author, String text) {

        CommentDAO dao = getDAO(Comment.class);
        Comment result = newInstance(dao);

        if (account != null) {
            result.setPollAccount(account);
        }
        result.setAuthor(author);
        result.setText(text);
        result.setPostDate(serviceContext.getCurrentTime());
        return result;
    }

    public boolean isCanDeleteComment(Comment comment,
                                      PollAccount pollAccount,
                                      UserAccount userAccount) {

        boolean result;

        PollAccount commentAccount = comment.getPollAccount();

        // loggued
        if (userAccount != null) {

            boolean isAdmin = userAccount.isAdministrator();
            boolean isCommentAccount = userAccount.equals(commentAccount.getUserAccount());

            result = isAdmin || isCommentAccount;

        } else if (pollAccount != null) {

            result = pollAccount.equals(commentAccount);

        } else {
            result = false;
        }
        return result;
    }

    public List<Comment> getComments(TopiaFilterPagerUtil.FilterPagerBean pager,
                                     String pollId) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(pollId);

        try {
            CommentDAO dao = getDAO(Comment.class);
            List<Comment> result = dao.getComments(pager, pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain comments", e);
        }
    }

    public List<Comment> getAllComments(String pollId) {

        Preconditions.checkNotNull(pollId);

        try {
            CommentDAO dao = getDAO(Comment.class);
            List<Comment> result = dao.getAllComments(pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain comments", e);
        }
    }

}
