/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPollAccountConverter;
import org.chorem.pollen.business.converters.DataVoteConverter;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.VoteDTO;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;

/**
 * Implémentation du service de gestion des votes.
 *
 * @author rannou
 * @version $Id: ServiceVoteImpl.java 2862 2010-02-09 15:31:45Z jruchaud $
 */
public class ServiceVoteImpl implements ServiceVote {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private VoteDAO voteDAO = null;
    private DataVoteConverter converter = new DataVoteConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceVoteImpl.class);

    public ServiceVoteImpl() {
    }

    @Override
    public String createVote(VoteDTO voteDTO, PollAccountDTO pollAccountDTO) {
        TopiaContext transaction = null;
        String topiaId = null;
        try {
            transaction = rootContext.beginTransaction();

            // création du compte associé au vote s'il n'existe pas
            // sinon mise à jour du compte
            //ServicePollAccount spa = new ServicePollAccountImpl();
            if (pollAccountDTO != null) {
                pollAccountDTO.setHasVoted(true);
                PollAccountDAO pollAccountDAO = PollenModelDAOHelper.getPollAccountDAO(transaction);

                PollAccount pollAccountEntity =
                        pollAccountDAO.findByTopiaId(pollAccountDTO.getId());

//                PollAccountDTO account = spa.findPollAccountById(pollAccountDTO
//                        .getId());
                if (pollAccountEntity == null) {
                    log.debug("Nouveau compte associé au vote");
                    pollAccountEntity = pollAccountDAO.create();
//                     voteDTO.setPollAccountId(spa
//                            .createPollAccount(pollAccountDTO));
                } else {
                    log.debug("Compte associé au vote : " + pollAccountEntity.getTopiaId());
                    //spa.updatePollAccount(pollAccountDTO);
                    //voteDTO.setPollAccountId(account.getId());

                }
                // Fill pollAccountEntity from DTO in argument
                DataPollAccountConverter pollAccountConverter = new DataPollAccountConverter();
                pollAccountConverter.setTransaction(transaction);
                pollAccountConverter.populatePollAccountEntity(pollAccountDTO,
                    pollAccountEntity);
                // Update pollAccountEntity
                pollAccountDAO.update(pollAccountEntity);

                voteDTO.setPollAccountId(pollAccountEntity.getTopiaId());
                voteDTO.setVotingListId(pollAccountDTO.getVotingListId());
                voteDTO.setWeight(pollAccountDTO.getWeight());
                log.debug("Poids du vote : " + voteDTO.getWeight());
            } else {
                voteDTO.setPollAccountId(null);
            }

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);

            Vote voteEntity = voteDAO.create();
            converter.setTransaction(transaction);
            converter.populateVoteEntity(voteDTO, voteEntity);

            transaction.commitTransaction();

            converter.populateChoiceVote(voteDTO, voteEntity);

            topiaId = voteEntity.getTopiaId();
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (Exception eee) {
            ContextUtil.doCatch(eee, transaction, "Problem when creating vote with pollAccount");
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean updateVote(VoteDTO voteDTO) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteDTO.getId());
            converter.setTransaction(transaction);
            converter.populateVoteEntity(voteDTO, voteEntity);
            voteEntity.update();

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + voteDTO.getId());
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean deleteVote(String voteId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteId);
            voteEntity.delete();

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + voteId);
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<VoteDTO> selectVotes(Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<VoteDTO> results = null;
        List<Vote> voteEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);

            if (properties == null) {
                voteEntities = voteDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les votes ont été sélectionnés !");
                }
            } else {
                voteEntities = voteDAO.findAllByProperties(properties);
            }

            converter.setTransaction(transaction);
            results = converter.createVoteDTOs(voteEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }
}