/*
 * #%L
 * bow
 * 
 * $Id: BookmarkUtils.java 192 2011-02-14 14:53:06Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/bow/tags/bow-0.4/src/main/java/org/chorem/bow/BookmarkUtils.java $
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import org.nuiton.util.StringUtil;

import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.FacetTopic;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.Search;

public class BookmarkUtils {

    static public Criteria getBookmarkListCriteriaByUser(User user, String searchLine) {
        Criteria criteria = null;
        if (user != null) {
            if (searchLine != null && !searchLine.isEmpty()) {
                String[] words = searchLine.split("\\s+"); //Puts the tags in an array
                List<String> tags = new ArrayList<String>(Arrays.asList(words));
                criteria = Search.query().eq(Bookmark.FQ_FIELD_BOOKMARK_EMAIL, user.getEmail()).
                        eq(Bookmark.FQ_FIELD_BOOKMARK_TAGS, tags).criteria().addFacetField(Bookmark.FQ_FIELD_BOOKMARK_TAGS);
            } else {
                criteria = Search.query().eq(Bookmark.FQ_FIELD_BOOKMARK_EMAIL, user.getEmail()).
                        criteria().addFacetField(Bookmark.FQ_FIELD_BOOKMARK_TAGS);
            }
        }
        return criteria;
    }

    static public BookmarkActions createBookmarkActions(HttpServletRequest request, PagedResult<Bookmark> result, String searchLine) {
        String fullText = request.getParameter("fullTextLine");
        BookmarkActions bookmarkActions = new BookmarkActions();
        bookmarkActions.setFullTextLine(fullText);
        List<Bookmark> bookList = result.getAll();

        if (bookList != null) {
            bookmarkActions.setBookmarks(bookList);
        }
        if (fullText == null || fullText.isEmpty()) {
            if (searchLine != null && searchLine.isEmpty()) {
                bookmarkActions.emptySearchline();
            } else {
                bookmarkActions.addTags(searchLine); //Adds the new tags
            }
        }
        List<FacetTopic> topics = result.getTopic(Bookmark.FQ_FIELD_BOOKMARK_TAGS);
        bookmarkActions.createTagCloud(topics);
        return bookmarkActions;
    }

    static public List<FacetTopic> getBookmarksByImportDate(HttpServletRequest request, User user) {
        BowSession session = BowSession.getBowSession(request);
        WikittyProxy proxy = session.getProxy();
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Import.EXT_IMPORT).eq(Bookmark.FQ_FIELD_BOOKMARK_EMAIL, user.getEmail()).criteria().
                addFacetField(Import.FQ_FIELD_IMPORT_DATE);
        List<FacetTopic> bookmarksImportDate =
                proxy.findAllByCriteria(Import.class, criteria).getTopic(Import.FQ_FIELD_IMPORT_DATE);
        request.setAttribute("bookmarksImportDate", bookmarksImportDate);
        return bookmarksImportDate;
    }

    /* @param html      String html
     * @param name      String name
     * @param user      User user
     * @return Bookmark the bookmark filled
     */
    public static Bookmark createBookmarkFromHtml(String html, String name, User user) {
        String link = getUrlFromHtml(html);
        String tags = getTagsFromHtml(html);
        Date date = getDateFromHtml(html);
        Bookmark bookmark = createBookmark(link, name, tags, user, null, date);
        return bookmark;
    }

    /* @param html      String html
     * @return Date     from the html
     */
    public static Date getDateFromHtml(String html) {
        Date date = null;
        if (html != null) {
            Pattern p = Pattern.compile("ADD_DATE=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                String str = m.group(1);
                if (str != null && !str.isEmpty()) {
                    long time = Long.valueOf(str) * 1000; // the date in ms
                    date = new Date(time);
                }
            }
        }
        return date;
    }

    /* @param bookmarks List<Bookmarks> bookmarks
     * @return String   the html file
     */
    public static String getExportHtmlBookmark(List<Bookmark> bookmarks) {
        String export;
        export = "<!DOCTYPE NETSCAPE-Bookmark-file-1>\n"
                + "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">\n"
                + "<DL><p>\n";
        if (bookmarks != null) {
            for (Bookmark bookmark : bookmarks) {
                export += "<DT><A HREF=\"";
                export += bookmark.getLink() + "\" ";
                Date date = bookmark.getDate();
                if (date != null) {
                    long milli = date.getTime();
                    String time = String.valueOf(milli / 1000);
                    export += "ADD_DATE=\"" + time + "\" LAST_MODIFIED=\"" + time + "\" ";
                }
                export += "LAST_CHARSET=\"UTF-8\" ";
                Set<String> tags = bookmark.getTags();
                if (tags != null && !tags.isEmpty()) {
                    export += "SHORTCUTURL=\"" + getBookmarkTagsString(bookmark) + "\"";
                }
                export += ">" + bookmark.getDescription() + "</A>\n";
            }
        }
        export += "</DL><p>";
        return export;
    }

    /* @param html   String html
     *
     */
    public static String getUrlFromHtml(String html) {
        String url = "";
        if (html != null) {
            Pattern p = Pattern.compile("A HREF=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                url = m.group(1);
            }
        }
        return url;
    }

    public static String getTagsFromHtml(String html) {
        String tags = "";
        if (html != null) {
            Pattern p = Pattern.compile("SHORTCUTURL=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                tags = m.group(1);
            }
        }
        return tags;
    }

    /* @param url           String which contains the bookmark url
     * @param nameAndTags   String which contains the name and the tags of the
     *                      bookmark separated by '|' ==> name|tag1 tag2...
     * @param user          bookmark user
     * @return null         if all fields are not correctly filled
     * @return Bookmark     the website bookmark
     */
    public static Bookmark createBookmark(String url, String nameAndTags, User user) {

        Bookmark bookmark = new BookmarkImpl();

        int nameIndex = nameAndTags.indexOf('|'); // get the index name of the website
        if (nameIndex > 0) {
            String name = nameAndTags.substring(0, nameIndex); // get the website name
            if (!name.isEmpty()) {
                name = name.trim();
                bookmark.setDescription(name); // set the description (website name)
                bookmark.setAlias(name);
            }
        }
        String tags = nameAndTags;
        if (nameIndex >= 0) {
            tags = nameAndTags.substring(nameIndex + 1); // get tags, +1 because of '|'
        }
        addTagsToBookmark(tags, bookmark);
        addUrlToBookmark(url, bookmark);
        bookmark.setClick(0);
        bookmark.setEmail(user.getEmail()); // set the email (user name)
        bookmark.setDate(new Date()); // set the date
        return bookmark;
    }

    public static void addUrlToBookmark(String url, Bookmark bookmark) {
        if (url != null && !url.isEmpty()) {
            String link = url.trim();
            if (! link.matches("[a-z]+://.+") ) {
                link = "http://" + link;
            }
            bookmark.setLink(link);
        } else {
            bookmark.setLink("");
        }
    }

    public static void addTagsToBookmark(String tags, Bookmark bookmark) {
        if (tags != null && !tags.isEmpty()) {
            String[] tagsTab = tags.split("\\s+"); // put the tags in an array
            List<String> tagList = Arrays.asList(tagsTab);
            addTagsToBookmark(tagList, bookmark);
        }
    }

    public static void addTagsToBookmark(List<String> tagList, Bookmark bookmark) {
        if (tagList != null) {
            for (String tag : tagList) {
                if (tag != null && !tag.isEmpty()) {
                    // on peut ajouter plusieurs fois le meme tag, car on les
                    // stocke dans un Set
                    bookmark.addTags(tag);
                }
            }
        }
    }

    /**
     * Create new Bookmark
     *
     * @param url
     * @param name
     * @param tags
     * @param user
     * @param alias
     * @param date
     * @return
     */
    public static Bookmark createBookmark(String url, String name, String tags,
            User user, String alias, Date date) {
        Bookmark bookmark = new BookmarkImpl();
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        } else {
            bookmark.setDescription("");
        }

        addTagsToBookmark(tags, bookmark);

        addUrlToBookmark(url, bookmark);

        bookmark.setClick(0);

        bookmark.setEmail(user.getEmail()); // set the email (user name)
        if (date == null) {
            date = new Date();
        }
        bookmark.setDate(date); // set the date
        if (alias != null) {
            alias = alias.trim();
            bookmark.setAlias(alias);
        } else {
            bookmark.setAlias("");
        }
        return bookmark;
    }

    public static void updateBookmarkName(Bookmark bookmark, String name) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
    }

    public static void updateBookmark(Bookmark bookmark, String name, String link, String tags, String alias) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
        addUrlToBookmark(link, bookmark);
        bookmark.clearTags();
        addTagsToBookmark(tags, bookmark);
        if (alias != null) {
            alias = alias.trim();
            bookmark.setAlias(alias);
        }
    }

    public static String getBookmarkTagsString(Bookmark bookmark) {
        if (bookmark != null) {
            Set<String> tags = bookmark.getTags();
            if (tags != null) {
                return StringUtil.join(tags, " ", true);
            }
        }
        return "";
    }

}
