package fr.onema.sispea.struts.exchange.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.common.MessageType;
import fr.onema.sispea.service.exchange.ExchangeMessageDto;
import fr.onema.sispea.service.exchange.FileType;
import fr.onema.sispea.service.exchange.ImportParameters;
import fr.onema.sispea.service.exchange.UploadedFileDto;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Locale;

/**
 * This action validates the uploaded file.
 * It allows the end user to upload data and referential.
 *
 * The validation process is asynchronous.
 * Thus the action is not waiting for the validation process to end.
 *
 * The user will be notified by e-mail of the validation status.
 *
 * @author CS
 */
public class ValidateFileAction extends AbstractFileUploadAction {

    protected static final Locale DEFAULT_LOCALE = Locale.FRENCH;

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ValidateFileAction.class);

    /**
     * The uploaded file.
     */
    protected File upload;

    /**
     * The content type of the uploaded file.
     */
    protected String uploadContentType;

    /**
     * The name of the uploaded file.
     */
    protected String uploadFileName;

    /**
     * The caption of the file entered by user.
     */
    protected String fileCaption;

    /**
     * Indicates whether the uploaded file has been submitted for validation.
     */
    protected boolean validated;

    /**
     * Indicates whether the data of the uploaded file can be added to the db.
     */
    protected boolean fileValid;

    /**
     * The messages returned from the file upload validation.
     */
    protected Collection<ExchangeMessageDto> messages;

    @Override
    protected String doExecute() throws Exception {

        if (logger.isDebugEnabled()) {
            logger.debug("Validating the uploaded file");
            logger.debug("File: " + upload.getAbsolutePath());
            logger.debug("Filename: " + uploadFileName);
            logger.debug("Content type: " + uploadContentType);
            logger.debug("File caption: " + fileCaption);
        }

        // Nothing to do but menu
        openMenu(MenuConstants.MENU_KEY_SENDDATA);

        if (upload == null) {
            String msg = "No uploaded file";
            logger.debug(msg);
            throw new SispeaException(msg);
        }

        // Validate the file
        // The uploaded file has been submitted for validation.
        validated = true;
        messages = new ArrayList<>();
        try {
            messages.add(new ExchangeMessageDto("", MessageType.INFO, "fr.onema.sispea.exchange.validating.file", uploadFileName));

            // Pass the file to the file reader that parses the file to create the beans

            // Get the content type of the uploaded file
            // The uploaded file name has already been validated by struts. No need do redo it
            FileType fileType;
            if (uploadFileName.toLowerCase(DEFAULT_LOCALE).endsWith("xml")) {
                fileType = FileType.XML;
            } else if (uploadFileName.toLowerCase(DEFAULT_LOCALE).endsWith("csv")) {
                fileType = FileType.CSV;
            } else {
                throw new SispeaException("Cannot handle file " + uploadFileName + ". Unknown mime type.");
            }

            // Check if it's not a standard import
            if (standardImport.equals(Boolean.FALSE)) {
                if (!getCurrentUser().isGeneralAdmin()) {
                    // A non standard import is forbidden to a user not credited of Admin role
                    messages.add(new ExchangeMessageDto("", MessageType.ERROR, "fr.onema.sispea.exchange.upload.file.validation.badUser"));
                    throw new SispeaException("User is not general administrator");
                } else if (!fileType.equals(FileType.XML)) {
                    // A non standard import is only allowed with XML files
                    messages.add(new ExchangeMessageDto("", MessageType.ERROR, "fr.onema.sispea.exchange.upload.file.validation.badFormat"));
                    throw new SispeaException("File type CSV is not allowed for non standard imports");
                }
            }

            UploadedFileDto importedFile = new UploadedFileDto(uploadFileName, upload, fileType);
            // if user is here, he has view right
            ImportParameters parameters = new ImportParameters(getSispeaSession().getUser(),
                                                               messages,
                                                               getLocale().getLanguage(),
                                                               checkUserRight(Right.View),
                                                               importedFile,
                                                               true);

            if (standardImport.equals(Boolean.FALSE)) {
                parameters.setAdminImport(true);
                if (overwriteData != null) {
                    parameters.setOverwriteData(overwriteData);
                } else {
                    parameters.setOverwriteData(false);
                }
                if (preserveStatus != null) {
                    parameters.setPreserveStatus(preserveStatus);
                } else {
                    parameters.setPreserveStatus(false);
                }
                parameters.setStatusId(statusId);
                parameters.setOriginTypeId(originTypeId);
                parameters.setComment(comment);
            }

            // Submit the file for validation
            fileExchangeService.validateUploadedFile(parameters);

            //
            // Update the lastUploadDate
            //
            userService.updateLastUploadDateWithCurrentTime(getCurrentUser());

            // The uploaded file is ok, access rights are granted.
            // The file has been submitted for validation.
            messages.add(new ExchangeMessageDto("", MessageType.INFO, "fr.onema.sispea.exchange.validation.in.progress"));
        } catch (Exception e) {
            // Transform the exception into an error message
            messages.add(new ExchangeMessageDto("", MessageType.ERROR, "fr.onema.sispea.exchange.error.validating.file", uploadFileName));
            logger.error("Error caught while processing the uploaded file", e);
        }

        // Loop through the error messages to check if the file may be added
        fileValid = true;
        for (Iterator<ExchangeMessageDto> msgIter = messages.iterator(); msgIter.hasNext() && fileValid; ) {
            ExchangeMessageDto msgBean = msgIter.next();
            if (MessageType.ERROR.equals(msgBean.getType())) {
                fileValid = false;
            }
        }

        // Put the messages into the session
        getSispeaSession().setUploadedFileMsgs(messages);

        return SUCCESS;
    }

    /**
     * Returns true if the uploaded file has been submitted for validation.
     */
    public boolean isValidated() {
        return validated;
    }

    public boolean isFileValid() {
        return fileValid;
    }

    public Collection<ExchangeMessageDto> getMessages() {
        return messages;
    }

    public File getUpload() {
        return upload;
    }

    public void setUpload(File pUpload) {
        upload = pUpload;
    }

    public String getUploadContentType() {
        return uploadContentType;
    }

    public void setUploadContentType(String pUploadContentType) {
        uploadContentType = pUploadContentType;
    }

    public String getUploadFileName() {
        return uploadFileName;
    }

    public void setUploadFileName(String pUploadFileName) {
        uploadFileName = pUploadFileName;
    }

    public String getFileCaption() {
        return fileCaption;
    }

    public void setFileCaption(String pFileCaption) {
        fileCaption = pFileCaption;
    }
}
