package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.common.PagingDto;
import fr.onema.sispea.service.data.DataRatioDto;
import fr.onema.sispea.service.data.SPEALotStatus;
import fr.onema.sispea.service.referential.DDTDto;
import fr.onema.sispea.service.referential.DdtRefUpdateDto;
import fr.onema.sispea.service.referential.MunicipalityDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.SPEADto;
import fr.onema.sispea.service.referential.SPEATransferedDto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This action display a municipality.
 *
 * @author CS
 */
public class ShowMunicipalityAction extends AbstractReferentialCardAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ShowMunicipalityAction.class);

    /**
     * Municipality.
     */
    protected MunicipalityDto municipality;

    /**
     * Store for each spea the lot status associated.
     */
    protected Map<Integer, SPEALotStatus> speaLotStatusMap = new HashMap<>();

    /**
     * Store for each spea the ratio associated.
     */
    protected Map<Integer, DataRatioDto> speaDataRatioMap = new HashMap<>();

    /**
     * The paging information for services.
     */
    protected PagingDto pagingService = new PagingDto();

    /**
     * Territory associated to current organism displayed.
     */
    protected TerritoryDto territory;

    /**
     * True if the referential of the municipality has been uploaded for the requested exercise.
     */
    protected boolean hasReferential = true;

    @Override
    protected String doExecute() throws Exception {
        String result = SUCCESS;

        if (logger.isDebugEnabled()) {
            logger.debug("show Municipality");
        }

        // Open menu according to the current JSP item.
        openMenu(MenuConstants.MENU_KEY_SEE_MUNICIPALITY);

        // No right check : data visible for everybody

        // Check parameters
        checkInputParameters();

        manageViewActions();

        Integer exerciceId = getExercise().getId();
        for (SPEATransferedDto speaTransferedDto : getMunicipality().getSortedSpeas()) {

            SPEADto spea = speaTransferedDto.getSpea();
            Integer speaId = spea.getId();

            if (logger.isDebugEnabled()) {
                logger.debug("Loading data for spea " + speaId);
            }

            SPEALotStatus speaLotStatus = speaLotService.readSpeaLotStatus(speaId, exerciceId);
            speaLotStatusMap.put(speaId, speaLotStatus);

            DataRatioDto displayRatio = speaLotService.getDisplayRatio(speaId, exerciceId, spea.getHasViewAccess());
            speaDataRatioMap.put(speaId, displayRatio);

        }

        return result;
    }

    public SPEALotStatus getSpeaLotStatusMap(Integer speaId) {
        return speaLotStatusMap.get(speaId);
    }

    public DataRatioDto getSpeaDataRatio(Integer speaId) {
        return speaDataRatioMap.get(speaId);
    }

    public TerritoryDto getTerritory() {
        return territory;
    }

    public void setTerritory(TerritoryDto territory) {
        this.territory = territory;
    }

    public PagingDto getPagingService() {
        return pagingService;
    }

    public boolean isHasReferential() {
        return hasReferential;
    }

    public MunicipalityDto getMunicipality() {
        return municipality;
    }

    public void setMunicipality(MunicipalityDto pMunicipality) {
        municipality = pMunicipality;
    }

    /**
     * This method verifies the input parameters: municipality id and exercise.
     *
     * @throws SispeaException
     */
    protected void checkInputParameters() throws SispeaException {

        checkId();
        checkExercise();
    }

    protected void manageViewActions() throws SispeaException {

        // If initial state or exercise has changed, load the Municipality.
        // Initialize Session
        initSession();

        // Get Municipality model instance
        municipality = referentialService.readMunicipality(orgId, exercise);

        territory = territoryService.readTerritory(municipality.getId(), OrganismType.Municipality);

        // The JSP will loop over the speas of the municipality
        // It will display a link to the data of the spea lot for the current exercise
        // if the user has access right to this data.
        // If not, the end user should be presented only the status.
        //
        // Loop through the spea lot to update this access right
        for (SPEATransferedDto transferedSpea : municipality.getSpeas()) {

            // Compute the access right.
            // See the javadoc for the constraints
            boolean hasViewAccess = hasViewAccess(transferedSpea.getSpea(), exercise);
            transferedSpea.getSpea().setHasViewAccess(hasViewAccess);
        }

        if (municipality.getSpeas().isEmpty()) {

            hasReferential = false;

            // If there is not any services, check if the DDT has uploaded the referential
            // Get the DDT of the municipality
            municipality.setTerritory(territoryService.readTerritory(municipality));
            DDTDto ddt = referentialService.readAssociatedDdtFromTerritory(municipality.getTerritory(), exercise);

            // There are some municipalities without any DDT. For these municipalities, consider that there
            // is not any referential
            if (ddt != null) {
                Collection<DdtRefUpdateDto> updates = ddt.getDdtUpdates();

                // Loop through the uploaded referential to find to one for the requested exercise
                for (Iterator<DdtRefUpdateDto> refUpdateIter = updates.iterator(); refUpdateIter.hasNext() && !hasReferential; ) {
                    DdtRefUpdateDto ddtRefUpdate = refUpdateIter.next();
                    if (exercise.getId().intValue() == ddtRefUpdate.getExercise().getId().intValue()) {
                        hasReferential = true;
                    }
                }
            }

            if (logger.isDebugEnabled()) {
                if (hasReferential) {
                    logger.debug("Referential uploaded for municipality " + municipality.getName() + ", exercise " + exercise.getYear());
                } else {
                    logger.debug("No referential uploaded for municipality " + municipality.getName() + ", exercise " + exercise.getYear());
                }
            }
        }

        // paging
        manageServicePaging();
    }

    /**
     * manage services (SPEA) paging after municipality loading.
     */
    protected void manageServicePaging() {

        int lCount = municipality.getSpeas().size();

        // set member result count
        pagingService.setResultCount(lCount);

        // get start and end idx to extract
        Integer lStartIdx = pagingService.getStartIndex() * pagingService.getPageSize();
        Integer lEndIdx = lStartIdx + pagingService.getPageSize();
        if (lEndIdx >= lCount) {
            lEndIdx = lCount;
        }

        ArrayList<SPEATransferedDto> lMembers = (ArrayList<SPEATransferedDto>) municipality.getSpeas();

        // make sub list
        List<SPEATransferedDto> lSubList = lMembers.subList(lStartIdx, lEndIdx);

        // set sub list
        municipality.setSpeas(lSubList);

        // set page offset
        pagingService.setPageOffset(pagingService.getStartIndex());

    }

}
