package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import com.opensymphony.xwork2.Action;
import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.common.PagingDto;
import fr.onema.sispea.service.data.DataRatioDto;
import fr.onema.sispea.service.data.ExerciseDto;
import fr.onema.sispea.service.data.SPEALotStatus;
import fr.onema.sispea.service.exchange.ExportParameters;
import fr.onema.sispea.service.exchange.FileExchangeService;
import fr.onema.sispea.service.exchange.FileType;
import fr.onema.sispea.service.referential.CollectivityMemberDto;
import fr.onema.sispea.service.referential.CollectivityType;
import fr.onema.sispea.service.referential.DDTDto;
import fr.onema.sispea.service.referential.DepartmentDto;
import fr.onema.sispea.service.referential.OrganismDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.SPEADto;
import fr.onema.sispea.service.referential.SPEATransferedDto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.user.MandateDto;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This action display a collectivity.
 *
 * @author CS
 */
public class ShowCollectivityAction extends AbstractCollectivityAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ShowCollectivityAction.class);

    @Autowired
    protected transient FileExchangeService fileExchangeService;

    /**
     * Store for each spea the lot status associated.
     */
    protected Map<Integer, SPEALotStatus> speaLotStatusMap = new HashMap<>();

    /**
     * Store for each spea the ratio associated.
     */
    protected Map<Integer, DataRatioDto> speaDataRatioMap = new HashMap<>();

    /**
     * Store for each transfered spea the lot status associated.
     */
    protected Map<Integer, SPEALotStatus> transferedSpeaLotStatusMap = new HashMap<>();

    /**
     * Store for each transfered spea the ratio associated.
     */
    protected Map<Integer, DataRatioDto> transferedSpeaDataRatioMap = new HashMap<>();

    /**
     * Store for collectivity member his ddt or departement name.
     */
    protected Map<CollectivityMemberDto, String> memberDepartementOrDdtName = new HashMap<>();

    protected DepartmentDto department;

    /**
     * true if current user can view this territory's users
     */
    protected boolean canViewUsers;

    protected boolean userHasCheckRights;

    protected boolean userHasMandateRights;

    protected boolean userHasRights;

    protected int associatedTerritoryId;

    /**
     * The paging information for members.
     */
    protected PagingDto pagingMember = new PagingDto();

    /**
     * The paging information for services.
     */
    protected PagingDto pagingService = new PagingDto();

    /**
     * The paging information for transfered spea.
     */
    protected PagingDto pagingTransfered = new PagingDto();

    /**
     * The mandate paging information.
     */
    protected PagingDto mandatePaging = new PagingDto();

    /**
     * The rpqs link.
     */
    protected String rpqsName;

    /**
     * The rpqs type.
     */
    protected String rpqsType;

    /**
     * The rpqs link.
     */
    protected String rpqsLink;

    /**
     * Territory associated to current organism displayed.
     */
    protected TerritoryDto territory;

    /**
     * Flag to know if RPQS Publication is currently modified.
     */
    protected boolean modifPublication;

    /**
     * true if current user can view this territory's mandates
     */
    protected boolean canViewMandates;

    /**
     * true if current user can mandate on this territory
     */
    protected boolean canMandate;

    /**
     * Input parameter.
     * Set to true to generate the RPQS for the collectivity.
     */
    protected boolean generateRpqs;

    /**
     * The list of spea whose competence is transfered to
     */
    protected List<SPEATransferedDto> transferedSpeas;

    /**
     * The list of mandates concerning all services of the collectivity.
     */
    protected List<MandateDto> mandates;

    public ShowCollectivityAction() {
        edit = false;
        generateRpqs = false;
        mandates = new ArrayList<>();
    }

    @Override
    protected String doExecute() throws Exception {

        // Initialize the next page
        String nextPage = SUCCESS;

        if (logger.isDebugEnabled()) {
            logger.debug("show collectivity");
        }

        // Open menu according to the current JSP item.
        openMenu(MenuConstants.MENU_KEY_SEE_COLLECTIVITY);

        // no right check : data visible for everybody

        try {
            // Check input parameters
            validateInputParameters();

            manageViewActions();

            associatedTerritoryId = getAssociatedTerritoryId(territory);

            Integer exerciceId = getExercise().getId();
            for (SPEADto spea : getCollectivity().getSortedSpeas()) {

                Integer speaId = spea.getId();

                if (logger.isDebugEnabled()) {
                    logger.debug("Loading data for spea " + speaId);
                }

                SPEALotStatus speaLotStatus = speaLotService.readSpeaLotStatus(speaId, exerciceId);
                speaLotStatusMap.put(speaId, speaLotStatus);

                DataRatioDto displayRatio = speaLotService.getDisplayRatio(speaId, exerciceId, spea.getHasViewAccess());
                speaDataRatioMap.put(speaId, displayRatio);

            }

            for (SPEATransferedDto speaTransferedDto : getTransferedSpeas()) {

                SPEADto spea = speaTransferedDto.getSpea();
                Integer speaId = spea.getId();

                if (logger.isDebugEnabled()) {
                    logger.debug("Loading data for transfered spea " + speaId);
                }

                SPEALotStatus speaLotStatus = speaLotService.readSpeaLotStatus(speaId, exerciceId);
                transferedSpeaLotStatusMap.put(speaId, speaLotStatus);

                DataRatioDto displayRatio = speaLotService.getDisplayRatio(speaId, exerciceId, spea.getHasViewAccess());
                transferedSpeaDataRatioMap.put(speaId, displayRatio);

            }

            for (CollectivityMemberDto collectivityMember : getCollectivity().getCollectivityMembership().getMembers()) {

                OrganismDto organism = collectivityMember.getOrganism();

                String name;
                Integer organismTypeId = (int) organism.getType().getDbId();
                if (organismTypeId == 4) {
                    DDTDto ddt = readDdt(organism.getId(), organismTypeId);
                    name = ddt.getName();

                } else {

                    DepartmentDto department = readDepartment(organism.getId(), organismTypeId);
                    name = department.getName();

                }

                memberDepartementOrDdtName.put(collectivityMember, name);

            }

            userHasCheckRights = userHasCheckRights(territory, exercise);
            userHasMandateRights = userHasMandateRights(getCurrentUser(), territory, exercise);
            userHasRights = userHasRights(territory, exercise);

        } catch (SispeaException e) {
            nextPage = Action.INPUT;
        }

        return nextPage;
    }

    /**
     * Validates the input parameters.
     *
     * @throws SispeaException
     */
    protected void validateInputParameters() throws SispeaException {

        // Check collectivity id and exercise
        checkId();
        checkExercise();
    }

    public SPEALotStatus getSpeaLotStatus(Integer speaId) {
        return speaLotStatusMap.get(speaId);
    }

    public DataRatioDto getSpeaDataRatio(Integer speaId) {
        return speaDataRatioMap.get(speaId);
    }

    public SPEALotStatus getTransferedSpeaLotStatusMap(Integer speaId) {
        return transferedSpeaLotStatusMap.get(speaId);
    }

    public DataRatioDto getTransferedSpeaDataRatio(Integer speaId) {
        return transferedSpeaDataRatioMap.get(speaId);
    }

    public String getMemberDepartementOrDdtName(CollectivityMemberDto member) {
        return memberDepartementOrDdtName.get(member);
    }

    public boolean isCanViewUsers() {
        return canViewUsers;
    }

    public void setCanViewUsers(boolean canViewUsers) {
        this.canViewUsers = canViewUsers;
    }

    public boolean isUserHasCheckRights() {
        return userHasCheckRights;
    }

    public boolean isUserHasMandateRights() {
        return userHasMandateRights;
    }

    public boolean isUserHasRights() {
        return userHasRights;
    }

    public boolean isCanMandate() {
        return canMandate;
    }

    public int getAssociatedTerritoryId() {
        return associatedTerritoryId;
    }

    public void setAssociatedTerritoryId(int associatedTerritoryId) {
        this.associatedTerritoryId = associatedTerritoryId;
    }

    public String getRpqsName() {
        return rpqsName;
    }

    public void setRpqsName(String rpqsName) {
        this.rpqsName = rpqsName;
    }

    public String getRpqsType() {
        return rpqsType;
    }

    public void setRpqsType(String rpqsType) {
        this.rpqsType = rpqsType;
    }

    public String getRpqsLink() {
        return rpqsLink;
    }

    public void setRpqsLink(String rpqsLink) {
        this.rpqsLink = rpqsLink;
    }

    public TerritoryDto getTerritory() {
        return territory;
    }

    public void setTerritory(TerritoryDto territory) {
        this.territory = territory;
    }

    public boolean getModifPublication() {
        return modifPublication;
    }

    public void setModifPublication(boolean modifPublication) {
        this.modifPublication = modifPublication;
    }

    public boolean isCanViewMandates() {
        return canViewMandates;
    }

    public void setCanViewMandates(boolean canViewMandates) {
        this.canViewMandates = canViewMandates;
    }

    public boolean getGenerateRpqs() {
        return generateRpqs;
    }

    public void setGenerateRpqs(boolean pRpqs) {
        generateRpqs = pRpqs;
    }

    public String getRpqsDownloadUrl() {
        return RPQS_DOWNLOAD_URL;
    }

    public PagingDto getPagingMember() {
        return pagingMember;
    }

    public PagingDto getPagingService() {
        return pagingService;
    }

    public PagingDto getPagingTransfered() {
        return pagingTransfered;
    }

    public PagingDto getMandatePaging() {
        return mandatePaging;
    }

    public List<SPEATransferedDto> getTransferedSpeas() {
        return transferedSpeas;
    }

    public void setTransferedSpeas(List<SPEATransferedDto> pTransferedSpeas) {
        transferedSpeas = pTransferedSpeas;
    }

    public List<MandateDto> getMandates() {
        return mandates;
    }

    /**
     * @param pUserLogin The login to test
     * @return true if user's login is in mandated list
     */
    public boolean isUserMandated(String pUserLogin) {
        boolean isUserMandated = false;

        if (pUserLogin != null) {
            // Loop through the mandates
            for (Iterator<MandateDto> mandateIter = mandates.iterator(); mandateIter.hasNext() && !isUserMandated; ) {
                MandateDto mandate = mandateIter.next();
                if (pUserLogin.equals(mandate.getMandated().getLogin())) {
                    isUserMandated = true;
                }
            }
        }
        return isUserMandated;
    }

    /**
     * This method test if the user has checking rights on this collectivity.
     * If the user's territory equals the currently displayed collectivity's territory, then, user has RPQS publication rights.
     *
     * @return true if user has RPQS publication rights, false otherwise.
     * @throws SispeaException
     */
    protected boolean userHasCheckRights(TerritoryDto territory, ExerciseDto exercise) throws SispeaException {

        // init
        boolean lRights = false;

        if (collectivity != null && territory != null) {

            lRights = checkUserRight(Right.Check, territory, exercise);

        }

        return lRights;
    }

    /**
     * This method test if the user has mandate rights on this collectivity.
     * If the user's territory equals the currently displayed collectivity's territory, then, user has RPQS publication rights.
     *
     * @return true if user has RPQS publication rights, false otherwise.
     * @throws SispeaException
     */
    protected boolean userHasMandateRights(UserDto currentUser, TerritoryDto territory, ExerciseDto exercise) throws SispeaException {

        // init
        Boolean lRights = false;

        // check rights
        if (collectivity != null) {

            // check if this user has a mandate on one of the spea of this collectivity
            int lMandatesCount = userService.countMandatesWhereMandatedOnCollectivity(currentUser, territory, exercise);

            // mandates ?
            if (lMandatesCount > 0) {
                lRights = true;
            }

        }

        return lRights;
    }

    /**
     * This method test if the user has publishing rights on this collectivity.
     * If the user's territory equals the currently displayed collectivity's territory, then, user has RPQS publication rights.
     *
     * @return true if user has RPQS publication rights, false otherwise.
     * @throws SispeaException
     */
    protected boolean userHasRights(TerritoryDto territory, ExerciseDto exercise) throws SispeaException {

        boolean lRights = false;

        if (collectivity != null && territory != null) {

            lRights = checkUserRight(Right.Edit, territory, exercise);

        }

        return lRights;
    }

    protected void manageViewActions() throws SispeaException {

        try {

            // Initialize the session
            initSession();

            if (orgId == null) {
                throw new SispeaException(getText("fr.onema.sispea.jsp.fixRef.showCollectivity.error.noCollectivityId"));
            }

            // Get collectivity model instance.
            collectivity = referentialService.readCollectivity(orgId, exercise, false);

            if (collectivity == null) {
                throw new SispeaException(getText("fr.onema.sispea.jsp.fixRef.showCollectivity.error.noCollectivity", "", orgId.toString()));
            }

            // load territory
            territory = territoryService.readTerritory(collectivity.getId(), OrganismType.Collectivity);

            // Compute the view access right to the SPEA lots of the exercise exercise_
            for (SPEADto spea : collectivity.getSpeas()) {
                boolean hasViewAccess = hasViewAccess(spea, exercise);
                spea.setHasViewAccess(hasViewAccess);
            }

            // Load associated users.
            loadUsers(territory);

            // load transfered speas
            // If it is the first page, then get the number of results too
            if (pagingTransfered.getPageOffset() == PagingDto.FIRST_PAGE - 1) {
                int resultCount = referentialService.countTransferedSpeas(collectivity,
                                                                          exercise,
                                                                          collectivity.getCollType() == CollectivityType.Municipality);
                pagingTransfered.setResultCount(resultCount);
                if (logger.isDebugEnabled()) {
                    logger.debug(resultCount + " transfered spea(s) found");
                }
            }
            transferedSpeas = referentialService.readTransferedSpeas(collectivity,
                                                                     exercise,
                                                                     collectivity.getCollType() == CollectivityType.Municipality,
                                                                     pagingTransfered.getStartIndex() * pagingTransfered.getPageSize(),
                                                                     pagingTransfered.getPageSize());
            pagingTransfered.setPageOffset(pagingTransfered.getStartIndex());

            // Compute the view access right to the transfered SPEA lots of the exercise exercise_
            for (SPEATransferedDto transferedSpea : transferedSpeas) {
                boolean hasViewAccess = hasViewAccess(transferedSpea.getSpea(), exercise);
                transferedSpea.getSpea().setHasViewAccess(hasViewAccess);
            }

            // Do we need to load the mandates?
            canViewMandates = computeCanViewMandates(getCurrentUser(), exercise, territory);
            canViewUsers = computeCanViewUsers(getCurrentUser(), territory, exercise);
            if (canViewMandates) {

                canMandate = canMandate(getCurrentUser(), territory);

                // Load all mandates
                // Loop through the speas
                mandates.clear();
                for (SPEADto spea : collectivity.getSpeas()) {

                    // Do not ask the service to do the paging
                    spea.setMandates(userService.readSpeaMandates(spea.getId()));
                    mandates.addAll(spea.getMandates());
                }
                // Page the mandates
                manageMandatePaging();
            }

            // Change menu ?
            try {
                openMenu(getProperMenuKey(territory));
            } catch (Exception e) {
                throw new SispeaException(e);
            }

            // Paging
            manageMemberPaging();
            manageServicePaging();
            manageTransferedPaging();

            // Do we need to generate the RPQS
            if (Boolean.TRUE.equals(generateRpqs)) {

                // Generate the RPQS
                generateRpqs(getCurrentUser(), collectivity, exercise);

                // Set the action message
                addActionMessage(getText("fr.onema.sispea.jsp.fixRef.showCollectivity.rpqs.generating"));
            }

        } catch (SispeaException e) {
            String lMessage = e.getMessage();
            addActionError(getText("fr.onema.sispea.jsp.fixRef.showCollectivity.error.main", "", lMessage));
            throw new SispeaException();
        }

    }

    /**
     * This method generates the RPQS for the SPEA.
     *
     * @param pCurrentUser
     * @param pOrganism
     * @param pExercise
     * @throws SispeaException
     */
    protected void generateRpqs(UserDto pCurrentUser, OrganismDto pOrganism, ExerciseDto pExercise) throws SispeaException {

        String territoryName = null;
        try {
            // Prepare the export parameters.
            //
            // The export is run with the SPEA territory
            TerritoryDto territory = territoryService.readTerritory(pOrganism);
            territoryName = territory.getName();

            // For huge territories, cut it in smaller ones
            List<TerritoryDto> territories = territoryService.buildExtractTerritoryList(territory, pExercise.getId(), pCurrentUser.getId());

            // View right
            Boolean hasViewRight = userService.checkUserRight(pCurrentUser, Right.View, territory, exercise);

            ExportParameters parameters = new ExportParameters();
            parameters.setUser(pCurrentUser);
            parameters.setTerritories(territories);
            parameters.setLang(getLocale().getLanguage());
            parameters.setHasViewRight(hasViewRight);
            parameters.setFileType(FileType.RPQS);
            parameters.setWithData(false);
            parameters.setWithReferential(false);
            parameters.setExtractTerritoryName(territoryName);
            parameters.setFileNamePrefix("RPQS_");
            parameters.setFileNameDateFormat("yyyyMMdd_HHmmss");
            parameters.setExercise(pExercise);
            // Do not set any competences. They are not needed to generate the RPQS.
            parameters.setCompetences(null);

            // Generate the export
            fileExchangeService.download(parameters);
        } catch (Exception e) {
            String msg = "Error while generating RPQS for the territory '" + territoryName + "'";
            logger.error(msg, e);
            throw new SispeaException(getText("fr.onema.sispea.jsp.fixRef.showCollectivity.rpqs.error", pOrganism.getName()));
        }
    }

    /**
     * manage equipment paging after spea load
     */
    protected void manageMandatePaging() {

        // set mandate result count
        mandatePaging.setResultCount(mandates.size());

        // Get the start and end indexes to extract
        Integer startIndex = mandatePaging.getStartIndex() * mandatePaging.getPageSize();
        Integer endIndex = startIndex + mandatePaging.getPageSize();
        if (endIndex >= mandates.size()) {
            endIndex = mandates.size();
        }

        // Make sub list
        List<MandateDto> lSubList = mandates.subList(startIndex, endIndex);

        // set sub list
        mandates = lSubList;

        // set page offset
        mandatePaging.setPageOffset(mandatePaging.getStartIndex());
    }

    /**
     * manage member paging after Collectivity loading.
     */
    protected void manageMemberPaging() {

        int lCount = collectivity.getCollectivityMembership().getMembers().size();

        // set member result count
        pagingMember.setResultCount(lCount);

        // get start and end idx to extract
        Integer lStartIdx = pagingMember.getStartIndex() * pagingMember.getPageSize();
        Integer lEndIdx = lStartIdx + pagingMember.getPageSize();
        if (lEndIdx >= lCount) {
            lEndIdx = lCount;
        }

        ArrayList<CollectivityMemberDto> lMembers = (ArrayList<CollectivityMemberDto>) collectivity.getCollectivityMembership().getMembers();

        // make sub list
        List<CollectivityMemberDto> lSubList = lMembers.subList(lStartIdx, lEndIdx);

        // set sub list
        collectivity.getCollectivityMembership().setMembers(lSubList);

        // set page offset
        pagingMember.setPageOffset(pagingMember.getStartIndex());
    }

    /**
     * manage services (SPEA) paging after Collectivity loading.
     */
    protected void manageServicePaging() {

        int lCount = collectivity.getSpeas().size();

        // set member result count
        pagingService.setResultCount(lCount);

        // get start and end idx to extract
        Integer lStartIdx = pagingService.getStartIndex() * pagingService.getPageSize();
        Integer lEndIdx = lStartIdx + pagingService.getPageSize();
        if (lEndIdx >= lCount) {
            lEndIdx = lCount;
        }

        ArrayList<SPEADto> lMembers = (ArrayList<SPEADto>) collectivity.getSpeas();

        // make sub list
        List<SPEADto> lSubList = lMembers.subList(lStartIdx, lEndIdx);

        // set sub list
        collectivity.setSpeas(lSubList);

        // set page offset
        pagingService.setPageOffset(pagingService.getStartIndex());
    }

    /**
     * manage transfered spea paging after collectivity loading.
     */
    protected void manageTransferedPaging() {

    }

    /**
     * @param currentUser
     * @param exercise
     * @param territory
     * @return true if current user can view the mandate list
     */
    protected boolean computeCanViewMandates(UserDto currentUser, ExerciseDto exercise, TerritoryDto territory) {

        boolean canViewMandates = false;

        // check user right
        if (currentUser != null && currentUser.getType().getIncludedTerritoryTypes().contains(territory.getType())) {
            try {
                canViewMandates = checkUserRight(Right.ViewMandates, territory, exercise);
            } catch (SispeaException lEx) {
                canViewMandates = false;
            }
        }

        return canViewMandates;

    }

    /**
     * @return true if current user can mandate users on the current territory
     */
    protected boolean canMandate(UserDto currentUser, TerritoryDto territory) {

        boolean canMandate = false;

        // check user right
        if ((currentUser != null) && (currentUser.getType().getIncludedTerritoryTypes().contains(territory.getType()))) {
            try {
                canMandate = checkUserRight(Right.Mandate, territory, exercise);
            } catch (SispeaException lEx) {
                canMandate = false;
            }
        }

        return canMandate;
    }

}
