package fr.inra.agrosyst.services.users;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: UserServiceImpl.java 4440 2014-10-16 17:25:26Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/users/UserServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.entities.security.UserRoleTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.referential.ImportResult;
import fr.inra.agrosyst.api.services.security.AuthorizationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.UserFilter;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.api.services.users.Users;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.EmailService;
import fr.inra.agrosyst.services.security.TrackerServiceImpl;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.StringUtil;

import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class UserServiceImpl extends AbstractAgrosystService implements UserService {

    protected static final String PROPERTY_ROLE_USER_TPIA_ID = UserRole.PROPERTY_AGROSYST_USER + "." + AgrosystUser.PROPERTY_TOPIA_ID;

    private static final Log log = LogFactory.getLog(UserServiceImpl.class);

    protected static final Function<UserRole, RoleType> GET_ROLE_TYPE = new Function<UserRole, RoleType>() {
        @Override
        public RoleType apply(UserRole userRole) {
            return userRole.getType();
        }
    };

    public static String hashPassword(String clearPassword) {
        String hashedPassword = StringUtil.encodeSHA1(Strings.nullToEmpty(clearPassword));
        return hashedPassword;
    }

    protected EmailService emailService;
    protected TrackerServiceImpl trackerService;
    protected AuthorizationService authorizationService;

    protected AgrosystUserTopiaDao agrosystUserDao;
    protected UserRoleTopiaDao userRoleDao;

    public void setEmailService(EmailService emailService) {
        this.emailService = emailService;
    }

    public void setTrackerService(TrackerServiceImpl trackerService) {
        this.trackerService = trackerService;
    }

    public void setAgrosystUserDao(AgrosystUserTopiaDao agrosystUserDao) {
        this.agrosystUserDao = agrosystUserDao;
    }

    public void setUserRoleDao(UserRoleTopiaDao userRoleDao) {
        this.userRoleDao = userRoleDao;
    }

    public void setAuthorizationService(AuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    protected void copyFromDto(UserDto dto, AgrosystUser entity) {
        entity.setFirstName(dto.getFirstName());
        entity.setLastName(dto.getLastName());
        entity.setEmail(StringUtils.lowerCase(dto.getEmail()));
        entity.setOrganisation(dto.getOrganisation());
        entity.setPhoneNumber(dto.getPhoneNumber());
        entity.setBanner(dto.getBanner());
        entity.setActive(dto.isActive());
    }

    @Override
    public boolean isEmailInUse(String email, String currentUserId) {
        boolean result = false;
        if (!Strings.isNullOrEmpty(email)) {

            AgrosystUser user = agrosystUserDao.forEmailEquals(email.toLowerCase()).findAnyOrNull();
            result = user != null && !user.getTopiaId().equals(currentUserId);
        }
        return result;
    }

    public UserDto createUser(UserDto dto, String password) {

        AgrosystUser user = agrosystUserDao.newInstance();
        copyFromDto(dto, user);
        String hashedPassword = hashPassword(password);
        user.setPassword(hashedPassword);

        AgrosystUser createdUser = agrosystUserDao.create(user);

        trackerService.userCreated(createdUser);

        getTransaction().commit();
        UserDto result = Users.TO_USER_DTO.apply(createdUser);
        return result;
    }

    @Override
    public UserDto getUser(String topiaId) {

        AgrosystUser user = agrosystUserDao.forTopiaIdEquals(topiaId).findUnique();

        UserDto result = Users.TO_USER_DTO.apply(user);
        return result;
    }

    @Override
    public UserDto updateUser(UserDto dto, String password) {

        AgrosystUser user = agrosystUserDao.forTopiaIdEquals(dto.getTopiaId()).findUnique();
        copyFromDto(dto, user);
        boolean passwordChange = false;
        if (!Strings.isNullOrEmpty(password)) {
            String hashedPassword = hashPassword(password);
            user.setPassword(hashedPassword);
            passwordChange = true;
        }
        AgrosystUser updatedUser = agrosystUserDao.update(user);

        trackerService.userModified(updatedUser, passwordChange);

        getTransaction().commit();
        UserDto result = Users.TO_USER_DTO.apply(updatedUser);
        return result;
    }

    public void unactivateUsers(Set<String> topiaIds, boolean activate) {
        if (topiaIds != null && !topiaIds.isEmpty()) {

            for (String topiaId : topiaIds) {
                AgrosystUser user = agrosystUserDao.forTopiaIdEquals(topiaId).findUnique();
                user.setActive(activate);
                AgrosystUser updatedUser = agrosystUserDao.update(user);

                trackerService.userActivation(updatedUser, activate);
            }

            getTransaction().commit();
        }
    }

    @Override
    public ResultList<UserDto> getFilteredUsers(UserFilter userFilter, boolean includeRoles) {

        ResultList<AgrosystUser> usersList = agrosystUserDao.getFilteredUsers(userFilter);

        ResultList<UserDto> result = ResultList.of(Lists.transform(usersList.getElements(), Users.TO_USER_DTO), usersList.getPager());

        // TODO AThimel 02/10/13 Improve
        if (includeRoles) {
            for (UserDto userDto : result) {
                List<UserRole> roles = userRoleDao.forProperties(PROPERTY_ROLE_USER_TPIA_ID, userDto.getTopiaId()).findAll();
                Set<RoleType> roleTypes = Sets.newLinkedHashSet(Iterables.transform(roles, GET_ROLE_TYPE));
                userDto.setRoles(roleTypes);
            }
        }

        return result;
    }

    @Override
    public List<UserDto> getNameFilteredActiveUsers(String research, Integer nbResult) {

        List<AgrosystUser> usersList = agrosystUserDao.getNameFilteredActiveUsers(research, nbResult);
        List<UserDto> result = Lists.transform(usersList, Users.TO_USER_DTO);
        return result;
    }

    @Override
    public long getUsersCount(Boolean active) {
        if (active == null) {
            return agrosystUserDao.count();
        }
        return agrosystUserDao.forActiveEquals(active).count();
    }

    public boolean askForPasswordReminder(String email) {
        Optional<AgrosystUser> any = agrosystUserDao.forEmailEquals(email.toLowerCase()).tryFindUnique();
        boolean result = false;
        if (any.isPresent()) {
            String token = UUID.randomUUID().toString();
            AgrosystUser agrosystUser = any.get();

            agrosystUser.setReminderToken(hashPassword(token));
            getTransaction().commit();

            emailService.sendPasswordReminder(agrosystUser, token);
            result = true;
        }
        return result;
    }

    @Override
    public UserDto preparePasswordChange(String token, String userId) {
        UserDto result = null;
        Optional<AgrosystUser> optional = agrosystUserDao.forProperties(
                AgrosystUser.PROPERTY_TOPIA_ID, userId,
                AgrosystUser.PROPERTY_REMINDER_TOKEN, hashPassword(token)).tryFindUnique();
        if (optional.isPresent()) {
            result = Users.TO_USER_DTO.apply(optional.get());
        }
        return result;
    }

    @Override
    public boolean updatePassword(String token, String userId, String password) {

        Optional<AgrosystUser> optional = agrosystUserDao.forProperties(
                AgrosystUser.PROPERTY_TOPIA_ID, userId,
                AgrosystUser.PROPERTY_REMINDER_TOKEN, hashPassword(token)).tryFindUnique();
        if (optional.isPresent()) {
            AgrosystUser agrosystUser = optional.get();
            agrosystUser.setReminderToken(null);
            agrosystUser.setPassword(hashPassword(password));
            getTransaction().commit();
            return true;
        }
        return false;
    }

    @Override
    public void sendFeedback(String env, String location, String locationTitle, String category, String feedback, String requested, String referer, byte[] screenshotData) {
        Preconditions.checkArgument(feedback != null);
        String user = "unknown";
        try {
            String userId = getSecurityContext().getUserId();
            AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUnique();
            user = String.format("%s %s <%s>", agrosystUser.getFirstName(), agrosystUser.getLastName(), agrosystUser.getEmail());
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Unable to get current user", eee);
            }
        }
        emailService.sendFeedback(user, env, location, locationTitle, category, feedback, requested, referer, screenshotData);
    }

    @Override
    public ImportResult importUsers(InputStream userFileStream) {
        
        Import<AgrosystUser> importer = null;
        ImportResult result = new ImportResult();
        try {
            UserImportModel model = new UserImportModel();
            importer = Import.newImport(model, userFileStream);

            for (AgrosystUser entity : importer) {

                // make sure email is in lower case
                if (entity.getEmail() != null) {
                    entity.setEmail(entity.getEmail().toLowerCase());
                }

                AgrosystUser agrosystUser = agrosystUserDao.forEmailEquals(entity.getEmail()).findAnyOrNull();
                if (agrosystUser != null) {
                    // as requested, do nothing for existing accounts
                    result.incIgnored();
                } else {

                    // create token
                    String token = UUID.randomUUID().toString();
                    entity.setReminderToken(hashPassword(token));

                    // generate password
                    String password = RandomStringUtils.randomAlphanumeric(8);
                    entity.setPassword(hashPassword(password));

                    // create user
                    entity.setActive(true);
                    agrosystUser = agrosystUserDao.create(entity);

                    // send notification
                    try {
                        emailService.sendCreatedAccountNotification(agrosystUser, token);
                    } catch (AgrosystTechnicalException exception) {
                        String errorMessage = "Unable to send created account email, for user user with email :%s.";
                        errorMessage = String.format(errorMessage, agrosystUser.getEmail());
                        result.addError(errorMessage);
                    }

                    result.incCreated();

                    trackerService.userCreated(agrosystUser);
                }
            }

            getTransaction().commit();
        } finally {
            IOUtils.closeQuietly(importer);
        }
        
        return result;
    }

    @Override
    public UserDto acceptCharter() {

        String userId = getSecurityContext().getUserId();
        AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUnique();
        agrosystUser.setCharterVersion(Users.CURRENT_CHART_VERSION);

        agrosystUserDao.update(agrosystUser);

        getTransaction().commit();

        UserDto userDto = Users.TO_USER_DTO.apply(agrosystUser);

        return userDto;
    }

    @Override
    public UserDto readInfoMessages(Date lastMessageReadDate) {

        String userId = getSecurityContext().getUserId();
        AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUnique();
        agrosystUser.setLastMessageReadDate(lastMessageReadDate);

        agrosystUserDao.update(agrosystUser);

        getTransaction().commit();

        UserDto userDto = Users.TO_USER_DTO.apply(agrosystUser);

        return userDto;
    }
}
